from datetime import date
from random import choice
try:
    import hashlib
    sha_constructor = hashlib.sha1
    sha_hmac = sha_constructor
except ImportError:
    import sha
    sha_constructor = sha.new
    sha_hmac = sha


# Base 36 functions: useful for generating compact URLs

def base36_to_int(s):
    """
    Convertd a base 36 string to an integer
    """
    return int(s, 36)

def int_to_base36(i):
    """
    Converts an integer to a base36 string
    """
    digits = "0123456789abcdefghijklmnopqrstuvwxyz"
    factor = 0
    # Find starting factor
    while True:
        factor += 1
        if i < 36 ** factor:
            factor -= 1
            break
    base36 = []
    # Construct base36 representation
    while factor >= 0:
        j = 36 ** factor
        base36.append(digits[i / j])
        i = i % j
        factor -= 1
    return ''.join(base36)


class TokenGenerator(object):
    """
    Strategy object used to generate and check tokens
    """

    def __init__(self):
        self.secret_key = ''.join([choice('abcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*(-_=+)') for i in range(50)])

    def make_token(self, email):
        """ Returns a token that can be used once """
        return self._make_token_with_timestamp(email, self._num_days(self._today()))

    def check_token(self, email, token, timeout=7):
        """ Check that a token is correct for a given email. """
        # Parse the token
        try:
            ts_b36, hash = token.split("-")
        except ValueError:
            return False

        try:
            ts = base36_to_int(ts_b36)
        except ValueError:
            return False

        # Check that the timestamp/uid has not been tampered with
        if self._make_token_with_timestamp(email, ts) != token:
            return False

        # Check the timestamp is within limit
        if (self._num_days(self._today()) - ts) > timeout:
            return False

        return True
        

    def _make_token_with_timestamp(self, email, timestamp):
        # timestamp is number of days since 2001-1-1.  Converted to
        # base 36, this gives us a 3 digit string until about 2121
        ts_b36 = int_to_base36(timestamp)

        # We limit the hash to 20 chars to keep URL short
        hash = sha_constructor(self.secret_key + unicode(email) + unicode(timestamp)).hexdigest()[::2]
        return "%s-%s" % (ts_b36, hash)

    def _num_days(self, dt):
        return (dt - date(2001,1,1)).days

    def _today(self):
        # Used for mocking in tests
        return date.today()

default_token_generator = TokenGenerator()
