# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Andrei Laza, Eau De Web

#Python imports
from unittest import TestSuite, makeSuite
from BeautifulSoup import BeautifulSoup
from datetime import timedelta

#Zope imports
import transaction

#Product imports
from ws.common.tests import WatsanTestCase
from Products.UserRegistration.UserRegistration import manage_addUserRegistration, UserRegistration
from Products.UserRegistration import tokens

class MockUserRegistration(UserRegistration):
    info = []
    def send_registration_email(self, email, token):
        self.info.append({'email': email, 'token': token})

    def captcha_errors(self, contact_word, REQUEST):
        return None

class UserRegistrationTestCase(WatsanTestCase):
    def afterSetUp(self):
        self.website._setObject('test_accounts', MockUserRegistration('test_accounts', ''))
        self.userRegistration = self.website._getOb('test_accounts')

        self.signup_url = self.userRegistration.absolute_url() + '/signup'
        self.register_url = self.userRegistration.absolute_url() + '/register?activation_code=%s'

    def beforeTearDown(self):
        self.website.manage_delObjects('test_accounts')


    def test_good_data(self):
        email = 'test@eaudeweb.ro'

        #goto signup/_signup
        self.browser.open(self.signup_url)
        #enter email
        form = self.browser.getForm(name='request_account')
        form.getControl(name='email').value= email
        #submit email
        form.getControl(name='submit').click()

        #got to signup/_accept
        self.assertEqual(self.browser.url, self.signup_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

        #created token
        self.assertEqual(self.userRegistration.info[-1]['email'], email)

        #get token
        token = self.userRegistration.info[-1]['token']

        #goto register/_register
        self.browser.open(self.register_url % token)
        #enter info
        form = self.browser.getForm(name='register_account')
        form.getControl(name='username').value = 'testusername'
        form.getControl(name='first_name').value = 'test_firstname'
        form.getControl(name='last_name').value = 'test_lastname'
        #enter password
        form.getControl(name='password').value = 'test_passwd'
        #enter confirm password
        form.getControl(name='password_confirm').value = 'test_passwd'
        #submit
        form.getControl(name='submit').click()

        #got to home page
        self.assertEqual(self.browser.url, self.website.absolute_url())
        #TODO browser could not use cookies (got logged in)
        #self.assertEqual(self.website.REQUEST.AUTHENTICATED_USER.getUserName(), 'testusername')

    def test_bad_email(self):
        email = 'test@'

        #goto signup/_signup
        self.browser.open(self.signup_url)
        #enter email
        form = self.browser.getForm(name='request_account')
        form.getControl(name='email').value= email
        #submit email
        form.getControl(name='submit').click()

        #still in signup/_signup
        self.assertEqual(self.browser.url, self.signup_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is not None)
        #got error list
        self.assertTrue(soup.find('ul', id='errors') is not None)

    def test_bad_token(self):
        token = ''

        self.browser.open(self.register_url % token)
        soup = BeautifulSoup(self.browser.contents)
        ul = soup.find('ul', id='errors')
        self.assertTrue(ul is not None)
        li = ul.find('li', id='token')
        self.assertTrue(li is not None)

    def test_expired_token(self):
        email = 'test@eaudeweb.ro'

        #goto signup/_signup
        self.browser.open(self.signup_url)
        #enter email
        form = self.browser.getForm(name='request_account')
        form.getControl(name='email').value= email
        #submit email
        form.getControl(name='submit').click()

        #got to signup/_accept
        self.assertEqual(self.browser.url, self.signup_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

        #created token
        self.assertEqual(self.userRegistration.info[-1]['email'], email)

        #get token
        token = self.userRegistration.info[-1]['token']

        UserRegistration.token_timeout = -1
        try:
            #goto register/_register
            self.browser.open(self.register_url % token)
            soup = BeautifulSoup(self.browser.contents)
            ul = soup.find('ul', id='errors')
            self.assertTrue(ul is not None)
            li = ul.find('li', id='token_expired')
            self.assertTrue(li is not None)
            self.assertTrue(soup.find('form', attrs={'name': 'register_account'}) is None)
        finally:
            UserRegistration.token_timeout = 7

    def test_bad_passwords_match(self):
        email = 'test@eaudeweb.ro'

        #goto signup/_signup
        self.browser.open(self.signup_url)
        #enter email
        form = self.browser.getForm(name='request_account')
        form.getControl(name='email').value= email
        #submit email
        form.getControl(name='submit').click()

        #got to signup/_accept
        self.assertEqual(self.browser.url, self.signup_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

        #created token
        self.assertEqual(self.userRegistration.info[-1]['email'], email)

        #get token
        token = self.userRegistration.info[-1]['token']

        #goto register/_register
        self.browser.open(self.register_url % token)
        #enter info
        form = self.browser.getForm(name='register_account')
        form.getControl(name='username').value = 'testusername'
        form.getControl(name='first_name').value = 'test_firstname'
        form.getControl(name='last_name').value = 'test_lastname'
        #enter password
        form.getControl(name='password').value = 'test_passwd'
        #enter confirm password
        form.getControl(name='password_confirm').value = 'test_pwd'
        #submit
        form.getControl(name='submit').click()

        #got to register/_register
        self.assertEqual(self.browser.url, self.register_url % token)
        soup = BeautifulSoup(self.browser.contents)
        ul = soup.find('ul', id='errors')
        self.assertTrue(ul is not None)
        li = ul.find('li', id='form')
        self.assertTrue(li is not None)

    def test_not_unique_username(self):
        email = 'test@eaudeweb.ro'

        self.website.acl_users._doAddUser('testusername', 'test_passwd', [], [], 'test_firstname', 'test_lastname', email)
        try:
            #goto signup/_signup
            self.browser.open(self.signup_url)
            #enter email
            form = self.browser.getForm(name='request_account')
            form.getControl(name='email').value= email
            #submit email
            form.getControl(name='submit').click()

            #got to signup/_accept
            self.assertEqual(self.browser.url, self.signup_url)
            soup = BeautifulSoup(self.browser.contents)
            self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

            #created token
            self.assertEqual(self.userRegistration.info[-1]['email'], email)

            #get token
            token = self.userRegistration.info[-1]['token']

            #goto register/_register
            self.browser.open(self.register_url % token)
            #enter info
            form = self.browser.getForm(name='register_account')
            form.getControl(name='username').value = 'testusername'
            form.getControl(name='first_name').value = 'test_firstname'
            form.getControl(name='last_name').value = 'test_lastname'
            #enter password
            form.getControl(name='password').value = 'test_passwd'
            #enter confirm password
            form.getControl(name='password_confirm').value = 'test_passwd'
            #submit
            form.getControl(name='submit').click()

            #got to register/_register
            self.assertEqual(self.browser.url, self.register_url % token)
            soup = BeautifulSoup(self.browser.contents)
            ul = soup.find('ul', id='errors')
            self.assertTrue(ul is not None)
            li = ul.find('li', id='username')
            self.assertTrue(li is not None)
        finally:
            self.website.acl_users._doDelUsers(['testusername'])

    def test_register_after_username_error(self):
        email = 'test@eaudeweb.ro'

        self.website.acl_users._doAddUser('testusername', 'test_passwd', [], [], 'test_firstname', 'test_lastname', email)
        try:
            #goto signup/_signup
            self.browser.open(self.signup_url)
            #enter email
            form = self.browser.getForm(name='request_account')
            form.getControl(name='email').value= email
            #submit email
            form.getControl(name='submit').click()

            #got to signup/_accept
            self.assertEqual(self.browser.url, self.signup_url)
            soup = BeautifulSoup(self.browser.contents)
            self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

            #created token
            self.assertEqual(self.userRegistration.info[-1]['email'], email)

            #get token
            token = self.userRegistration.info[-1]['token']

            #goto register/_register
            self.browser.open(self.register_url % token)
            #enter info
            form = self.browser.getForm(name='register_account')
            form.getControl(name='username').value = 'testusername'
            form.getControl(name='first_name').value = 'test_firstname'
            form.getControl(name='last_name').value = 'test_lastname'
            #enter password
            form.getControl(name='password').value = 'test_passwd'
            #enter confirm password
            form.getControl(name='password_confirm').value = 'test_passwd'
            #submit
            form.getControl(name='submit').click()

            #got to register/_register
            self.assertEqual(self.browser.url, self.register_url % token)
            soup = BeautifulSoup(self.browser.contents)
            ul = soup.find('ul', id='errors')
            self.assertTrue(ul is not None)
            li = ul.find('li', id='username')
            self.assertTrue(li is not None)

            #enter info
            form = self.browser.getForm(name='register_account')
            form.getControl(name='username').value = 'testusername2'
            form.getControl(name='first_name').value = 'test_firstname'
            form.getControl(name='last_name').value = 'test_lastname'
            #enter password
            form.getControl(name='password').value = 'test_passwd'
            #enter confirm password
            form.getControl(name='password_confirm').value = 'test_passwd'
            #submit
            form.getControl(name='submit').click()

            #got to home page
            self.assertEqual(self.browser.url, self.website.absolute_url())
        finally:
            self.website.acl_users._doDelUsers(['testusername'])

    def test_double_activation(self):
        email = 'test@eaudeweb.ro'

        #goto signup/_signup
        self.browser.open(self.signup_url)
        #enter email
        form = self.browser.getForm(name='request_account')
        form.getControl(name='email').value= email
        #submit email
        form.getControl(name='submit').click()

        #got to signup/_accept
        self.assertEqual(self.browser.url, self.signup_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

        #created token
        self.assertEqual(self.userRegistration.info[-1]['email'], email)

        #get token
        token = self.userRegistration.info[-1]['token']

        #goto signup/_signup
        self.browser.open(self.signup_url)
        #enter email
        form = self.browser.getForm(name='request_account')
        form.getControl(name='email').value= email
        #submit email
        form.getControl(name='submit').click()

        #got to signup/_resend_activation_email
        self.assertEqual(self.browser.url, self.signup_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)
        self.assertTrue(soup.find('form', attrs={'action': 'resend_activation_email'}) is not None)



def test_suite():
    """
    Test suite for the watsan.common package
    """
    suite = TestSuite()
    suite.addTest(makeSuite(UserRegistrationTestCase))
    return suite

