# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Andrei Laza, Eau De Web

#Python imports
from unittest import TestSuite, makeSuite
from BeautifulSoup import BeautifulSoup
from datetime import timedelta

#Zope imports
import transaction

#Product imports
from ws.common.tests import WatsanTestCase
from Products.UserRegistration.UserRegistration import manage_addUserRegistration, UserRegistration
from Products.UserRegistration import tokens

class MockUserRegistration(UserRegistration):
    info = []
    def send_forgot_password_email(self, email, token):
        self.info.append({'email': email, 'token': token})

    def captcha_errors(self, contact_word, REQUEST):
        return None

class PasswordChangeTestCase(WatsanTestCase):
    def afterSetUp(self):
        self.website._setObject('test_accounts', MockUserRegistration('test_accounts', ''))
        self.userRegistration = self.website._getOb('test_accounts')

        self.test_username = 'test'
        self.test_email = 'test_email@eaudeweb.ro'
        auth_tool = self.website.getAuthenticationTool()
        auth_tool.manage_addUser(self.test_username, 'old_pass', 'old_pass',
                [], [], 'first_name', 'last_name', self.test_email)

        self.forgot_password_url = self.userRegistration.absolute_url() + '/forgot_password'
        self.change_password_url = self.userRegistration.absolute_url() + '/change_password?change_password_code=%s'

    def beforeTearDown(self):
        self.website.manage_delObjects('test_accounts')

        auth_tool = self.website.getAuthenticationTool()
        auth_tool.manage_delUsers([self.test_username])


    def _step_forgot_password_good(self):
        #goto forgot_password/_forgot_password
        self.browser.open(self.forgot_password_url)
        #enter username
        form = self.browser.getForm(name='request_account')
        form.getControl(name='username').value = self.test_username
        #submit email
        form.getControl(name='submit').click()

        #got to forgot_password/_forgot_password_email_sent
        self.assertEqual(self.browser.url, self.forgot_password_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is None)

        #created token
        self.assertEqual(self.userRegistration.info[-1]['email'], self.test_email)



    def test_good_data(self):
        self._step_forgot_password_good()

        #get token
        token = self.userRegistration.info[-1]['token']

        #goto change_password/_change_password
        self.browser.open(self.change_password_url % token)
        #enter info
        form = self.browser.getForm(name='register_account')
        #enter password
        form.getControl(name='password').value = 'new_passwd'
        #enter confirm password
        form.getControl(name='password_confirm').value = 'new_passwd'
        #submit
        form.getControl(name='submit').click()

        #got to home page
        self.assertEqual(self.browser.url, self.website.absolute_url())
        #check user has new password
        auth_tool = self.website.getAuthenticationTool()
        user_obj = auth_tool.getUser(self.test_username)
        #TODO: does not seem to work to test for the new password (probably browser related)
        #self.assertEqual(auth_tool.getUserPassword(user_obj), 'new_passwd')
        self.assertEqual(auth_tool.getUserFirstName(user_obj), 'first_name')
        self.assertEqual(auth_tool.getUserLastName(user_obj), 'last_name')
        self.assertEqual(auth_tool.getUserEmail(user_obj), self.test_email)

    def test_bad_username(self):
        username = ''

        #goto forgot_password/_forgot_password
        self.browser.open(self.forgot_password_url)
        #enter username
        form = self.browser.getForm(name='request_account')
        form.getControl(name='username').value = username
        #submit email
        form.getControl(name='submit').click()

        self.assertEqual(self.browser.url, self.forgot_password_url)
        soup = BeautifulSoup(self.browser.contents)
        self.assertTrue(soup.find('form', attrs={'name': 'request_account'}) is not None)
        #got error list
        self.assertTrue(soup.find('ul', id='errors') is not None)

    def test_bad_token(self):
        token = ''

        self.browser.open(self.change_password_url % token)
        soup = BeautifulSoup(self.browser.contents)
        ul = soup.find('ul', id='errors')
        self.assertTrue(ul is not None)
        li = ul.find('li', id='token')
        self.assertTrue(li is not None)

    def test_expired_token(self):
        self._step_forgot_password_good()

        #get token
        token = self.userRegistration.info[-1]['token']

        UserRegistration.token_timeout = -1
        try:
            #goto change_password/_change_password
            self.browser.open(self.change_password_url % token)
            soup = BeautifulSoup(self.browser.contents)
            ul = soup.find('ul', id='errors')
            self.assertTrue(ul is not None)
            li = ul.find('li', id='token_expired')
            self.assertTrue(li is not None)
        finally:
            UserRegistration.token_timeout = 7

    def test_bad_passwords_match(self):
        self._step_forgot_password_good()

        #get token
        token = self.userRegistration.info[-1]['token']

        #goto change_password/_change_password
        self.browser.open(self.change_password_url % token)
        #enter info
        form = self.browser.getForm(name='register_account')
        #enter password
        form.getControl(name='password').value = 'new_passwd'
        #enter confirm password
        form.getControl(name='password_confirm').value = 'new_2_passwd'
        #submit
        form.getControl(name='submit').click()

        #got to change_password/_change_password
        self.assertEqual(self.browser.url, self.change_password_url % token)
        soup = BeautifulSoup(self.browser.contents)
        ul = soup.find('ul', id='errors')
        self.assertTrue(ul is not None)
        li = ul.find('li', id='form')
        self.assertTrue(li is not None)

        #enter info
        form = self.browser.getForm(name='register_account')
        #enter password
        form.getControl(name='password').value = 'new_passwd'
        #enter confirm password
        form.getControl(name='password_confirm').value = 'new_passwd'
        #submit
        form.getControl(name='submit').click()

        #got to home page
        self.assertEqual(self.browser.url, self.website.absolute_url())
        #check user has new password
        auth_tool = self.website.getAuthenticationTool()
        user_obj = auth_tool.getUser(self.test_username)
        #TODO: does not seem to work to test for the new password (probably browser related)
        #self.assertEqual(auth_tool.getUserPassword(user_obj), 'new_passwd')
        self.assertEqual(auth_tool.getUserFirstName(user_obj), 'first_name')
        self.assertEqual(auth_tool.getUserLastName(user_obj), 'last_name')
        self.assertEqual(auth_tool.getUserEmail(user_obj), self.test_email)

    def test_unsecure_password(self):
        self._step_forgot_password_good()

        #get token
        token = self.userRegistration.info[-1]['token']

        #goto change_password/_change_password
        self.browser.open(self.change_password_url % token)
        #enter info
        form = self.browser.getForm(name='register_account')
        #enter password
        form.getControl(name='password').value = 'pasta'
        #enter confirm password
        form.getControl(name='password_confirm').value = 'pasta'
        #submit
        form.getControl(name='submit').click()

        #got to change_password/_change_password
        self.assertEqual(self.browser.url, self.change_password_url % token)
        soup = BeautifulSoup(self.browser.contents)
        ul = soup.find('ul', id='errors')
        self.assertTrue(ul is not None)
        li = ul.find('li', id='form')
        self.assertTrue(li is not None)

def test_suite():
    """
    Test suite for the watsan.common package
    """
    suite = TestSuite()
    suite.addTest(makeSuite(PasswordChangeTestCase))
    return suite


