# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cornel Nitu, Eau De Web
# Andrei Laza, Eau De Web

#Python imports
from formencode import validators, Schema, FancyValidator, Invalid, All, compound
import re
import os

#Zope imports
from Globals import package_home

class SecurePassword(FancyValidator):
    words_filename = os.path.join(package_home(globals()), 'DICT.TXT')

    def _to_python(self, value, state):
        f = open(self.words_filename)
        lower = value.strip().lower()
        for line in f:
            if line.strip().lower() == lower:
                raise Invalid('Please do not base your password on a dictionary term', value, state)
        return value

class VerifyPassword(FancyValidator):
    def _to_python(self, value, state):
        auth_tool = state.getSite().getAuthenticationTool()
        userid = auth_tool.get_current_userid()
        if auth_tool.encrypt_passwords:
            if auth_tool.getUser(userid).__ != auth_tool._encryptPassword(value):
                raise Invalid('Your old password was entered incorrectly. Please enter it again.',value, state)
        else:
            if auth_tool.getUser(userid).__ != value:
                raise Invalid('Your old password was entered incorrectly. Please enter it again.',value, state)
        return value

class UniqueUsername(FancyValidator):
    def _to_python(self, value, state):
        auth_tool = state.getSite().getAuthenticationTool()
        if value in auth_tool.getUserNames():
            raise Invalid('This username is already taken. Please choose another.',value, state)
        return value

#forms validation
class RequestAccount(Schema):
    email = validators.Email(not_empty=True, resolve_domain=False)   #email must be unique
    filter_extra_fields = True
    allow_extra_fields = True

class RegisterAccount(Schema):
    username = All(validators.PlainText(not_empty=True), UniqueUsername())
    first_name = validators.String(not_empty=True)
    last_name = validators.String(not_empty=True)
    password = compound.All(validators.String(not_empty=True), SecurePassword())
    password_confirm = validators.String()
    chained_validators = [validators.FieldsMatch('password', 'password_confirm')]
    filter_extra_fields = True
    allow_extra_fields = True


class EditProfile(Schema):
    first_name = validators.String(not_empty=True)
    last_name = validators.String(not_empty=True)
    email = validators.Email(not_empty=True, resolve_domain=False)   #email must be unique
    filter_extra_fields = True
    allow_extra_fields = True

class ForgotPassword(Schema):
    username = validators.PlainText()
    filter_extra_fields = True
    allow_extra_fields = True

class ChangePassword(Schema):
    old_password = All(validators.PlainText(not_empty=True), VerifyPassword())
    password = compound.All(validators.String(not_empty=True), SecurePassword())
    password_confirm = validators.String()
    chained_validators = [validators.FieldsMatch('password', 'password_confirm')]
    filter_extra_fields = True
    allow_extra_fields = True

class ResetPassword(Schema):
    username = validators.PlainText()
    password = compound.All(validators.String(not_empty=True), SecurePassword())
    password_confirm = validators.String()
    chained_validators = [validators.FieldsMatch('password', 'password_confirm')]
    filter_extra_fields = True
    allow_extra_fields = True

