#Python imports
from formencode import validators
from datetime import timedelta, datetime

#Zope imports
from OFS.SimpleItem import SimpleItem
from AccessControl import ClassSecurityInfo
from AccessControl.Permissions import view
from Globals import InitializeClass
from BTrees.OOBTree import OOBTree

#Product imports
from Products.NaayaCore.FormsTool.NaayaTemplate import NaayaPageTemplateFile
from Products.NaayaCore.EmailTool.EmailPageTemplate import EmailPageTemplateFile
from naaya.core.StaticServe import StaticServeFromFolder
import forms
from tokens import default_token_generator


email_templates = {
    'activate_account': EmailPageTemplateFile('emailpt/activate_account.zpt', globals()),
    'password_reset': EmailPageTemplateFile('emailpt/password_reset.zpt', globals()),
}


def manage_addUserRegistration(self, id="accounts", REQUEST=None):
    """ constructor """
    ob = UserRegistration(id, title='Accounts')
    self._setObject(id, ob)
    if REQUEST is not None:
        return self.manage_main(self, REQUEST, update_menu=1)
    return ob

class UserRegistration(SimpleItem):
    """ """
    meta_type = "UserRegistration"

    security = ClassSecurityInfo()

    token_timeout = 7 #days

    def __init__(self, id, title):
        """ """
        self.id = id
        self.title = title
        self.signup_tokens = OOBTree()
        self.signup_emails = OOBTree()
        self.forgot_pass_tokens = OOBTree()

    index_html = NaayaPageTemplateFile('zpt/index', globals(), 'registration_index')

    def _get_template(self, name):
        template = email_templates.get(name, None)
        if template is not None:
            return template.render_email
        raise ValueError('template for %r not found' % name)

    security.declarePrivate('send_activation')
    def send_activation(self, email, template, **kwargs):
        """ """
        portal = self.getSite()
        email_tool = portal.getEmailTool()

        message = {}
        message['site_url'] = portal.absolute_url()
        message['site_title'] = portal.site_title
        message['activation_url'] = '%s/register?activation_code=%s' % (self.absolute_url(), kwargs['token'])
        
        
        template = self._get_template(template)
        mail_data = template(**message)
        email_tool.sendEmail(p_content = mail_data['body_text'],
                             p_to = email,
                             p_from = email_tool._get_from_address(),
                             p_subject = mail_data['subject'])

    security.declarePrivate('send_password')
    def send_password(self, email, template, **kwargs):
        portal = self.getSite()
        email_tool = portal.getEmailTool()

        message = {}
        message['site_title'] = portal.site_title
        message['reset_url'] = '%s/reset_password?change_password_code=%s' % (self.absolute_url(), kwargs['token'])
        message['username'] = kwargs['username']
        message['email'] = email

        template = self._get_template(template)
        mail_data = template(**message)
        email_tool.sendEmail(p_content = mail_data['body_text'],
                             p_to = email,
                             p_from = email_tool._get_from_address(),
                             p_subject = mail_data['subject'])

    def get_user_email(self, username):
        auth_tool = self.getSite().getAuthenticationTool()
        try:
            user_obj = auth_tool.getUser(username)
            return auth_tool.getUserEmail(user_obj)
        except:
            return None

    def save_password(self, username, form):
        auth_tool = self.getSite().getAuthenticationTool()
        user_obj = auth_tool.getUser(username)

        auth_tool.manage_changeUser(
                name=username,
                password=form['password'],
                confirm=form['password_confirm'],
                firstname=auth_tool.getUserFirstName(user_obj),
                lastname=auth_tool.getUserLastName(user_obj),
                email=auth_tool.getUserEmail(user_obj)
                )
        auth_tool.credentialsChanged(username, form['password'])

    _signup = NaayaPageTemplateFile('zpt/signup', globals(), 'registration_signup')
    _activate = NaayaPageTemplateFile('zpt/activate', globals(), 'registration_activate')
    security.declareProtected(view, 'signup')
    def signup(self, REQUEST=None):
        """ request account """
        if REQUEST.REQUEST_METHOD == 'GET':
            return self._signup(REQUEST)
        elif REQUEST.REQUEST_METHOD == 'POST':
            contact_word = REQUEST.form.get('contact_word', None)
            errors = self.captcha_errors(contact_word, REQUEST)
            if errors is not None:
                return self._signup(REQUEST, error='captcha', error_msg=errors[0])

            try:
                form = forms.RequestAccount.to_python(REQUEST.form)
            except validators.Invalid, e:
                return self._signup(REQUEST, error='form', error_msg=e.msg)

            email = REQUEST.form['email']
            token = self.signup_emails.get(email, None)
            if token is not None:
                return self._resend_activation_email(REQUEST, email=email)

            token = default_token_generator.make_token(email)
            self.signup_tokens.insert(token, email)
            self.signup_emails.insert(email, token)

            #send the activation account link by email
            self.send_activation(email, template="activate_account", token=token)

            return self._activate(REQUEST)


    _resend_activation_email = NaayaPageTemplateFile('zpt/resend_activation_email', globals(), 'registration_resend_activation_email')
    security.declareProtected(view, 'resend_activation_email')
    def resend_activation_email(self, REQUEST=None):
        """ """
        if REQUEST.form['submit'] == 'No':
            return REQUEST.RESPONSE.redirect(self.getSite().absolute_url())

        email = REQUEST.form['email']
        #send the activation account link by email
        self.send_activation(email = email,
                               template = "activate_account",
                               token = self.signup_emails.get(email))
        return self._activate(REQUEST)


    _profile = NaayaPageTemplateFile('zpt/profile', globals(), 'registration_profile')
    security.declareProtected(view, 'profile')
    def profile(self, REQUEST=None):
        """ edit profile """
        error = success = None
        auth_tool = self.getSite().getAuthenticationTool()
        userid = auth_tool.get_current_userid()
        user = auth_tool.get_user_with_userid(userid)
        if REQUEST.REQUEST_METHOD == 'POST':
            try:
                form = forms.EditProfile.to_python(REQUEST.form)
            except validators.Invalid, e:
                error = e
            else:
                #@todo: save roles
                auth_tool.manage_changeUser(userid,
                                            firstname=form['first_name'],
                                            lastname=form['last_name'],
                                            email=form['email'])
                success = True
        return self._profile(REQUEST, user=user, error=error, success=success)


    _register = NaayaPageTemplateFile('zpt/register', globals(), 'registration_register')
    security.declareProtected(view, 'register')
    def register(self, REQUEST=None):
        """ register account """
        token = REQUEST.form.get('activation_code', None)
        email = self.signup_tokens.get(token, None)

        if token is None or email is None:
            return self._register(REQUEST, error='token')

        if not default_token_generator.check_token(email, token, self.token_timeout):
            #cleanup
            self.signup_tokens.pop(token)
            self.signup_emails.pop(email)
            return self._register(REQUEST, error='token_expired')

        if REQUEST.REQUEST_METHOD == 'POST':
            try:
                form = forms.RegisterAccount.to_python(REQUEST.form, state=self)
            except validators.Invalid, e:
                return self._register(REQUEST, email=email, error=e)

            auth_tool = self.getSite().getAuthenticationTool()
            auth_tool.manage_addUser(form['username'], form['password'],
                                     form['password_confirm'], [], [],
                                     form['first_name'], form['last_name'], email)
            auth_tool.credentialsChanged(form['username'], form['password'])

            #cleanup
            self.signup_tokens.pop(token)
            self.signup_emails.pop(email)

            return REQUEST.RESPONSE.redirect(self.getSite().absolute_url())
        else:
            return self._register(REQUEST, email=email)

    _forgot_password = NaayaPageTemplateFile('zpt/forgot_password', globals(), 'registration_forgot_password')
    _forgot_password_email_sent = NaayaPageTemplateFile('zpt/forgot_password_email_sent', globals(), 'registration_forgot_password_email_sent')
    security.declareProtected(view, 'forgot_password')
    def forgot_password(self, REQUEST=None):
        """ forgot password for account """
        if REQUEST.REQUEST_METHOD == 'GET':
            return self._forgot_password(REQUEST)
        elif REQUEST.REQUEST_METHOD == 'POST':
            contact_word = REQUEST.form.get('contact_word', None)
            errors = self.captcha_errors(contact_word, REQUEST)
            if errors is not None:
                return self._forgot_password(REQUEST, error='captcha', error_msg=errors[0])

            try:
                form = forms.ForgotPassword.to_python(REQUEST.form)
            except validators.Invalid, e:
                return self._signup(REQUEST, error='form', error_msg=e.msg)

            username = REQUEST.form['username']
            token = default_token_generator.make_token(username)
            self.forgot_pass_tokens.insert(token, username)

            email = self.get_user_email(username)
            if not email:
                return self._signup(REQUEST, error='username')

            #send the reset password link by email
            self.send_password(email, template="password_reset", token=token , username=username)

            return self._forgot_password_email_sent(REQUEST, token=token)


    _change_password = NaayaPageTemplateFile('zpt/change_password', globals(), 'registration_change_password')
    def change_password(self, REQUEST=None):
        """ change password for account """
        auth_tool = self.getSite().getAuthenticationTool()
        username = auth_tool.get_current_userid()

        error = success = None
        if REQUEST.REQUEST_METHOD == 'POST':
            try:
                form = forms.ChangePassword.to_python(REQUEST.form, state=self)
            except validators.Invalid, e:
                error = e
            else:
                self.save_password(username, form)
                success = True
        return self._change_password(REQUEST, error=error, success=success)


    _reset_password = NaayaPageTemplateFile('zpt/reset_password', globals(), 'registration_reset_password')
    security.declareProtected(view, 'reset_password')
    def reset_password(self, REQUEST=None):
        """ reset password for account """
        token = REQUEST.form.get('change_password_code', None)
        username = self.forgot_pass_tokens.get(token, None)
        if token is None or username is None:
            return self._reset_password(REQUEST, error='token')
        if not default_token_generator.check_token(username, token, self.token_timeout):
            #cleanup
            self.forgot_pass_tokens.pop(token)
            return self._reset_password(REQUEST, error='token_expired')

        if REQUEST.REQUEST_METHOD == 'GET':
            return self._reset_password(REQUEST, username=username)
        elif REQUEST.REQUEST_METHOD == 'POST':
            try:
                form = forms.ResetPassword.to_python(REQUEST.form)
            except validators.Invalid, e:
                return self._reset_password(REQUEST, username=username, error='form', error_msg=e.msg)

            self.save_password(username, form)
            return REQUEST.RESPONSE.redirect(self.getSite().absolute_url())

    media = StaticServeFromFolder('media', globals(), cache=False)

InitializeClass(UserRegistration)
