# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cristian Romanescu, Eau De Web
# Andrei Laza, Eau De Web
from string import upper

import ConfigParser
from sqlalchemy.sql.expression import func
import os, sys, string, traceback
from operator import itemgetter

#Zope imports
import zLOG
from AccessControl.SecurityInfo import ClassSecurityInfo
from Products.PageTemplates.PageTemplateFile import PageTemplateFile
from AccessControl.Permissions import view, view_management_screens
from Globals import InitializeClass
import Globals
from z3c.sqlalchemy.util import createSAWrapper, getSAWrapper, registeredWrappers
from persistent.mapping import PersistentMapping

#Product imports
from Products.Naaya.NySite import NySite
from Products.WSToolkit.constants import WEBSITE_ID
from Products.WSToolkit import constants
from Products.Naaya.constants import PREFIX_SITE
from Products.NaayaCore.managers.utils import utils
from naaya.core.StaticServe import StaticServeFromFolder
from Products.NaayaCore.FormsTool.NaayaTemplate import NaayaPageTemplateFile

from Products.Naaya.NyFolder import addNyFolder

from Products.UserRegistration.UserRegistration import manage_addUserRegistration
from ws.common.sync.DictionarySync import DictionarySyncEndpoint
from ws.common.new_portal import PortalConfiguration, create_wsportal
import forms
from formencode import validators
from formencode.schema import SimpleFormValidator
from ws.common.catalogs.Catalogs import create_catalogs_object_callback
from ws.common.utilities import DICT_COUNTRIES_EN



PERMISSION_CREATE_NEW_COUNTRY_PORTAL = "Create new country portal"

def create_country_portal(parent, parameters, db = None):
    """
        Create new WSPortal instance inside database.
        Parameters:
            `parent`
                Where to attach this website to
            `lang`
                Language, default None
            `country_code`
                Country code (ISO 2-letter code)
            `languages`
                Languages available within portal
            `skin`
                Name of the skin used for portal
            `moderation`
                Type of moderation for user contribution (pre/post posting)
            `db`
                Dictionary with database connection information
            `REQUEST`
                Zope's HTTP request object
        Returns newly created object or redirects if REQUEST is not None
        Raises exception if any error occurrs during portal creation
    """
    toolkit = parent._getOb(WEBSITE_ID)
    session = toolkit.get_db_session()

    country_code = parameters['country'].lower()
    country_name = DICT_COUNTRIES_EN[country_code.upper()]

    cfg = PortalConfiguration(**parameters)
    cfg.gmap_key = toolkit.gmap_key
    cfg.country_name = country_name

    if db:
        # We are running on server that hosts toolkit portal. This means that
        # We have database connection information taken from the WSToolkit
        # object properties.
        zLOG.LOG(__name__, zLOG.INFO, 'Reading database connection information from toolkit')
        cfg.db_host = db['db_host']
        cfg.db_username = db['db_username']
        cfg.db_password = db['db_password']
        cfg.db_super_database = db['db_super_name']
        cfg.db_super_username = db['db_super_username']
        cfg.db_super_password = db['db_super_password']
    else:
        # We are running standalone or toolkit mode so for the moment we read
        # database configuration parameters from the setup.ini file. This could
        # be changed to a form letting the user enter database configuration
        # information. Marking as TODO:
        zLOG.LOG(__name__, zLOG.INFO, 'Reading configuration from setup.ini')
        cp = ConfigParser.ConfigParser()
        cp.read(os.path.join(os.path.dirname(__file__), 'sql', 'setup.ini'))
        cfg.db_host = cp.get('database', 'database_host')
        cfg.db_username = cp.get('database', 'username')
        cfg.db_password = cp.get('database', 'password')
        cfg.db_super_database = cp.get('database', 'super_database')
        cfg.db_super_username = cp.get('database', 'super_username')
        cfg.db_super_password = cp.get('database', 'super_password')

    ob = create_wsportal(parent, cfg)

    return ob



manage_addWSToolkit_html = PageTemplateFile('zpt/site_manage_add', globals())
def manage_addWSToolkit(self, lang='en', db_host='',
                        db_super_username='', db_super_password='',
                        db_username='', db_password='',
                        db_super_name='watsan',
                        gmap_key='', REQUEST=None):
    """
        Create new WSToolkit instance inside database
        @param parent: Where to attach this website to
        @param lang: Language, default None
        @param REQUEST: Zope's HTTP request object
        @return: Newly created object
        @raise exception: Any error encountered during portal creation
    """
    ut = utils()
    id = constants.WEBSITE_ID
    if not id: id = PREFIX_SITE + ut.utGenRandomId(6)
    portal_uid = '%s_%s' % (PREFIX_SITE, ut.utGenerateUID())

    gmap_key_dict = {REQUEST.HTTP_X_FORWARDED_SERVER: gmap_key}
    zLOG.LOG(__name__, zLOG.INFO, 'Creating new Watsan toolkit on toolkit/')
    obj = WSToolkit(id, portal_uid, constants.WEBSITE_TITLE, 'en',
                    db_host, db_username, db_password, db_super_name, gmap_key_dict,
                    db_super_username, db_super_password)
    self._setObject(id, obj)
    ob = self._getOb(id)
    ob.createPortalTools()
    ob.loadDefaultData()

    if REQUEST is not None:
        return self.manage_main(self, REQUEST, update_menu=1)
    return ob



class WSToolkit(NySite):
    """
        WSToolkit object, folder-type that contains items described within specs. This is the root of an WatSan toolkit.
    """
    meta_type = 'WatSan Toolkit'
    security = ClassSecurityInfo()

    product_paths = NySite.product_paths + [Globals.package_home(globals())]


    def __init__(self, id, portal_uid, title, lang,
                 db_host='', db_username='', db_password='', db_super_name='', gmap_key='',
                 db_super_username='', db_super_password=''):
        self.db_host = db_host
        self.db_username = db_username
        self.db_password = db_password
        self.db_super_name = db_super_name
        self.db_super_username = db_super_username
        self.db_super_password = db_super_password
        self.gmap_key = PersistentMapping(gmap_key)
        NySite.__init__(self, id, portal_uid, title, lang)


    security.declarePrivate('loadDefaultData')
    def loadDefaultData(self, *args, **kwargs):
        """ Load the initial data into a new created WatSan toolkit """
        super(WSToolkit, self).loadDefaultData(*args, **kwargs)
        zLOG.LOG(__name__, zLOG.INFO, 'Loading initial data into the WatSan toolkit')
        #Create the MailHost for emailing
        #from Products.MailHost.MailHost import manage_addMailHost

        #remove Naaya default content
        self.getLayoutTool().manage_delObjects('skin')

        #add watsan website content
        self.loadSkeleton(Globals.package_home(globals()))

        #remove the images folders, we don't use it
        for skin in self.getLayoutTool().getSkinsList():
            skin.manage_delObjects('images')

        # Add left portlets
        site = self.getSite()
        portlets_tool = site.getPortletsTool()
        site_base_url = site.absolute_url(1)
        location = self.absolute_url(1)
        if site_base_url and location.startswith(site_base_url):
            location = location[len(site_base_url)+1:]
        portlets_tool.unassign_portlet(location, position='left', portlet_id='portlet_maincategories')
        portlets_tool.assign_portlet(location, position='left', portlet_id='toolkit_logo', inherit=True)

    def suggest_mapserver_url(self, REQUEST=None):
        ret = ''
        try:
            if REQUEST:
                ret = '/cgi-bin/mapserv'
        except:
            pass
        return ret


    def list_countries_without_portal(self):
        ret = []
        items = DICT_COUNTRIES_EN.items()
        items.sort(key=itemgetter(1))
        for item in items:
            key = item[0].lower()
            try:
                getattr(self.aq_parent, key)
            except:
                ret.append({'adcncodex' : key, 'adcnname' : item[1]})
        return ret


    def get_db_session(self):
        wrapper = None
        if WEBSITE_ID in registeredWrappers.keys():
            wrapper = getSAWrapper(WEBSITE_ID)
        else:
            wrapper = createSAWrapper('postgresql://%s:%s@%s/%s' % (self.db_username, self.db_password, self.db_host, WEBSITE_ID), name=WEBSITE_ID)
        return wrapper.session


    _step1 = NaayaPageTemplateFile('zpt/new/step1', globals(), 'ws_new_portal_step1')
    _step2 = NaayaPageTemplateFile('zpt/new/step2', globals(), 'ws_new_portal_step2')
    _step3 = NaayaPageTemplateFile('zpt/new/step3', globals(), 'ws_new_portal_step3')
    _forbidden = NaayaPageTemplateFile('zpt/new/forbidden', globals(), 'ws_new_portal_forbidden')
    _new_portal = NaayaPageTemplateFile('zpt/new/introduction', globals(), 'ws_new_portal')
    _finish = NaayaPageTemplateFile('zpt/new/finish', globals(), 'ws_new_portal_finish')


    def get_google_map_default_key(self):
        """ """
        return self.gmap_key.get(self.REQUEST.HTTP_X_FORWARDED_SERVER, '')

    def get_google_maps_url(self):
        """ """
        return """
        <script type="text/javascript">
document.write([
  '<script src="http://maps.google.com/maps?file=api&amp;v=2&amp;key=', 
    %s
  [window.location.host],
  '" type="text/javascript"><\/script>'
].join(''));
</script>
""" % self.rstk.json_dumps(self.gmap_key.data)


    def new_portal(self, REQUEST):
        """ Wizard to create a new portal """
        error = None

        if REQUEST.form.has_key('step1'):
            return self._step1(REQUEST)

        if REQUEST.form.has_key('btn-step2'):
            session = self.get_db_session()
            try:
                form = forms.AddNewPortalFormStep1.to_python(REQUEST.form, state=session)
            except validators.Invalid, e:
                error = e
                return self._step1(REQUEST, error=error)
            else:
                return self._step2(REQUEST)

        if REQUEST.form.has_key('btn-step2-back'):
            return self._step1(REQUEST)

        if REQUEST.form.has_key('btn-step3'):
            session = self.get_db_session()
            try:
                form = forms.AddNewPortalFormStep2.to_python(REQUEST.form, state=session)
            except validators.Invalid, e:
                error = e
                return self._step2(REQUEST, error=error)
            else:
                country_code = form['country']
                country_name = DICT_COUNTRIES_EN[country_code.upper()]
                return self._step3(REQUEST, country = country_name)

        if REQUEST.form.has_key('btn-step3-back'):
            return self._step2(REQUEST)

        if REQUEST.form.has_key('btn-step4'):
            session = self.get_db_session()
            try:
                form = forms.AddNewPortalFormStep2.to_python(REQUEST.form, state=session)
            except validators.Invalid, e:
                error = e
                return self._step2(REQUEST, error=error)
            else:
                # Form is valid, proceed to creation
                db = {
                    'db_host' : self.db_host,
                    'db_username' : self.db_username,
                    'db_password' : self.db_password,
                    'db_super_name' : self.db_super_name,
                    'db_super_username' : self.db_super_username,
                    'db_super_password' : self.db_super_password
                }
                ob = create_country_portal(self.aq_parent, form, db)
                return self._finish(REQUEST, url = ob.absolute_url())

        return self._new_portal(REQUEST)
    index_html = new_portal

    security.declarePublic('new_portal_finish')
    new_portal_finish = NaayaPageTemplateFile('zpt/new/finish', globals(), 'ws_new_portal_finish')

    #security.declareProtected(view, 'getting_started')
    getting_started = NaayaPageTemplateFile('zpt/getting_started', globals(), 'ws_getting_started')

    wat_media = StaticServeFromFolder('media', globals(), cache=False)

InitializeClass(WSToolkit)
