--
-- DDL for views used within portal
-- Author(s): Cristian Romanescu <cristian.romanescu@eaudeweb.ro>
-- 


DROP SEQUENCE IF EXISTS synthetic_oid_seq CASCADE;
CREATE SEQUENCE synthetic_oid_seq CACHE 300 CYCLE;
COMMENT ON SEQUENCE synthetic_oid_seq IS 'Generates surrogate primary key for views that cannot build one within a single column. Needed by GIS tools and mapserver';



--
-- view_subdivision_locality :: List of localities each having hierarchical information about its parent subdivisions
--
-- This view is dynamic: %s is completed depending on the number of administrative levels for each country ([a, b, c, d, e] for levels [1, 2, 3, 4, 5])
--

DROP VIEW IF EXISTS view_subdivision_locality CASCADE;
CREATE OR REPLACE VIEW view_subdivision_locality AS
	SELECT nextval('synthetic_oid_seq') as oid, a.code AS level1, b.code AS level2,
			c.code AS level3, d.code AS level4, e.code AS level5, 
			loc.adlocode, loc.adloname, loc.adlocoordinate
		FROM t_adreglev a
		LEFT JOIN t_adreglev b ON a.code = b.parent_code
		LEFT JOIN t_adreglev c ON b.code = c.parent_code
		LEFT JOIN t_adreglev d ON c.code = d.parent_code
		LEFT JOIN t_adreglev e ON d.code = e.parent_code
		LEFT JOIN t_adloc loc ON {level}.code = loc.loadrcode
		ORDER BY a.code, b.code, c.code, d.code, e.code;
COMMENT ON VIEW view_subdivision_locality IS 'List of localities each having hierarchical information about its parent subdivisions';



--
-- view_subdivision :: Subdivision hierarchy
--
DROP VIEW IF EXISTS view_subdivision CASCADE;
CREATE OR REPLACE VIEW view_subdivision AS 
	SELECT a.code AS level1, b.code AS level2, c.code AS level3, d.code AS level4, e.code AS level5
		FROM t_adreglev a
		LEFT JOIN t_adreglev b ON a.code = b.parent_code
		LEFT JOIN t_adreglev c ON b.code = c.parent_code
		LEFT JOIN t_adreglev d ON c.code = d.parent_code
		LEFT JOIN t_adreglev e ON d.code = e.parent_code
		ORDER BY a.code, b.code, c.code, d.code, e.code;

COMMENT ON VIEW view_subdivision IS 'View with subdivisions hierarchical view';



--
-- view_water_point :: View with complete information about an water point (t_adloc + t_upis + t_upwp)
--
DROP VIEW IF EXISTS view_water_point CASCADE;
CREATE OR REPLACE VIEW view_water_point AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code,
			a.upisadr5code, loc.adlocode, loc.adloname, a.upiscode, a.upiscodex, a.upistype, a.upisname,
			a.upisname2, a.upisplace, a.upisloctype, a.upisx, a.upisy, a.upisz, a.upisyear, a.upisobj,
			a.upisdend, a.upistdesc, a.upiscomment, b.upwpmainuse, a.upismanprof, a.upisowner, a.upisprogram,
			a.upisfinance, a.upisupdate, a.upisupwho, a.upiscoordinate, b.upwptype, b.upwpbench, b.upwpantiq,
			b.upwpdiam, b.upwpdepth, b.upwpcoph, b.upwpcatch, b.upwprain, b.upwpnotfr, b.upwpnotfd, b.upwdlfdate,
			b.upwpasscw, b.upwpasspl, b.upwpasshy, b.upwpfence, b.upwpwturb, b.upwpwtast, b.upwpwodor,
			b.upwptreat, b.upwptreatf, b.upwppaymd, b.upwppayseas, b.upwphptype, b.upwphpfunc, b.upwphpyear,
			b.upwphpiwho, b.upwphpcraf, b.upwpmeter, b.upwpyield, b.upwpprero, b.upwpprpol, b.upwpdry,
			b.upwpclos, b.upwpabsq, b.upwpwlev, b.upwpeprice, b.upwpesaleu, d.code AS upisobj_code,
			d.name AS upisobj_name, e.ewp
		FROM t_upis a
		JOIN t_upwp b ON a.upiscode = b.upiscode
		LEFT JOIN t_adloc loc on a.upisadlocode = loc.adlocode
		LEFT JOIN lex_wp051 d ON a.upisobj = d.code
		LEFT JOIN lex_wp007 e ON b.upwptype = e.code
		WHERE a.approved = TRUE;

COMMENT ON VIEW view_water_point IS 'View with complete information about an water point (t_adloc + t_upis + t_upwp)';



--
-- view_water_point_unapproved :: Creates overview of unapproved water points from perspective t_adloc + t_upis + t_upwp
--
DROP VIEW IF EXISTS view_water_point_unapproved CASCADE;
CREATE OR REPLACE VIEW view_water_point_unapproved AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code,
			a.upisadr5code, loc.adlocode, loc.adloname, a.upiscode, a.upiscodex, a.upistype, a.upisname,
			a.upisname2, a.upisplace, a.upisloctype, a.upisx, a.upisy, a.upisz, a.upisyear, a.upisobj,
			a.upisdend, a.upistdesc, a.upiscomment, b.upwpmainuse, a.upismanprof, a.upisowner, a.upisprogram,
			a.upisfinance, a.upisupdate, a.upisupwho, a.upiscoordinate, b.upwptype, b.upwpbench, b.upwpantiq,
			b.upwpdiam, b.upwpdepth, b.upwpcoph, b.upwpcatch, b.upwprain, b.upwpnotfr, b.upwpnotfd, b.upwdlfdate,
			b.upwpasscw, b.upwpasspl, b.upwpasshy, b.upwpfence, b.upwpwturb, b.upwpwtast, b.upwpwodor,
			b.upwptreat, b.upwptreatf, b.upwppaymd, b.upwppayseas, b.upwphptype, b.upwphpfunc, b.upwphpyear,
			b.upwphpiwho, b.upwphpcraf, b.upwpmeter, b.upwpyield, b.upwpprero, b.upwpprpol, b.upwpdry, b.upwpclos,
			b.upwpabsq, b.upwpwlev, b.upwpeprice, b.upwpesaleu, d.code AS upisobj_code, d.name AS upisobj_name
		FROM t_upis a
		JOIN t_upwp b ON a.upiscode = b.upiscode
		LEFT JOIN t_adloc loc on a.upisadlocode = loc.adlocode
		LEFT JOIN lex_wp051 d ON a.upisobj = d.code
		WHERE a.approved <> TRUE;
COMMENT ON VIEW view_water_point_unapproved IS 'Creates overview of unapproved water points';



--
-- view_sanitation_facility :: Creates overview of sanitation facilities from perspective t_adloc + t_upis + t_upsf
--
DROP VIEW IF EXISTS view_sanitation_facility CASCADE;
CREATE OR REPLACE VIEW view_sanitation_facility AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code,
			a.upisadr5code, loc.adlocode, loc.adloname, a.upiscode, a.upiscodex, a.upistype, a.upisname, a.upisname2,
			a.upisplace, a.upisloctype, a.upisx, a.upisy, a.upisz, a.upisyear, a.upisobj, a.upisdend, a.upistdesc,
			a.upiscomment, a.upismanprof, a.upisowner, a.upisprogram, a.upisfinance, a.upisupdate, a.upisupwho,
			a.upiscoordinate, b.upsftype, b.upsfmatype, b.upsfgrtype, b.upsfmwsep, b.upsfblocnbr, b.upsfblomnbr,
			b.upsfblownbr, b.upsfshower, b.upsfwhand, b.upwswbas, b.upsfwater, b.upsffunc, b.upsfnotfr,
			b.upsfnotfd, b.upsfasscw, b.upsfasspl, b.upsfasscl, b.upsfpaymd, d.code as upisobj_code,
			d.name as upisobj_name
		FROM t_upis a
		JOIN t_upsf b ON a.upiscode = b.upiscode
		LEFT JOIN t_adloc loc on a.upisadlocode = loc.adlocode
		LEFT JOIN lex_wp051 d ON a.upisobj = d.code
		WHERE a.approved = TRUE;
COMMENT ON VIEW view_sanitation_facility IS 'Creates overview of approved sanitation facilities from perspective t_adloc + t_upis + t_upsf. This way we can get full information for sanitation facility (t_upis + t_upsf) and filter water points on locality (t_adloc)';



--
-- view_sanitation_facility_unapproved :: Creates overview of unapproved sanitation facilities from perspective t_adloc + t_upis + t_upsf
--
DROP VIEW IF EXISTS view_sanitation_facility_unapproved CASCADE;
CREATE OR REPLACE VIEW view_sanitation_facility_unapproved AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code,
			a.upisadr5code, loc.adlocode, loc.adloname, a.upiscode, a.upiscodex, a.upistype, a.upisname,
			a.upisname2, a.upisplace, a.upisloctype, a.upisx, a.upisy, a.upisz, a.upisyear, a.upisobj,
			a.upisdend, a.upistdesc, a.upiscomment, a.upismanprof, a.upisowner, a.upisprogram, a.upisfinance,
			a.upisupdate, a.upisupwho, a.upiscoordinate, b.upsftype, b.upsfmatype, b.upsfgrtype, b.upsfmwsep,
			b.upsfblocnbr, b.upsfblomnbr, b.upsfblownbr, b.upsfshower, b.upsfwhand, b.upwswbas, b.upsfwater,
			b.upsffunc, b.upsfnotfr, b.upsfnotfd, b.upsfasscw, b.upsfasspl, b.upsfasscl, b.upsfpaymd,
			d.code as upisobj_code, d.name as upisobj_name
		FROM t_upis a
		JOIN t_upsf b ON a.upiscode = b.upiscode
		LEFT JOIN t_adloc loc on a.upisadlocode = loc.adlocode
		LEFT JOIN lex_wp051 d ON a.upisobj = d.code
		WHERE a.approved <> TRUE;
COMMENT ON VIEW view_sanitation_facility_unapproved IS 'Creates overview of unapproved sanitation facilities';



--
-- view_locality_statistics :: List of localities statistics data used for localities list inside Access section.
--
DROP VIEW IF EXISTS view_locality_statistics CASCADE;
CREATE OR REPLACE VIEW view_locality_statistics AS 
	SELECT a.adloname, a.adlocode, b.population, b.water_access, a.sf_access, b.pop_served_water,
			b.pop_served_sanitation, c.level1, c.level2, c.level3, c.level4, c.level5
		FROM t_adloc a
		LEFT JOIN t_adloc_stat b ON a.adlocode = b.adlocode
		LEFT JOIN view_subdivision_locality c ON c.adlocode = a.adlocode;
COMMENT ON VIEW view_locality_statistics IS 'List of localities statistics data used for localities list inside Access section';



--
-- view_subdivision_statistics :: View used to compute the statistical data on subdivisions
--
DROP VIEW IF EXISTS view_subdivision_statistics CASCADE;
CREATE OR REPLACE VIEW view_subdivision_statistics AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.level1, a.level2, a.level3, a.level4, a.level5, b.adlocode,
			b.loadrcode, b.adloname, b.adlopop, b.adlopopyear, b.adlopopgr, b.adlopophh, b.adlostatus, b.adlohgroup,
			b.sf_access, b.adlowamain, b.adlosfreq, b.adloprivwell, b.adlopubwell, b.adloprivlatr, b.adloschool,
			b.adlohealth, c.population, c.adlopopupe, c.adlohhnbr, c.pop_served_water, c.water_access,
			c.water_functionality_rate, c.water_inventory_rate, c.national_rank, c.water_points, c.wp_inventoried,
			c.wp_func_equiv, c.wp_nonfunc_equiv, c.pop_served_sanitation, c.sf_coverage_rate, c.sf_func_rate,
			c.sf_inventory_rate, c.sf_existing_required, c.sf_existing_inventoried, c.sf_functional,
			c.sf_nonfunctional
		FROM view_subdivision_locality a
		JOIN t_adloc b ON a.adlocode = b.adlocode
		JOIN t_adloc_stat c ON c.adlocode = b.adlocode;
COMMENT ON VIEW view_subdivision_statistics IS 'View used to compute the statistical data on subdivisions';



--
-- view_pws :: View with complete information about a piped water scheme(t_adloc + t_upis + t_uppws)
--
DROP VIEW IF EXISTS view_pws CASCADE;
CREATE OR REPLACE VIEW view_pws AS 
	SELECT nextval('synthetic_oid_seq'::regclass) AS oid, a.upisadr1code, a.upisadr2code, a.upisadr3code,
			a.upisadr4code, a.upisadr5code, loc.adlocode, loc.adloname, a.upiscode, a.upiscodex, a.upistype,
			a.upisname, a.upisname2, a.upisplace, a.upisloctype, a.upisx, a.upisy, a.upisz, a.upisyear, a.upisdend,
			a.upistdesc, a.upiscomment, a.upismanprof, a.upisowner, a.upisprogram, a.upisfinance, a.upisupdate,
			a.upisupwho, a.upiscoordinate, b.uppwsowntype, b.uppwsctrar, b.uppwspop, b.uppwslonbr, b.uppwsstcpdi,
			b.uppwsminhd, b.uppwsstcp, b.uppwslentot, b.uppwstapnbr, b.uppwsconnbr, b.uppwscapnbr, b.uppwscatnbr,
			b.uppwswdpnbr, b.uppwsfunc, b.uppwsabdm, b.uppwscapday, b.uppwscapdr, b.uppwsenerc, b.uppwsextent,
			b.uppwsschlen, b.uppwsabstpe, b.uppwschldis, b.uppwstpsys, b.uppwspricept, b.uppwspricedt,
			b.uppwspriceos, b.uppwspriceun, d.name AS uppwsfunc_name
		FROM t_upis a
		INNER JOIN t_uppws b ON a.upiscode = b.upiscode
		LEFT JOIN t_adloc loc ON a.upisadlocode = loc.adlocode
		LEFT JOIN lex_pws020 d ON b.uppwsfunc = d.code
		WHERE a.approved = TRUE;
COMMENT ON VIEW view_pws IS 'View with complete information about a piped water scheme(t_adloc + t_upis + t_uppws)';



-----------------------------------------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------- GIS VIEWS --------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------

--
-- view_gis_wp :: List of water points for postgis-MapServer relation
-- This view is separated from view_water_point due to performance reasons. Joined columns slows down postgis engine and we don't use those columns for GIS
--
DROP VIEW IF EXISTS view_gis_wp CASCADE;
CREATE OR REPLACE VIEW view_gis_wp AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.upiscode, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code, a.upisadr5code, 
			a.upisadlocode, b.upwptype, a.upisloctype, a.upisobj, a.upiscoordinate
		FROM t_upis a
		JOIN t_upwp b ON a.upiscode = b.upiscode
		LEFT JOIN t_adloc loc on a.upisadlocode = loc.adlocode
		WHERE a.approved = TRUE;
COMMENT ON VIEW view_gis_wp IS 'List of water points for postgis-MapServer relation';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type") 
	VALUES ('', 'public', 'view_gis_wp', 'upiscoordinate', 2, 4326, 'POINT');



--
-- view_gis_sf :: List of sanitation facilities for postgis-MapServer relation
-- This view is separated from view_sanitation_facility due to performance reasons. Joined columns slows down postgis engine.
--
DROP VIEW IF EXISTS view_gis_sf CASCADE;
CREATE OR REPLACE VIEW view_gis_sf AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.upiscode, b.upsftype, a.upisloctype, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code, a.upisadr5code, 
			a.upisadlocode, a.upisobj, a.upiscoordinate
		FROM t_upsf b 
		JOIN t_upis a ON b.upiscode = a.upiscode
		LEFT JOIN t_adloc loc on a.upisadlocode = loc.adlocode
		WHERE a.approved = TRUE;
COMMENT ON VIEW view_gis_sf IS 'List of sanitation facilities for postgis-MapServer relation';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type") 
	VALUES ('', 'public', 'view_gis_sf', 'upiscoordinate', 2, 4326, 'POINT');



--
-- view_gis_proj_loc :: List of localities attached to projects. Used for GIS queries
--
DROP VIEW IF EXISTS view_gis_proj_loc CASCADE;
CREATE OR REPLACE VIEW view_gis_proj_loc AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.oploadopcode AS project, c.level1, c.level2, c.level3, c.level4, c.level5,
			loc.adlocode AS locality, loc.adlocoordinate
		FROM tl_adoploc a
		INNER JOIN t_adloc loc on a.oploadlocode = loc.adlocode
		JOIN view_subdivision c ON loc.loadrcode = c.{level_column};
COMMENT ON VIEW view_gis_proj_loc IS 'List of localities attached to projects. Used for GIS queries';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type")
	VALUES ('', 'public', 'view_gis_proj_loc', 'adlocoordinate', 2, 4326, 'POINT');



--
-- view_gis_tl_adopr :: List of subdivisions attached to projects. Used for GIS, but NOT by mapserver
--
DROP VIEW IF EXISTS view_gis_tl_adopr CASCADE;
CREATE OR REPLACE VIEW view_gis_tl_adopr AS 
	SELECT b.code, a.opradopcode FROM tl_adopr AS a INNER JOIN t_adreglev AS b on b.code = a.opradrcode;
COMMENT ON VIEW view_gis_tl_adopr IS 'List of subdivisions attached to projects. Used for GIS, but NOT by mapserver';



--
-- view_gis_subdivision_locality :: List of localities with subdivision information on all levels. Used for GIS.
--
-- This view is dynamic: %s is completed depending on the number of administrative 
-- levels for each country ([a, b, c, d, e] for levels [1, 2, 3, 4, 5])
--
DROP VIEW IF EXISTS view_gis_subdivision_locality CASCADE;
CREATE OR REPLACE VIEW view_gis_subdivision_locality AS 
	SELECT nextval('synthetic_oid_seq') as oid, a.code AS level1, b.code AS level2,
			c.code AS level3, d.code AS level4, e.code AS level5, loc.adlocode, loc.adlocoordinate
		FROM t_adreglev a
		LEFT JOIN t_adreglev b ON a.code = b.parent_code
		LEFT JOIN t_adreglev c ON b.code = c.parent_code
		LEFT JOIN t_adreglev d ON c.code = d.parent_code
		LEFT JOIN t_adreglev e ON d.code = e.parent_code
		LEFT JOIN t_adloc loc ON {level}.code = loc.loadrcode
		ORDER BY a.code, b.code, c.code, d.code, e.code;

COMMENT ON VIEW view_gis_subdivision_locality IS 'List of localities with subdivision information on all levels. Used for GIS.';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type")
	VALUES ('', 'public', 'view_gis_subdivision_locality', 'adlocoordinate', 2, 4326, 'POINT');



--
-- view_gis_pws :: List of piped water schemes for postgis-MapServer relation
-- This view is separated from view_pws due to performance reasons. Joined columns slows down postgis engine and we don't use those columns for GIS
--
DROP VIEW IF EXISTS view_gis_pws CASCADE;
CREATE OR REPLACE VIEW view_gis_pws AS
	SELECT nextval('synthetic_oid_seq') as oid, a.upiscode, a.upisadr1code, a.upisadr2code, a.upisadr3code, a.upisadr4code, a.upisadr5code, 
			a.upisadlocode, b.uppwsowntype, a.upisloctype, b.uppwsfunc, a.upiscoordinate
		FROM t_upis a
		JOIN t_uppws b ON a.upiscode::text = b.upiscode::text
		LEFT JOIN t_adloc loc ON a.upisadlocode::text = loc.adlocode::text
		LEFT JOIN lex_pws020 d ON b.uppwsfunc::text = d.code::text
		WHERE a.approved = true;

COMMENT ON VIEW view_gis_pws IS 'List of piped water schemes for postgis-MapServer relation';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type")
	VALUES ('', 'public', 'view_gis_pws', 'upiscoordinate', 2, 4326, 'POINT');



--
-- view_gis_pws_loc :: List of localities linked to piped water scheme 
--
DROP VIEW IF EXISTS view_gis_pws_loc CASCADE;
CREATE OR REPLACE VIEW view_gis_pws_loc AS
	SELECT nextval('synthetic_oid_seq') as oid, A.pwsisupiscode, B.adlocode, B.adlocoordinate 
		FROM tl_aulocpws A
		INNER JOIN t_adloc B on B.adlocode = A.loadlocode;

COMMENT ON VIEW view_gis_pws_loc IS 'List of localities linked to piped water scheme';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type")
	VALUES ('', 'public', 'view_gis_pws_loc', 'adlocoordinate', 2, 4326, 'POINT');



--
-- view_gis_pws_wp :: List of water points linked to piped water scheme 
--
DROP VIEW IF EXISTS view_gis_pws_wp CASCADE;
CREATE OR REPLACE VIEW view_gis_pws_wp AS
	SELECT nextval('synthetic_oid_seq') as oid, A.pwsisupiscode, B.upiscode, B.upiscoordinate 
		FROM tl_uppwswp A
		INNER JOIN t_upis B on A.wpisupiscode = B.upiscode;

COMMENT ON VIEW view_gis_pws_wp IS 'List of water points linked to piped water scheme';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type")
	VALUES ('', 'public', 'view_gis_pws_wp', 'upiscoordinate', 2, 4326, 'POINT');



--
-- view_gis_uws :: List of undeground water sources
--
DROP VIEW IF EXISTS view_gis_uws CASCADE;
CREATE OR REPLACE VIEW view_gis_uws AS
	SELECT nextval('synthetic_oid_seq') as oid, A.upiscode, A.upisname, 
			A.upisadr1code, A.upisadr2code, A.upisadr3code, A.upisadr4code, A.upisadr5code, 
			A.upisadlocode, C.adloname, a.upiscoordinate, B.upwputype, D.name as upwputype_name, 
			A.upisyear, B.upwpwlev, B.upwpdiam, B.upwpdepth, B.upwpuyelm, B.upwpudepin,
			B.fk_aqfcode, E.aqfname, A.upisx, A.upisy, B.upwpudraw
		FROM t_upis A
		INNER JOIN t_upwp B ON B.upiscode = A.upiscode
		LEFT JOIN t_adloc C on C.adlocode = A.upisadlocode
		JOIN lex_uws005 D on D.code = B.upwputype
		LEFT JOIN t_hyaquifer E on b.fk_aqfcode = E.aqfcode
		WHERE A.upistype = 'UWS';

COMMENT ON VIEW view_gis_uws IS 'List of undeground water sources';
INSERT INTO geometry_columns(f_table_catalog, f_table_schema, f_table_name, f_geometry_column, coord_dimension, srid, "type")
	VALUES ('', 'public', 'view_gis_uws', 'upiscoordinate', 2, 4326, 'POINT');
