--
-- DDL for portal tables
-- Author(s): Cristian Romanescu <cristian.romanescu@eaudeweb.ro>
-- 
-- This file contains the database structure for the WatSan country portal
-- IMPORTANT NOTES, PAY ATTENTION!
-- * When finishing all import (including data), manually run VACUUM FULL to optimize database structures
-- * Rules to fill this script
--		* 3 spaces between entities
--		* constraints on a SINGLE line
--		* BEFORE IMPORT THIS SCRIPT IMPORT ALSO COMMON.SQL OTHERWISE YOU WILL GET ERRORS
--		* Index naming: idx_table_column ...
-- 		* Foreign key format: fk_src_table_src_column_foreign_table ...
--		* DO NOT, I repeat, DO NOT put comments on same line with SQL (sql parser will be broken and portal creation fails)!
--		* If unsure how to format, look at other tables, they have all (95%) the possible cases
--		* There will be no ALTER TABLE * OWNER TO *!
--		* Be very careful with order of entities, they may depend on each other and script will fail.
--		* PLEASE TEST SCRIPT BEFORE COMMITING TO REPOSITORY, IT IS A LIVE SCRIPT, USED BY ONLINE PORTALS.
--		* Definition order is: 
--				1. tables (create/index/comments) - starts with t_
--				2. linked_tables - starts with tl_
--				3. views - starts with view_

-- Clear geometry_columns, are added again by each script below
DELETE FROM geometry_columns;

--
-- Table: t_adcountry_stat
--
DROP TABLE IF EXISTS t_adcountry_stat CASCADE;
CREATE TABLE t_adcountry_stat
(
	"year" integer NOT NULL,
	water_access_rural real NOT NULL DEFAULT 0,
	water_access_urban real NOT NULL DEFAULT 0,
	sanitation_access_rural real NOT NULL DEFAULT 0,
	sanitation_access_urban real NOT NULL DEFAULT 0,
	CONSTRAINT pk_t_adcountry_stat PRIMARY KEY ("year")
);

CREATE INDEX idx_t_adcountry_stat_year ON t_adcountry_stat USING btree ("year");

COMMENT ON TABLE t_adcountry_stat IS 'Descriptive data related to access to water and sanitation at country level. Historical data (CTRY_STAT)';
COMMENT ON COLUMN t_adcountry_stat.year IS 'Year of reference for the statistical data (CST002)';
COMMENT ON COLUMN t_adcountry_stat.water_access_rural IS 'Access rate to water in rural and semi-urban areas. Subunitar number (CST003)';
COMMENT ON COLUMN t_adcountry_stat.water_access_urban IS 'Access rate to water in urban areas. Subunitar number (CST004)';
COMMENT ON COLUMN t_adcountry_stat.sanitation_access_rural IS 'Access rate to sanitation in rural and semi-urban areas. Subunitar number (CST005)';
COMMENT ON COLUMN t_adcountry_stat.sanitation_access_urban IS 'Access rate to sanitation in urban areas. Subunitar number (CST006)';



--
-- Table: t_adreglev (SU5)
--
DROP TABLE IF EXISTS t_adreglev CASCADE;
CREATE TABLE t_adreglev
( 
	id serial NOT NULL,
	code character varying NOT NULL,
	codex character varying(20),
	parent_code character varying,
	level smallint NOT NULL,
	name character varying NOT NULL,
	name_en character varying,
	
	CONSTRAINT pk_t_adreglev PRIMARY KEY (id),
	CONSTRAINT unq_t_adreglev_code UNIQUE (code),
	CONSTRAINT fk_t_adreglev_parent_code FOREIGN KEY (parent_code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE INDEX idx_t_adreglev_code ON t_adreglev USING btree (code);
CREATE INDEX idx_t_adreglev_parent_code ON t_adreglev USING btree (parent_code);
CREATE INDEX idx_t_adreglev_level ON t_adreglev USING btree (level);
CREATE INDEX idx_t_adreglev_name ON t_adreglev USING btree (name);

COMMENT ON TABLE t_adreglev IS 'Administrative regions for this country (SU5)';
COMMENT ON COLUMN t_adreglev.id IS 'ID of the region (primary key)';
COMMENT ON COLUMN t_adreglev.code IS 'Administrative code of the subdivision (SU001)';
COMMENT ON COLUMN t_adreglev.codex IS 'Code of the subdivision in sector database (SU003)';
COMMENT ON COLUMN t_adreglev.parent_code IS 'Parent administrative region ID (NULL for level 1 regions) (SU002)';
COMMENT ON COLUMN t_adreglev.level IS 'Region hierarchical level';
COMMENT ON COLUMN t_adreglev.name IS 'Region name in original language (SU004)';
COMMENT ON COLUMN t_adreglev.name_en IS 'Region name in english';



--
-- Table: t_adreglev_stat (SU5)
--
DROP TABLE IF EXISTS t_adreglev_stat;
CREATE TABLE t_adreglev_stat
(
	id serial NOT NULL,
	code character varying(20) NOT NULL,
	localities integer DEFAULT 0,
	pop_estimated integer DEFAULT 0,
	households_estimated integer DEFAULT 0,
	pop_served_water integer DEFAULT 0,
	water_access double precision DEFAULT 0,
	water_functionality_rate double precision DEFAULT 0,
	water_inventory_rate double precision DEFAULT 0,
	national_rank smallint,
	water_points integer DEFAULT 0,
	wp_inventoried integer DEFAULT 0,
	wp_func_equiv double precision DEFAULT 0,
	wp_nonfunc_equiv double precision DEFAULT 0,
	pop_served_sanitation integer DEFAULT 0,
	sanitation_access double precision DEFAULT 0,
	sf_coverage_rate double precision DEFAULT 0,
	sf_func_rate double precision DEFAULT 0,
	sf_inventory_rate double precision DEFAULT 0,
	sf_required integer DEFAULT 0,
	sf_existing_required integer DEFAULT 0,
	sf_existing_inventoried integer DEFAULT 0,
	sf_functional integer DEFAULT 0,
	sf_nonfunctional integer DEFAULT 0,
	school integer DEFAULT 0,
	health integer DEFAULT 0,
	CONSTRAINT pk_t_adreglev_stat PRIMARY KEY (id),
	CONSTRAINT uniq_t_adreglev_code UNIQUE (code),
	CONSTRAINT fk_t_adreglev_stat_code FOREIGN KEY (code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_adreglev_stat_code ON t_adreglev_stat USING btree (code);

COMMENT ON TABLE t_adreglev_stat IS 'Statistical data on administrative regions (SU5)';
COMMENT ON COLUMN t_adreglev_stat.id IS 'ID of the entry';
COMMENT ON COLUMN t_adreglev_stat.code IS 'Administrative code of the subdivision (SU5001)';
COMMENT ON COLUMN t_adreglev_stat.localities IS 'Number of localities in the subdivision, formula:[COUNT(t_adloc) WHERE loadrcode = code] (SU5005)';
COMMENT ON COLUMN t_adreglev_stat.pop_estimated IS 'Total estimated population of the subdivision for the present year, formula:[SUM(t_adloc_stat.population(LOC012)) WHERE loadrcode = code] (SU5006)';
COMMENT ON COLUMN t_adreglev_stat.households_estimated IS 'Total estimated households of the subdivision for the present year, formula:[SUM(t_adloc.adlohhnbr(LOC014)) WHERE loadrcode = code] (SU5007)';
COMMENT ON COLUMN t_adreglev_stat.pop_served_water IS 'Population served with safe water, formula:[(= sum([LOC017]) for the localities of the subdivision)] (SU5008)';
COMMENT ON COLUMN t_adreglev_stat.water_access IS 'Access rate to water, formula:[[pop_served_water(SU5008)] / [pop_estimated(SU5006)]] (SU5009)';
COMMENT ON COLUMN t_adreglev_stat.water_functionality_rate IS 'Functionaliy rate of the water points, formula:[wp_func_equiv(SU5015) / ( wp_func_equiv(SU5015) + wp_nonfunc_equiv(SU5016) )] (SU5010)';
COMMENT ON COLUMN t_adreglev_stat.water_inventory_rate IS 'Inventory rate of the public water points, formula: [wp_inventoried(SU5014) / water_points(SU5013)] (SU5011)';
COMMENT ON COLUMN t_adreglev_stat.national_rank IS 'National rank of the locality for access to water, formula: Determine the quintiles of the LOC018 series of values, i.e. the 5 values of access rate AR1, AR2, AR3, AR4, AR5 for which there is an equivalent number of localities in each interval (SU5012)';
COMMENT ON COLUMN t_adreglev_stat.water_points IS 'Number of existing public water points, formula: [(= sum([LOC022]) for the localities of the subdivision)] (SU5013)';
COMMENT ON COLUMN t_adreglev_stat.wp_inventoried IS 'Number of inventoried public water points, formula: [(= sum([LOC023]) for the localities of the subdivision)] (SU5014)';
COMMENT ON COLUMN t_adreglev_stat.wp_func_equiv IS 'Number of functional equivalent water points, formula:[(= sum([LOC024]) for the localities of the subdivision)] (SU5015)';
COMMENT ON COLUMN t_adreglev_stat.wp_nonfunc_equiv IS 'Number of non-functional equivalent water points, formula: [(= sum([LOC025]) for the localities of the subdivision)] (SU5016)';
COMMENT ON COLUMN t_adreglev_stat.pop_served_sanitation IS 'Population served with sanitation, formula: [(= sum([LOC026) for the localities of the subdivision)] (SU5017)';
COMMENT ON COLUMN t_adreglev_stat.sanitation_access IS 'Access rate to sanitation estimated by survey, formula: [pop_served_sanitation(SU5017) / pop_estimated(SU5006)] (SU5018)';
COMMENT ON COLUMN t_adreglev_stat.sf_coverage_rate IS 'Coverage rate of needs for public toilets, formula: [sf_required(SU5023) / sf_existing_required(SU5022)] (SU5019)';
COMMENT ON COLUMN t_adreglev_stat.sf_func_rate IS 'Functionaliy rate of the public toilets, formula: [sf_functional(SU5025) / ( sf_functional(SU5025) + sf_nonfunctional(SU5026) )] (SU5020)';
COMMENT ON COLUMN t_adreglev_stat.sf_inventory_rate IS 'Inventory rate of the public toilets, formula: [sf_existing_inventoried(SU5024) / sf_required(SU5023)] (SU5021)';
COMMENT ON COLUMN t_adreglev_stat.sf_required IS 'Number of required public toilets, formula: [(= sum([LOC031]) for the localities of the subdivision)] (SU5022)';
COMMENT ON COLUMN t_adreglev_stat.sf_existing_required IS 'Number of required existing toilets, formula: [(= sum([LOC032]) for the localities of the subdivision)] (SU5023)';
COMMENT ON COLUMN t_adreglev_stat.sf_existing_inventoried IS 'Number of inventoried existing toilets, formula: [(= sum([LOC033]) for the localities of the subdivision)] (SU5024)';
COMMENT ON COLUMN t_adreglev_stat.sf_functional IS 'Number of functional public toilets, formula: [(= sum([LOC034]) for the localities of the subdivision)] (SU5025)';
COMMENT ON COLUMN t_adreglev_stat.sf_nonfunctional IS 'Number of non-functional public toilets, formula: [(= sum([LOC035]) for the localities of the subdivision)] (SU5026)';
COMMENT ON COLUMN t_adreglev_stat.school IS 'Number of scholar infrastructures in the subdivision, formula: [(= sum([LOC036]) for the localities of the subdivision)] (SU5027)';
COMMENT ON COLUMN t_adreglev_stat.health IS 'Number of health infrastructures in the subdivision, formula: [(= sum([LOC037]) for the localities of the subdivision)] (SU5028)';

--
-- Table: t_adreglev_comments
--

CREATE TABLE t_adreglev_comments
(
	id serial NOT NULL,
	code character varying(20),
	"user" character varying(20),
	user_name character varying(100),
	user_email character varying(100),
	submit_date date,
	rating integer,
	"comment" text,
	ip_address character varying(16),
	is_public boolean,
	is_removed boolean,
	CONSTRAINT pk_adreglev_comments PRIMARY KEY (id),
	CONSTRAINT fk_t_adreglev_comments_code_t_adreglev FOREIGN KEY (code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_adreglev_comments_fk_code ON t_adreglev_comments USING btree (code);

COMMENT ON TABLE t_adreglev_comments IS 'Comments for localities';
COMMENT ON COLUMN t_adreglev_comments.id IS 'Identifier for the comment';
COMMENT ON COLUMN t_adreglev_comments.code IS 'Identifier for the commented locality';
COMMENT ON COLUMN t_adreglev_comments."user" IS 'Identifier for the user who posted the comment';
COMMENT ON COLUMN t_adreglev_comments.user_name IS 'User name';
COMMENT ON COLUMN t_adreglev_comments.user_email IS 'User email address';
COMMENT ON COLUMN t_adreglev_comments.submit_date IS 'Date of the post';
COMMENT ON COLUMN t_adreglev_comments.rating IS 'Rating given by user';
COMMENT ON COLUMN t_adreglev_comments."comment" IS 'The comment';
COMMENT ON COLUMN t_adreglev_comments.ip_address IS 'IP address of the commenter';
COMMENT ON COLUMN t_adreglev_comments.is_public IS '1 if comment is public, 0 for restricted';
COMMENT ON COLUMN t_adreglev_comments.is_removed IS '1 if comment is removed, 0 for available';


--
-- Table: t_adloc (LOC)
-- 
DROP TABLE IF EXISTS t_adloc CASCADE;
CREATE TABLE t_adloc
(
	adlocid serial NOT NULL,
	adlocode character varying(20) NOT NULL,
	loadrcode character varying(20),
	adlocodex character varying(20),
	adloname character varying,
	adloname2 character varying,
	adlox double precision,
	adloy double precision,
	adlopop integer,
	adlopopyear integer,
	adlopopgr double precision,
	adlopophh double precision,
	adlostatus character varying(2),
	adlohgroup character varying(2),
	sf_access double precision DEFAULT 0,
	adlowamain character varying(2),
	adlomarket boolean,
	adlovolt character varying(2),
	adlotelline boolean,
	adlotelmob boolean,
	adloweb boolean,
	adloupdate date DEFAULT ('now'::text)::date,
	adloupwho character varying,
	adlosfreq integer NOT NULL DEFAULT 0,
	adloprivwell integer NOT NULL DEFAULT 0,
	adlopubwell integer NOT NULL DEFAULT 0,
	adloprivlatr integer NOT NULL DEFAULT 0,
	adloschool integer NOT NULL DEFAULT 0,
	adlohealth integer NOT NULL DEFAULT 0,

	CONSTRAINT pk_t_adloc PRIMARY KEY (adlocid),
	CONSTRAINT unq_t_adloc_adlocode UNIQUE (adlocode),
	CONSTRAINT fk_t_adloc_adlohgroup_lex_loc016 FOREIGN KEY (adlohgroup) REFERENCES lex_loc016 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adloc_adlostatus_lex_loc015 FOREIGN KEY (adlostatus) REFERENCES lex_loc015 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adloc_adlovolt_lex_loc043 FOREIGN KEY (adlovolt) REFERENCES lex_loc043 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adloc_adlowamain_lex_loc041 FOREIGN KEY (adlowamain) REFERENCES lex_loc041 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adloc_loadrcode_t_adreglev FOREIGN KEY (loadrcode) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

SELECT AddGeometryColumn('public', 't_adloc', 'adlocoordinate', 4326, 'POINT', 2);

CREATE INDEX idx_t_adloc_adlocode ON t_adloc USING btree(adlocode);
CREATE INDEX idx_t_adloc_adloname ON t_adloc USING btree(adloname);
CREATE INDEX idx_t_adloc_loadrcode ON t_adloc USING btree(loadrcode);
CREATE INDEX idx_t_adloc_adlopop ON t_adloc USING btree(adlopop);
CREATE INDEX idx_t_adloc_adlohgroup ON t_adloc USING btree(adlohgroup);
CREATE INDEX idx_t_adloc_adlostatus ON t_adloc USING btree(adlostatus);
CREATE INDEX idx_t_adloc_sf_access ON t_adloc USING btree(sf_access);
CREATE INDEX idx_t_adloc_adlovolt ON t_adloc USING btree(adlovolt);
CREATE INDEX idx_t_adloc_adlowamain ON t_adloc USING btree(adlowamain);
CREATE INDEX idx_t_adloc_gis_adlocoordinate ON t_adloc USING gist (adlocoordinate);

COMMENT ON TABLE t_adloc IS 'List and characteristics of localities (municipalities and villages) as defined following national regulations. (LOC)';
COMMENT ON COLUMN t_adloc.adlocid IS 'Primary key of the table';
COMMENT ON COLUMN t_adloc.adlocode IS 'Unique locality code defined by official producer (LOC001)';
COMMENT ON COLUMN t_adloc.loadrcode IS 'Parent subdivision code - Code of the last administrative subdivision (LOC002)';
COMMENT ON COLUMN t_adloc.adlocodex IS 'Code of the locality in the sector database (LOC003)';
COMMENT ON COLUMN t_adloc.adloname IS 'Name defined by official producer (LOC004)';
COMMENT ON COLUMN t_adloc.adloname2 IS 'Other name of the locality (LOC005)';
COMMENT ON COLUMN t_adloc.adlox IS 'Coordinate X in decimal degree (LOC006)';
COMMENT ON COLUMN t_adloc.adloy IS 'Coordinate Y in decimal degree (LOC007)';
COMMENT ON COLUMN t_adloc.adlopop IS 'Population of the last census (LOC008)';
COMMENT ON COLUMN t_adloc.adlopopyear IS 'Estimated population for the present year (LOC009)';
COMMENT ON COLUMN t_adloc.adlopopgr IS 'Estimated growth rate of population (LOC010)';
COMMENT ON COLUMN t_adloc.adlopophh IS 'Mean number of persons per household (LOC011)';
COMMENT ON COLUMN t_adloc.adlostatus IS 'Urban / semi-urban / rural status (LOC015)';
COMMENT ON COLUMN t_adloc.adlohgroup IS 'Type of habitat grouping of the locality (LOC016)';
COMMENT ON COLUMN t_adloc.sf_access IS 'Access rate to sanitation estimated by survey (Entry data obtained from treatment of national household surveys) (LOC027)';
COMMENT ON COLUMN t_adloc.adlowamain IS 'Main access to water (LOC041)';
COMMENT ON COLUMN t_adloc.adlomarket IS 'Existence of weekly market (LOC042)';
COMMENT ON COLUMN t_adloc.adlovolt IS 'Locality served with a low voltage grid (LOC043)';
COMMENT ON COLUMN t_adloc.adlotelline IS 'Locality served with telephone land line (LOC044)';
COMMENT ON COLUMN t_adloc.adlotelmob IS 'Locality covered by a mobile network (LOC045)';
COMMENT ON COLUMN t_adloc.adloweb IS 'Access to internet in the locality (LOC046)';
COMMENT ON COLUMN t_adloc.adloupdate IS 'Date of last updating';
COMMENT ON COLUMN t_adloc.adloupwho IS 'Username of the responsible for last updating';
COMMENT ON COLUMN t_adloc.adlocoordinate IS 'Geography column with locality geographical coordinates (LOC098)';
COMMENT ON COLUMN t_adloc.adlosfreq IS 'Number of required public toilets (LOC031)';
COMMENT ON COLUMN t_adloc.adloprivwell IS 'Estimated number of traditional private wells in the locality (LOC036)';
COMMENT ON COLUMN t_adloc.adlopubwell IS 'Estimated number of traditional public wells in the locality (LOC037)';
COMMENT ON COLUMN t_adloc.adloprivlatr IS 'Estimated number of traditional private latrines in the locality (LOC038)';
COMMENT ON COLUMN t_adloc.adloschool IS 'Number of scholar infrastructures in the locality (LOC039)';
COMMENT ON COLUMN t_adloc.adlohealth IS 'Number of health infrastructures in the locality (LOC040)';



--
-- Table: t_adloc_stat
--
DROP TABLE IF EXISTS t_adloc_stat;
CREATE TABLE t_adloc_stat
(
	id serial NOT NULL,
	adlocode character varying(20) NOT NULL,
	population integer DEFAULT 0,
	adlopopupe integer DEFAULT 0,
	adlohhnbr integer DEFAULT 0,
	pop_served_water integer DEFAULT 0,
	water_access double precision DEFAULT 0,
	water_functionality_rate double precision DEFAULT 0,
	water_inventory_rate double precision DEFAULT 0,
	national_rank smallint,
	water_points integer DEFAULT 0,
	wp_inventoried integer DEFAULT 0,
	wp_func_equiv double precision DEFAULT 0,
	wp_nonfunc_equiv double precision DEFAULT 0,
	pop_served_sanitation integer DEFAULT 0,
	sf_coverage_rate double precision DEFAULT 0,
	sf_func_rate double precision DEFAULT 0,
	sf_inventory_rate double precision DEFAULT 0,
	sf_existing_required integer DEFAULT 0,
	sf_existing_inventoried integer DEFAULT 0,
	sf_functional integer DEFAULT 0,
	sf_nonfunctional integer DEFAULT 0,

	CONSTRAINT pk_t_adloc_stat PRIMARY KEY (id),
	CONSTRAINT uniq_t_adloc_stat_code UNIQUE (adlocode),
	CONSTRAINT fk_t_adloc_stat_code FOREIGN KEY (adlocode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_adloc_stat_adlocode ON t_adloc_stat USING btree (adlocode);

COMMENT ON TABLE t_adloc_stat IS 'Statistical data on localities';
COMMENT ON COLUMN t_adloc_stat.id IS 'ID of the entry';
COMMENT ON COLUMN t_adloc_stat.adlocode IS 'Code of the locality (LOC001)';
COMMENT ON COLUMN t_adloc_stat.population IS 'Calculated population of the present year, formula: LOC008 x (1 + LOC010)^(YEAR - LOC009) (LOC012)';
COMMENT ON COLUMN t_adloc_stat.adlopopupe IS 'Estimated population for the present year, formula: By default is LOC012 (LOC013)';
COMMENT ON COLUMN t_adloc_stat.adlohhnbr IS 'Estimated number of households, formula: LOC013/LOC011 (LOC014)';
COMMENT ON COLUMN t_adloc_stat.pop_served_water IS 'Population served with safe water, formula: [MIN(LOC013, wp_func_equiv(nb_EWP) x watsan.cfg.mnp_per_EWP)] (LOC017)';
COMMENT ON COLUMN t_adloc_stat.water_access IS 'Access rate to safe water, formula: [pop_served_water(LOC017) / t_adloc.adlopopupe(LOC013)] (LOC018)';
COMMENT ON COLUMN t_adloc_stat.water_functionality_rate IS 'Functionaliy rate of the water points, formula: [ wp_func_equiv(LOC024) / ( wp_func_equiv(LOC024) + wp_nonfunc_equiv(LOC025)) ] (LOC019)';
COMMENT ON COLUMN t_adloc_stat.water_inventory_rate IS 'Inventory rate of the public water points, formula: [wp_inventoried(LOC023) / water_points(LOC022)] (LOC020)';
COMMENT ON COLUMN t_adloc_stat.national_rank IS 'National rank of the locality for access to water, formula: Determine the quintiles of the LOC018 series of values, i.e. the 5 values of access rate AR1, AR2, AR3, AR4, AR5 for which there is an equivalent number of localities in each interval (LOC021)';
COMMENT ON COLUMN t_adloc_stat.water_points IS 'Number of existing public water points, formula: [SELECT SUM (LWP005) WHERE locality = "xxxx" AND LWP002 IN ("01", "02", "03", "04", "05", "09","13")] (LOC022)';
COMMENT ON COLUMN t_adloc_stat.wp_inventoried IS 'Number of inventoried public water points, formula:[SELECT COUNT(*) FROM WP WHERE locality = "xxxx" AND WP008 IN ("01", "03", "04", "05", "06")] (LOC023)';
COMMENT ON COLUMN t_adloc_stat.wp_func_equiv IS 'Number of functional (equivalent) water points, formula: [SELECT SUM(A.upwpnok * B.ewp) FROM t_upwpn A INNER JOIN lex_wp007 B ON A.upwpntype = B.code AND wpnadlocode=:locality] (LOC024)';
COMMENT ON COLUMN t_adloc_stat.wp_nonfunc_equiv IS 'Number of non-functional (equivalent) water points, formula: [SELECT SUM(A.upwpnoknot * B.ewp) FROM t_upwpn A INNER JOIN lex_wp007 B ON A.upwpntype = B.code AND wpnadlocode=:locality] (LOC025)';
COMMENT ON COLUMN t_adloc_stat.pop_served_sanitation IS 'Population served with sanitation, formula: [INT(sf_access(LOC027) * adlopopupe(LOC013))] (LOC026)';
COMMENT ON COLUMN t_adloc_stat.sf_coverage_rate IS 'Coverage rate of needs for public toilets, formula:[sf_existing_required(LOC032) / t_adloc.adlosfreq(LOC031)] (LOC028)';
COMMENT ON COLUMN t_adloc_stat.sf_func_rate IS 'Functionaliy rate of the public toilets, formula:[sf_functional(LOC034) / ( sf_functional(LOC034) + sf_nonfunctional(LOC035) )] (LOC029)';
COMMENT ON COLUMN t_adloc_stat.sf_inventory_rate IS 'Inventory rate of the public toilets, formula:[sf_existing_inventoried(LOC033) / sf_existing_required(LOC032)] (LOC030)';
COMMENT ON COLUMN t_adloc_stat.sf_existing_required IS 'Number of required existing toilets, formula: [sf_functional(LOC034) + sf_nonfunctional(LOC035)] (LOC032)';
COMMENT ON COLUMN t_adloc_stat.sf_existing_inventoried IS 'Number of inventoried existing toilets, formula:[nb d`enregistrements de la table "OuvAss" pour cette localité] (LOC033)';
COMMENT ON COLUMN t_adloc_stat.sf_functional IS 'Number of functional public toilets, formula: (LOC034)';
COMMENT ON COLUMN t_adloc_stat.sf_nonfunctional IS 'Number of non-functionnal public toilets (LOC035)';


--
-- Table: t_adloc_comments
--

DROP TABLE IF EXISTS t_adloc_comments CASCADE;

CREATE TABLE t_adloc_comments
(
	id serial NOT NULL,
	code character varying(20),
	"user" character varying(20),
	user_name character varying(100),
	user_email character varying(100),
	submit_date date DEFAULT ('now'::text)::date,
	rating integer,
	"comment" text,
	ip_address character varying(16),
	is_public boolean,
	is_removed boolean,
	CONSTRAINT pk_loc_comments PRIMARY KEY (id),
	CONSTRAINT fk_t_adloc_comments_code_t_adloc FOREIGN KEY (code) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_adloc_comments_fk_code ON t_adloc_comments USING btree (code);

COMMENT ON TABLE t_adloc_comments IS 'Comments for localities';
COMMENT ON COLUMN t_adloc_comments.id IS 'Identifier for the comment';
COMMENT ON COLUMN t_adloc_comments.code IS 'Identifier for the commented locality';
COMMENT ON COLUMN t_adloc_comments."user" IS 'Identifier for the user who posted the comment';
COMMENT ON COLUMN t_adloc_comments.user_name IS 'User name';
COMMENT ON COLUMN t_adloc_comments.user_email IS 'User email address';
COMMENT ON COLUMN t_adloc_comments.submit_date IS 'Date of the post';
COMMENT ON COLUMN t_adloc_comments.rating IS 'Rating given by user';
COMMENT ON COLUMN t_adloc_comments."comment" IS 'The comment';
COMMENT ON COLUMN t_adloc_comments.ip_address IS 'IP address of the commenter';
COMMENT ON COLUMN t_adloc_comments.is_public IS '1 if comment is public, 0 for restricted';
COMMENT ON COLUMN t_adloc_comments.is_removed IS '1 if comment is removed, 0 for available';



---
--- Table: t_adorga (DRY)
---
DROP TABLE IF EXISTS t_adorga CASCADE;
CREATE TABLE t_adorga
(
	adorcode serial NOT NULL,
	adorname character varying NOT NULL,
	adoracronym character varying,
	adortype character varying(2),
	adoradress character varying(255),
	adoradress1 character varying(255),
	adoradress2 character varying(255),
	adorzipcode character varying(10),
	adorcity character varying(64),
	adorcountry character varying(64),
	adortel character varying(50),
	adorfax character varying(50),
	adoremail character varying(50),
	adorwebsite character varying,
	adorlogo character varying(50),
	adorpres text,
	adorhqcode character varying(20),
	adorhqx double precision,
	adorhqy double precision,
	adorupwho character varying,
	adorupdate date DEFAULT ('now'::text)::date,

	CONSTRAINT pk_t_adorga PRIMARY KEY (adorcode), 
	CONSTRAINT fk_t_adorga_adorhqcode_t_adloc FOREIGN KEY (adorhqcode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adorga_adortype_lex_dry004 FOREIGN KEY (adortype) REFERENCES lex_dry004 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

SELECT AddGeometryColumn('public', 't_adorga', 'adorgacoordinate', 4326, 'POINT', 2);

CREATE INDEX idx_t_adorga_adorhqcode ON t_adorga USING btree (adorhqcode);
CREATE INDEX idx_t_adorga_adorname ON t_adorga USING btree (adorname);
CREATE INDEX idx_t_adorga_adortype ON t_adorga USING btree (adortype);
CREATE INDEX idx_t_adorga_adorcountry ON t_adorga USING btree (adorcountry);
CREATE INDEX idx_t_adorga_gis_adorgacoordinate ON t_adorga USING gist (adorgacoordinate);


COMMENT ON TABLE t_adorga IS 'This table contains the directory data (DRY)';
COMMENT ON COLUMN t_adorga.adorcode IS 'Organization unique ID. Primary Key autoincremented number (DRY001)';
COMMENT ON COLUMN t_adorga.adorname IS 'Name or corporate name (DRY002)';
COMMENT ON COLUMN t_adorga.adoracronym IS 'Short name or acronym (DRY003)';
COMMENT ON COLUMN t_adorga.adortype IS 'Category of organization (DRY004)';
COMMENT ON COLUMN t_adorga.adoradress IS 'Organisation address (DRY005)';
COMMENT ON COLUMN t_adorga.adorzipcode IS 'Zip code (DRY006)';
COMMENT ON COLUMN t_adorga.adorcity IS 'Town (DRY007)';
COMMENT ON COLUMN t_adorga.adorcountry IS 'Country (DRY008)';
COMMENT ON COLUMN t_adorga.adortel IS 'Telephone (DRY009)';
COMMENT ON COLUMN t_adorga.adorfax IS 'Fax (DRY010)';
COMMENT ON COLUMN t_adorga.adorlogo IS 'Logo picture name (DRY012)';
COMMENT ON COLUMN t_adorga.adoremail IS 'Email (DRY011)';
COMMENT ON COLUMN t_adorga.adorwebsite IS 'Website (DRY013)';
COMMENT ON COLUMN t_adorga.adorpres IS 'Presentation (DRY014)';
COMMENT ON COLUMN t_adorga.adorhqcode IS 'Locality code of the headquarter (DRY015)';
COMMENT ON COLUMN t_adorga.adorhqx IS 'GPS longitude of the headquarter (DRY016)';
COMMENT ON COLUMN t_adorga.adorhqy IS 'GPS latitude of the headquarter (DRY017)';
COMMENT ON COLUMN t_adorga.adorupdate IS 'Date of last record update';
COMMENT ON COLUMN t_adorga.adorupwho IS 'Name of the person responsible for last updating data on this organisation';



-----------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------- WATER MONITORING AND HYDROLOGIC ENTITIES ----------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------



---
--- Table: t_hybasin - This table keeps track of the hydrographic entities of the country (BAS)
---
DROP TABLE IF EXISTS t_hyentity CASCADE;
CREATE TABLE t_hyentity
(
	hyecode serial NOT NULL,
	hyeupdate date DEFAULT ('now'::text)::date,
	hyewho character varying,
	CONSTRAINT pk_t_hyentity PRIMARY KEY (hyecode)
);

CREATE INDEX idx_t_hyentity_hyecode ON t_hyentity USING btree(hyecode);

COMMENT ON TABLE t_hyentity IS 'Hydrographic entities have their ID recored here. Specific data is recored in each entity table (t_hyaquifer, t_hybasin, t_hyriver, t_hymostation)';
COMMENT ON COLUMN t_hyentity.hyecode IS 'Hydrographic entity code';
COMMENT ON COLUMN t_hyentity.hyeupdate IS 'Date of last update';
COMMENT ON COLUMN t_hyentity.hyewho IS 'Record author';



---
--- Table: t_hybasin - This table describes the hydrographic basins of the country (BAS)
---
DROP TABLE IF EXISTS t_hybasin CASCADE;
CREATE TABLE t_hybasin
(
	hybcode integer NOT NULL,
	hybname character varying(80),
	hybdesc text,
	hybsurf double precision,
	CONSTRAINT pk_t_hybasin PRIMARY KEY (hybcode),
	CONSTRAINT fk_t_hybasin_hybcode_t_hyentity FOREIGN KEY (hybcode) REFERENCES t_hyentity (hyecode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

COMMENT ON TABLE t_hybasin IS 'This table describes the hydrographic basins of the country (BAS)';
COMMENT ON COLUMN t_hybasin.hybcode IS 'Basin identification code (BAS001)';
COMMENT ON COLUMN t_hybasin.hybname IS 'Name (BAS002)';
COMMENT ON COLUMN t_hybasin.hybdesc IS 'Description (BAS003)';
COMMENT ON COLUMN t_hybasin.hybsurf IS 'Surface (km2)';



---
--- Table: t_hyriver - This table describes the rivers of the country. They are grouped by hydrographic basin (RIV)
---
DROP TABLE IF EXISTS t_hyriver CASCADE;
CREATE TABLE t_hyriver
(
	rivcode integer NOT NULL,
	hybascode integer,
	rivdesc text,
	CONSTRAINT pk_t_hyriver PRIMARY KEY (rivcode),
	CONSTRAINT fk_t_hyriver_rivcode_t_hyentity FOREIGN KEY (rivcode) REFERENCES t_hyentity (hyecode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

COMMENT ON TABLE t_hyriver IS 'This table describes the rivers of the country. They are grouped by hydrographic basin (RIV)';
COMMENT ON COLUMN t_hyriver.rivcode IS 'River identification code (RIV001)';
COMMENT ON COLUMN t_hyriver.hybascode IS 'Basin identification code (RIV002 -> BAS001)';
COMMENT ON COLUMN t_hyriver.rivdesc IS 'Description (RIV003)';



---
--- Table: t_hyaquifer - This table decribes the underground aquifers of the country (AQF)
---
DROP TABLE IF EXISTS t_hyaquifer CASCADE;
CREATE TABLE t_hyaquifer
(
	aqfcode serial NOT NULL,
	aqfname character varying,
	CONSTRAINT pk_t_hyaquifer PRIMARY KEY (aqfcode),
	CONSTRAINT fk_t_hyaquifer_aqfcode_t_hyentity FOREIGN KEY (aqfcode) REFERENCES t_hyentity (hyecode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

COMMENT ON TABLE t_hyaquifer IS 'This table decribes the underground aquifers of the country (AQF)';
COMMENT ON COLUMN t_hyaquifer.aqfcode IS 'Aquifer identification code (AQF001)';
COMMENT ON COLUMN t_hyaquifer.aqfname IS 'Basin identification code (AQF002)';



--
-- Table: t_hymostation
--
DROP TABLE IF EXISTS t_hymostation CASCADE;
CREATE TABLE t_hymostation
(
	moscode serial NOT NULL,
	moscodex character varying(50),
	mosnamee character varying,
	smoncode character varying(16),
	mslocation character varying,
	mosx double precision,
	mosy double precision,
	mosz real,
	mostypee character varying,
	mosyearsta integer,
	mosyearmdif integer,
	mosyearend integer,
	mossit character varying,
	mosknext double precision,
	moscatha real,
	moscomm character varying,
	manres character varying,
	CONSTRAINT pk_t_hymostation PRIMARY KEY (moscode),
	CONSTRAINT fk_t_hymostation_moscode_t_hyentity FOREIGN KEY (moscode) REFERENCES t_hyentity (hyecode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

SELECT AddGeometryColumn('public', 't_hymostation', 'moscoordinate', 4326, 'POINT', 2);

CREATE INDEX idx_t_hymostation_moscodex ON t_hymostation USING btree(moscodex);
CREATE INDEX idx_t_hymostation_mostypee ON t_hymostation USING btree(mostypee);

COMMENT ON TABLE t_hymostation IS 'Monitoring measurement stations (MST)';
COMMENT ON COLUMN t_hymostation.moscode IS 'Primary key';
COMMENT ON COLUMN t_hymostation.moscodex IS 'Monitoring station code';
COMMENT ON COLUMN t_hymostation.mosnamee IS 'Name of the station or of the nearest town';
COMMENT ON COLUMN t_hymostation.smoncode IS 'FK link with MONcode/ Network to which belongs the station';
COMMENT ON COLUMN t_hymostation.mslocation IS 'Location of the station in lake area';
COMMENT ON COLUMN t_hymostation.mosx IS 'MOS X (in decimal degree)';
COMMENT ON COLUMN t_hymostation.mosy IS 'MOS Y (in decimal degree)';
COMMENT ON COLUMN t_hymostation.mosz IS 'Altitude (m)';
COMMENT ON COLUMN t_hymostation.mostypee IS 'Type of station';
COMMENT ON COLUMN t_hymostation.mosyearsta IS 'Year of starting measurement';
COMMENT ON COLUMN t_hymostation.mosyearmdif IS 'Year of modified station';
COMMENT ON COLUMN t_hymostation.mosyearend IS 'Year of closing mostation measurement';
COMMENT ON COLUMN t_hymostation.mossit IS 'Station situation';
COMMENT ON COLUMN t_hymostation.mosknext IS 'Distance to the nearest station (positive value - upstream / negative value - downstream)';
COMMENT ON COLUMN t_hymostation.moscatha IS 'Catchment area of the station (km²)';
COMMENT ON COLUMN t_hymostation.moscomm IS 'Comments';
COMMENT ON COLUMN t_hymostation.manres IS 'Name of the person who is RESPONSIBLE on this station';



-----------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------- ONGOING PROGRAMS SECTION ------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------



---
--- Table: t_adope (OP)
---
DROP TABLE IF EXISTS t_adope CASCADE;
CREATE TABLE t_adope
(
	adopcode serial NOT NULL,
	eaadorcode integer,
	iaadorcode integer,
	adopname character varying(80) NOT NULL,
	adopacron character varying(20),
	adopdstart date,
	adopdend date,
	adopctpre boolean,
	adopctfea boolean,
	adopctcb boolean,
	adopctiec boolean,
	adopctidev boolean,
	adopctass boolean,
	adopctme boolean,
	adopctother text,
	adopctobj text,
	adopresexp text,
	adopactiv text,
	adopbudtot bigint,
	adopcontact character varying(80),
	adopurl character varying,
	adoptgpop integer,
	adoptglocn integer,
	adoptgwpn integer,
	adoptgrpw integer,
	adoptgwcn integer,
	adoptgsfpn integer,
	adoptgsfin integer,
	adopdate date NOT NULL DEFAULT ('now'::text)::date,
	adopwho character varying,
	CONSTRAINT pk_t_adope PRIMARY KEY (adopcode),
	CONSTRAINT fk_t_adope_eaadorcode_t_adorga FOREIGN KEY (eaadorcode) REFERENCES t_adorga (adorcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adope_iaadorcodet_adorga FOREIGN KEY (iaadorcode) REFERENCES t_adorga (adorcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT unq_t_adope_adopcode UNIQUE (adopcode)
);

CREATE INDEX idx_t_adope_adopcode ON t_adope USING btree (adopcode);
CREATE INDEX idx_t_adope_eaadorcode ON t_adope USING btree (eaadorcode);
CREATE INDEX idx_t_adope_iaadorcode ON t_adope USING btree (iaadorcode);
CREATE INDEX idx_t_adope_adopname ON t_adope USING btree (adopname);
CREATE INDEX idx_t_adope_adopdstart ON t_adope USING btree (adopdstart);
CREATE INDEX idx_t_adope_adopdend ON t_adope USING btree (adopdend);

COMMENT ON TABLE t_adope IS 'This table contains the descriptive data of ongoing programs. The term "Program" must be considered in a very large sense and refers to any activity whatever its scope and size (OP)';
COMMENT ON COLUMN t_adope.adopcode IS 'Program code (OPE001)';
COMMENT ON COLUMN t_adope.adopname IS 'Program name (OPE002)';
COMMENT ON COLUMN t_adope.adopacron IS 'Short name or acronym (OPE003)';
COMMENT ON COLUMN t_adope.adopdstart IS 'Starting date (OPE004)';
COMMENT ON COLUMN t_adope.adopdend IS 'Ending date (OPE005)';
COMMENT ON COLUMN t_adope.iaadorcode IS 'Implementing organization ID (OPE006)';
COMMENT ON COLUMN t_adope.eaadorcode IS 'Executing organization ID (OPE007)';
COMMENT ON COLUMN t_adope.adopctpre IS 'Content: Preparation study (OPE008)';
COMMENT ON COLUMN t_adope.adopctfea IS 'Content: Feasibility study (OPE009)';
COMMENT ON COLUMN t_adope.adopctcb IS 'Content: Capacity building (OPE010)';
COMMENT ON COLUMN t_adope.adopctiec IS 'Content: IEC (OPE011)';
COMMENT ON COLUMN t_adope.adopctidev IS 'Content: Infrastructure development (OPE012)';
COMMENT ON COLUMN t_adope.adopctass IS 'Content: Assessment study (OPE013)';
COMMENT ON COLUMN t_adope.adopctme IS 'Content: Audit/M&E (OPE014)';
COMMENT ON COLUMN t_adope.adopctother IS 'Content: other (OPE015)';
COMMENT ON COLUMN t_adope.adopctobj IS 'Objectives (OPE016)';
COMMENT ON COLUMN t_adope.adopresexp IS 'Expected results (OPE017)';
COMMENT ON COLUMN t_adope.adopactiv IS 'Activities/Components (OPE018)';
COMMENT ON COLUMN t_adope.adopbudtot IS 'Total budget (OPE019)';
COMMENT ON COLUMN t_adope.adopcontact IS 'Contact (OPE020)';
COMMENT ON COLUMN t_adope.adopurl IS 'URL to the program website (OPE021)';
COMMENT ON COLUMN t_adope.adoptgpop IS 'Target beneficiary population (OPE022)';
COMMENT ON COLUMN t_adope.adoptglocn IS 'Target number of localities (OPE023)';
COMMENT ON COLUMN t_adope.adoptgwpn IS 'Target number of new public water points (OPE024)';
COMMENT ON COLUMN t_adope.adoptgrpw IS 'Target number of rehabilitated public water points (OPE025)';
COMMENT ON COLUMN t_adope.adoptgwcn IS 'Target number of new water connections (OPE026)';
COMMENT ON COLUMN t_adope.adoptgsfpn IS 'Target number of new public sanitation facilities (OPE027)';
COMMENT ON COLUMN t_adope.adoptgsfin IS 'Target number of new individual sanitation facilities (OPE028)';
COMMENT ON COLUMN t_adope.adopdate IS 'Date when program was added in the database';
COMMENT ON COLUMN t_adope.adopwho IS 'Record author';



---
--- Table: t_adopfund (OP_BUD)
---
DROP TABLE IF EXISTS t_adopfund CASCADE;
CREATE TABLE t_adopfund
(
	opfadopcode integer NOT NULL,
	adopfcode integer NOT NULL,
	adopfamou integer,
	adopfmunit character varying(20),
	adopfcat character varying(2),
	CONSTRAINT pk_t_adopfund PRIMARY KEY (opfadopcode, adopfcode),
	CONSTRAINT fk_t_adopfund_t_adope FOREIGN KEY (opfadopcode) REFERENCES t_adope (adopcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adopfund_adopfcode_t_adorga FOREIGN KEY (adopfcode) REFERENCES t_adorga (adorcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_adopfund_adopfcat_lex_opb005 FOREIGN KEY (adopfcat) REFERENCES lex_opb005 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_adopfund_opfadopcode ON t_adopfund USING btree (opfadopcode);
CREATE INDEX idx_t_adopfund_adopfcode ON t_adopfund USING btree (adopfcode);

COMMENT ON TABLE t_adopfund IS 'Funding arrangements of the programs (OP_BUD)';
COMMENT ON COLUMN t_adopfund.opfadopcode IS 'Program code (OPB001)';
COMMENT ON COLUMN t_adopfund.adopfcode IS 'Funding source. Pointer to directory table (OPB002)';
COMMENT ON COLUMN t_adopfund.adopfamou IS 'Amount of funding (OPB003)';
COMMENT ON COLUMN t_adopfund.adopfmunit IS 'Money unit (OPB004)';
COMMENT ON COLUMN t_adopfund.adopfcat IS 'Category of funding (OPB005)';



---
--- Table: t_adopmo (OP_ADV)
---
DROP TABLE IF EXISTS t_adopmo CASCADE;
CREATE TABLE t_adopmo
(
	opmadopcode integer NOT NULL,
	adopmdate date NOT NULL DEFAULT ('now'::text)::date,
	adopmexpen integer,
	adopmdisbu integer,
	adopmpopws integer,
	adopmpopsf integer,
	adopmlocws integer,
	adopmpwpnbr integer,
	adopmwprnbr integer,
	adopmwcnbr integer,
	adopmsfpnbr integer,
	adopmsfinbr integer,
	CONSTRAINT pk_t_adopmo PRIMARY KEY (opmadopcode, adopmdate),
	CONSTRAINT fk_t_adopmo_t_adope FOREIGN KEY (opmadopcode) REFERENCES t_adope (adopcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_adopmo_opmadopcode ON t_adopmo USING btree (opmadopcode);
CREATE INDEX idx_t_adopmo_adopmdate ON t_adopmo USING btree (adopmdate);

COMMENT ON TABLE t_adopmo IS 'Monitoring data on quantitative progress of the programs (OP_ADV)';
COMMENT ON COLUMN t_adopmo.opmadopcode IS 'Program code (OPA001)';
COMMENT ON COLUMN t_adopmo.adopmdate IS 'Date (OPA002)';
COMMENT ON COLUMN t_adopmo.adopmexpen IS 'Cumulated expense commitments (OPA003)';
COMMENT ON COLUMN t_adopmo.adopmdisbu IS 'Cumulated disbursements (OPA004)';
COMMENT ON COLUMN t_adopmo.adopmpopws IS 'Cumulated population served with water (OPA005)';
COMMENT ON COLUMN t_adopmo.adopmpopsf IS 'Cumulated population served with sanitation (OPA006)';
COMMENT ON COLUMN t_adopmo.adopmlocws IS 'Cumulated number of localities served with water (OPA007)';
COMMENT ON COLUMN t_adopmo.adopmpwpnbr IS 'Cumulated number of new public water points (OPA008)';
COMMENT ON COLUMN t_adopmo.adopmwprnbr IS 'Cumulated number of rehabilitated water points (OPA009)';
COMMENT ON COLUMN t_adopmo.adopmwcnbr IS 'Cumulated number of new water connections (OPA010)';
COMMENT ON COLUMN t_adopmo.adopmsfpnbr IS 'Cumulated number of public sanitation facilities (OPA011)';
COMMENT ON COLUMN t_adopmo.adopmsfinbr IS 'Cumulated number of individual sanitation facilities (OPA012)';



--
-- Table: t_upis
--
DROP TABLE IF EXISTS t_upis CASCADE;
CREATE TABLE t_upis
(
	upiscode character varying(20) NOT NULL,
	upiscodex character varying(20),
	upiscodex2 character varying(20),
	upiscodex3 character varying(20),
	ishyecode integer,
	upisadlocode character varying(20),
	upisadr1code character varying(20),
	upisadr2code character varying(20),
	upisadr3code character varying(20),
	upisadr4code character varying(20),
	upisadr5code character varying(20),
	upistype character varying(4),
	upisname character varying,
	upisname2 character varying,
	upisplace character varying,
	upisloctype character varying(2),
	upisx double precision,
	upisy double precision,
	upisz double precision,
	upisyear integer,
	upisobj character varying(2),
	upisdend date,
	upistdesc character varying,
	upiscomment character varying,
	upismanprof character varying(2),
	upisowner character varying(50),
	upisprogram integer,
	upisfinance character varying(2),
	upisupdate date DEFAULT ('now'::text)::date,
	upisupwho character varying,
	approved boolean NOT NULL DEFAULT TRUE,
	contributed boolean NOT NULL DEFAULT FALSE,
	CONSTRAINT pk_t_upis PRIMARY KEY (upiscode),
	CONSTRAINT fk_t_upis_upistype_lex_upistype FOREIGN KEY (upistype) REFERENCES lex_upistype (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisadlocode_t_adloc FOREIGN KEY (upisadlocode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisadr1code_t_adreglev FOREIGN KEY (upisadr1code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisadr2code_t_adreglev FOREIGN KEY (upisadr2code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisadr3code_t_adreglev FOREIGN KEY (upisadr3code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisadr4code_t_adreglev FOREIGN KEY (upisadr4code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisadr5code_t_adreglev FOREIGN KEY (upisadr5code) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisloctype_lex_upisloctype FOREIGN KEY (upisloctype) REFERENCES lex_upisloctype (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upismanprof_lex_upismanprof FOREIGN KEY (upismanprof) REFERENCES lex_upismanprof (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisobj_lex_wp051 FOREIGN KEY (upisobj) REFERENCES lex_wp051 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisfinance_lex_upisfinance FOREIGN KEY (upisfinance) REFERENCES lex_upisfinance (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upisprogram_t_adope FOREIGN KEY (upisprogram) REFERENCES t_adope (adopcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_ishyecode_t_hyentity FOREIGN KEY (ishyecode) REFERENCES t_hyentity (hyecode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

SELECT AddGeometryColumn('public', 't_upis', 'upiscoordinate', 4326, 'POINT', 2);

CREATE UNIQUE INDEX idx_t_upis_pk ON t_upis USING btree(upiscode);
ALTER TABLE t_upis CLUSTER ON idx_t_upis_pk;
CREATE INDEX idx_t_upis_upisname ON t_upis USING btree(upisname);
CREATE INDEX idx_t_upis_upisadlocode ON t_upis USING btree(upisadlocode);
CREATE INDEX idx_t_upis_upisadr1code ON t_upis USING btree(upisadr1code);
CREATE INDEX idx_t_upis_upisadr2code ON t_upis USING btree(upisadr2code);
CREATE INDEX idx_t_upis_upisadr3code ON t_upis USING btree(upisadr3code);
CREATE INDEX idx_t_upis_upisadr4code ON t_upis USING btree(upisadr4code);
CREATE INDEX idx_t_upis_upisadr5code ON t_upis USING btree(upisadr5code);
CREATE INDEX idx_t_upis_upistype ON t_upis USING btree(upistype);
CREATE INDEX idx_t_upis_upiscoordinate ON t_upis USING gist (upiscoordinate);
CREATE INDEX idx_t_upis_upisloctype ON t_upis USING btree(upisloctype);
CREATE INDEX idx_t_upis_upisobj ON t_upis USING btree(upisobj);
CREATE INDEX idx_t_upis_approved ON t_upis USING btree (approved);
CREATE INDEX idx_t_upis_contributed ON t_upis USING btree(contributed);


COMMENT ON TABLE t_upis IS 'Infrastructures. Contains water points (WP), sanitation facilities (SF), piped water schemes (PWS), underground water sources (UWS). This tables manages common fields of these infrastructures';
COMMENT ON COLUMN t_upis.upiscode IS 'National identification code of the infrastructure (WP001/SF001/PWS001/UWS001)';
COMMENT ON COLUMN t_upis.upiscodex IS 'Communal identification code of the infrastructure (WP002/SF002)';

COMMENT ON COLUMN t_upis.upiscodex2 IS 'Additional codes';
COMMENT ON COLUMN t_upis.upiscodex3 IS 'Additional codes';
COMMENT ON COLUMN t_upis.ishyecode IS 'Hydrological entity code';
COMMENT ON COLUMN t_upis.upisadlocode IS 'Locality where the infrastructure is located (WP004/SF003/PWS004/UWS004)';
COMMENT ON COLUMN t_upis.upisadr1code IS 'Level 1 code for the subdivision where this point is located (PWS003/UWS003/WP004)';
COMMENT ON COLUMN t_upis.upisadr2code IS 'Level 2 code for the subdivision where this point is located (PWS003/UWS003/WP004)';
COMMENT ON COLUMN t_upis.upisadr3code IS 'Level 3 code for the subdivision where this point is located (PWS003/UWS003/WP004)';
COMMENT ON COLUMN t_upis.upisadr4code IS 'Level 4 code for the subdivision where this point is located (PWS003/UWS003/WP004)';
COMMENT ON COLUMN t_upis.upisadr5code IS 'Level 5 code for the subdivision where this point is located (PWS003/UWS003/WP004)';
COMMENT ON COLUMN t_upis.upistype IS 'Type of Infrastructure (WP, SF, PWS, UWS)';
COMMENT ON COLUMN t_upis.upisname IS 'Name of the infrastructure (WP006/SF005/PWS002/UWS002)';
COMMENT ON COLUMN t_upis.upisname2 IS 'Name of the infrastructure in original language';
COMMENT ON COLUMN t_upis.upisplace IS 'District / hamlet of location of the water point (WP005/SF004)';
COMMENT ON COLUMN t_upis.upisloctype IS 'Type of location (WP008/SF007 - lex_upisloctype)';
COMMENT ON COLUMN t_upis.upisx IS 'Coordinate X in decimal degree Ex: 12.8 (WP015/SF022/PWS098/UWS013)';
COMMENT ON COLUMN t_upis.upisy IS 'Coordinate Y in decimal degree Ex: 24.8 (WP016/SF023/PWS098/UWS014)';
COMMENT ON COLUMN t_upis.upisz IS 'Altitude (Z)';
COMMENT ON COLUMN t_upis.upisyear IS 'Year of completion of the facility (WP014/SF011/PWS010/UWS006)';
COMMENT ON COLUMN t_upis.upisobj IS 'Functional status of the infrastructure. Example: In service/Out of order/Abandonned/Not used/Destroyed (WP051)';
COMMENT ON COLUMN t_upis.upisdend IS 'Date end of operating infrastructure';
COMMENT ON COLUMN t_upis.upistdesc IS 'Technical detailed description of the infrastructure';
COMMENT ON COLUMN t_upis.upiscomment IS 'Comment on the infrastructure Example: Groundwater, Surface water (from river or lake), Spring water...';
COMMENT ON COLUMN t_upis.upismanprof IS 'Water point manager profile / Type of operator of the sanitation facility (WP010/SF008/PWS006)';
COMMENT ON COLUMN t_upis.upisowner IS 'Water point owner (WP011/)';
COMMENT ON COLUMN t_upis.upisprogram IS 'Implementing program (WP012/SF009)';
COMMENT ON COLUMN t_upis.upisfinance IS 'Source of financing (WP013/SF010)';
COMMENT ON COLUMN t_upis.upisupdate IS 'Date of last record update';
COMMENT ON COLUMN t_upis.upisupwho IS 'Username of the responsible for last updating data on this infrastructure';
COMMENT ON COLUMN t_upis.upiscoordinate IS 'Geography column with locality geographical coordinates';
COMMENT ON COLUMN t_upis.approved IS 'Specifies that this data is approved and used within application. Non-approved data does not appear on queries and appear as pending approval';
COMMENT ON COLUMN t_upis.contributed IS 'Specifies that this data is user-contributed and does not come from original data set';



--
-- Table: t_upsf
--
DROP TABLE IF EXISTS t_upsf CASCADE;
CREATE TABLE t_upsf
(
	upiscode character varying(20) NOT NULL,
	upsftype character varying(2),
	upsfmatype character varying(2),
	upsfgrtype character varying(2),
	upsfmwsep boolean,
	upsfblocnbr integer,
	upsfblomnbr integer,
	upsfblownbr integer,
	upsfshower boolean,
	upsfwhand boolean,
	upwswbas boolean,
	upsfwater character varying(2),
	upsffunc character varying(2),
	upsfnotfr character varying(2),
	upsfnotfd character varying(2),
	upsfasscw character varying(2),
	upsfasspl character varying(2),
	upsfasscl character varying(2),
	upsfpaymd character varying(2),
	CONSTRAINT pk_t_upsf PRIMARY KEY (upiscode),
	CONSTRAINT fk_t_upsf_upiscode_t_upis FOREIGN KEY (upiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT fk_t_upsf_upsfwater_lex_sf021 FOREIGN KEY (upsfwater) REFERENCES lex_sf021 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsftype_lex_sf006 FOREIGN KEY (upsftype) REFERENCES lex_sf006 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfmatype_lex_sf012 FOREIGN KEY (upsfmatype) REFERENCES lex_sf012 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfgrtype_lex_sf013 FOREIGN KEY (upsfgrtype) REFERENCES lex_sf013 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsffunc_lex_sf050 FOREIGN KEY (upsffunc) REFERENCES lex_sf050 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfnotfr_lex_sf051 FOREIGN KEY (upsfnotfr) REFERENCES lex_sf051 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfnotfd_lex_sf052 FOREIGN KEY (upsfnotfd) REFERENCES lex_sf052 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfasscw_lex_sf053 FOREIGN KEY (upsfasscw) REFERENCES lex_sf053 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfasspl_lex_sf054 FOREIGN KEY (upsfasspl) REFERENCES lex_sf054 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfasscl_lex_sf055 FOREIGN KEY (upsfasscl) REFERENCES lex_sf055 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upsf_upsfpaymd_lex_sf056 FOREIGN KEY (upsfpaymd) REFERENCES lex_sf056 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE UNIQUE INDEX t_upsf_pk ON t_upsf USING btree(upiscode);
CREATE INDEX idx_t_upsf_upsftype ON t_upsf USING btree(upsftype);
CREATE INDEX idx_t_upsf_upsfmatype ON t_upsf USING btree(upsfmatype);
CREATE INDEX idx_t_upsf_upsfgrtype ON t_upsf USING btree(upsfgrtype);
CREATE INDEX idx_t_upsf_upsfwater ON t_upsf USING btree(upsfwater);
CREATE INDEX idx_t_upsf_upsffunc ON t_upsf USING btree(upsffunc);
CREATE INDEX idx_t_upsf_upsfnotfr ON t_upsf USING btree(upsfnotfr);
CREATE INDEX idx_t_upsf_upsfnotfd ON t_upsf USING btree(upsfnotfd);
CREATE INDEX idx_t_upsf_upsfasscw ON t_upsf USING btree(upsfasscw);
CREATE INDEX idx_t_upsf_upsfasspl ON t_upsf USING btree(upsfasspl);
CREATE INDEX idx_t_upsf_upsfasscl ON t_upsf USING btree(upsfasscl);
CREATE INDEX idx_t_upsf_upsfpaymd ON t_upsf USING btree(upsfpaymd);

COMMENT ON TABLE t_upsf IS 'Specific characterisitics of of sanitation facilities ';
COMMENT ON COLUMN t_upsf.upiscode IS 'National identification code of the sanitation facility (SF001)';
COMMENT ON COLUMN t_upsf.upsftype IS 'Type of sanitation facility (SF006)';
COMMENT ON COLUMN t_upsf.upsfmatype IS 'Type of building materials (SF012)';
COMMENT ON COLUMN t_upsf.upsfgrtype IS 'Type of ground (SF013)';
COMMENT ON COLUMN t_upsf.upsfmwsep IS 'Separation men/women (SF014)';
COMMENT ON COLUMN t_upsf.upsfblocnbr IS 'No. of blocks (SF015)';
COMMENT ON COLUMN t_upsf.upsfblomnbr IS 'No. of blocks for men (SF016)';
COMMENT ON COLUMN t_upsf.upsfblownbr IS 'No. of blocks for women (SF017)';
COMMENT ON COLUMN t_upsf.upsfshower IS 'Presence of showers (SF018)';
COMMENT ON COLUMN t_upsf.upsfwhand IS 'Presence of hand washing facility (SF019)';
COMMENT ON COLUMN t_upsf.upwswbas IS 'Presence of washing basin with cesspool (SF020)';
COMMENT ON COLUMN t_upsf.upsfwater IS 'Availability of water at the sanitation facility (SF021)';
COMMENT ON COLUMN t_upsf.upsffunc IS 'Functionality of the sanitation facility (SF050)';
COMMENT ON COLUMN t_upsf.upsfnotfr IS 'Reasons for non functionality (SF051)';
COMMENT ON COLUMN t_upsf.upsfnotfd IS 'Duration of non functionality (SF052)';
COMMENT ON COLUMN t_upsf.upsfasscw IS 'Assessment of civil works on the sanitation facility (SF053)';
COMMENT ON COLUMN t_upsf.upsfasspl IS 'Assessment of plumbering on the sanitation facility (SF054)';
COMMENT ON COLUMN t_upsf.upsfasscl IS 'Assessment of cleanliness on the sanitation facility (SF055)';
COMMENT ON COLUMN t_upsf.upsfpaymd IS 'Modalities for the payment on the sanitation facility (SF056)';



--
-- Table: t_upwp
--
DROP TABLE IF EXISTS t_upwp CASCADE;
CREATE TABLE t_upwp
(
	upiscode character varying(20) NOT NULL,
	upwpcodex character varying(20),
	upwptype character varying(2),
	upwpbench character varying(2),
	upwpmainuse character varying(2),
	upwpantiq boolean,
	upwpdiam double precision,
	upwpdepth double precision,
	upwpcoph double precision,
	upwpcatch double precision,
	upwprain integer,
	upwpnotfr character varying(2),
	upwpnotfd character varying(2),
	upwdlfdate date,
	upwpasscw character varying(2),
	upwpasspl character varying(2),
	upwpasshy character varying(2),
	upwpfence character varying(2),
	upwpwturb character varying(2),
	upwpwtast character varying(2),
	upwpwodor character varying(2),
	upwptreat character varying(2),
	upwptreatf character varying(2),
	upwppaymd character varying(2),
	upwppayseas character varying(2),
	upwphptype character varying(2),
	upwphpfunc character varying(2),
	upwphpyear integer,
	upwphpiwho integer,
	upwphpcraf integer,
	upwpmeter character varying(2),
	upwpyield double precision,
	upwpprero character varying(2),
	upwpprpol character varying(2),
	upwpdry character varying(2),
	upwpclos character varying(2),
	upwpabsq character varying(2),
	upwpwlev double precision,
	upwpeprice double precision,
	upwpesaleu character varying(10),
	upwputype character varying(2),
	upwpuyelm double precision,
	upwpudepin smallint,
	fk_aqfcode integer,
	upwpudraw character varying,
	CONSTRAINT pk_t_upwp PRIMARY KEY (upiscode),
	CONSTRAINT fk_t_upwp_upiscode_t_upis FOREIGN KEY (upiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT fk_t_upwp_upwptype_lex_upisloctype FOREIGN KEY (upwptype) REFERENCES lex_upisloctype (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upis_upwpmainuse_lex_wp009 FOREIGN KEY (upwpmainuse) REFERENCES lex_wp009 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpbench_lex_wp017 FOREIGN KEY (upwpbench) REFERENCES lex_wp017 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpnotfr_lex_wp052 FOREIGN KEY (upwpnotfr) REFERENCES lex_wp052 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpnotfd_lex_wp053 FOREIGN KEY (upwpnotfd) REFERENCES lex_wp053 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpasscw_lex_wp055 FOREIGN KEY (upwpasscw) REFERENCES lex_wp055 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpasspl_lex_wp056 FOREIGN KEY (upwpasspl) REFERENCES lex_wp056 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpasshy_lex_wp057 FOREIGN KEY (upwpasshy) REFERENCES lex_wp057 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpfence_lex_wp058 FOREIGN KEY (upwpfence) REFERENCES lex_wp058 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpwturb_lex_wp059 FOREIGN KEY (upwpwturb) REFERENCES lex_wp059 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpwtast_lex_wp060 FOREIGN KEY (upwpwtast) REFERENCES lex_wp060 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpwodor_lex_wp061 FOREIGN KEY (upwpwodor) REFERENCES lex_wp061 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwptreat_lex_wp062 FOREIGN KEY (upwptreat) REFERENCES lex_wp062 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwptreatf_lex_wp063 FOREIGN KEY (upwptreatf) REFERENCES lex_wp063 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwppaymd_lex_wp064 FOREIGN KEY (upwppaymd) REFERENCES lex_wp064 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwppayseas_lex_wp065 FOREIGN KEY (upwppayseas) REFERENCES lex_wp065 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwphptype_lex_wp066 FOREIGN KEY (upwphptype) REFERENCES lex_wp066 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwphpfunc_lex_wp067 FOREIGN KEY (upwphpfunc) REFERENCES lex_wp067 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwphpiwho_t_adorga FOREIGN KEY (upwphpiwho) REFERENCES t_adorga (adorcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwphpcraf_t_adorga FOREIGN KEY (upwphpcraf) REFERENCES t_adorga (adorcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpmeter_lex_wp071 FOREIGN KEY (upwpmeter) REFERENCES lex_wp071 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpprpol_lex_wp074 FOREIGN KEY (upwpprpol) REFERENCES lex_wp074 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpdry_lex_wp075 FOREIGN KEY (upwpdry) REFERENCES lex_wp075 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwpclos_lex_wp076 FOREIGN KEY (upwpclos) REFERENCES lex_wp076 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_upwputype_lex_uws005 FOREIGN KEY (upwputype) REFERENCES lex_uws005 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwp_fk_aqfcode_t_hyaquifer FOREIGN KEY (fk_aqfcode) REFERENCES t_hyaquifer (aqfcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE UNIQUE INDEX t_upwp_pk ON t_upwp USING btree (upiscode);
CREATE INDEX idx_t_upwp_upwptype ON t_upwp USING btree (upwptype);
CREATE INDEX idx_t_upwp_upwpmainuse ON t_upwp USING btree (upwpmainuse);
CREATE INDEX idx_t_upwp_upwpbench ON t_upwp USING btree (upwpbench);
CREATE INDEX idx_t_upwp_upwpnotfr ON t_upwp USING btree (upwpnotfr);
CREATE INDEX idx_t_upwp_upwpnotfd ON t_upwp USING btree (upwpnotfd);
CREATE INDEX idx_t_upwp_upwpasscw ON t_upwp USING btree (upwpasscw);
CREATE INDEX idx_t_upwp_upwpasspl ON t_upwp USING btree (upwpasspl);
CREATE INDEX idx_t_upwp_upwpasshy ON t_upwp USING btree (upwpasshy);
CREATE INDEX idx_t_upwp_upwpfence ON t_upwp USING btree (upwpfence);
CREATE INDEX idx_t_upwp_upwpwturb ON t_upwp USING btree (upwpwturb);
CREATE INDEX idx_t_upwp_upwpwtast ON t_upwp USING btree (upwpwtast);
CREATE INDEX idx_t_upwp_upwpwodor ON t_upwp USING btree (upwpwodor);
CREATE INDEX idx_t_upwp_upwptreat ON t_upwp USING btree (upwptreat);
CREATE INDEX idx_t_upwp_upwptreatf ON t_upwp USING btree (upwptreatf);
CREATE INDEX idx_t_upwp_upwppaymd ON t_upwp USING btree (upwppaymd);
CREATE INDEX idx_t_upwp_upwppayseas ON t_upwp USING btree (upwppayseas);
CREATE INDEX idx_t_upwp_upwphptype ON t_upwp USING btree (upwphptype);
CREATE INDEX idx_t_upwp_upwphpfunc ON t_upwp USING btree (upwphpfunc);
CREATE INDEX idx_t_upwp_upwphpiwho ON t_upwp USING btree (upwphpiwho);
CREATE INDEX idx_t_upwp_upwphpcraf ON t_upwp USING btree (upwphpcraf);
CREATE INDEX idx_t_upwp_upwpmeter ON t_upwp USING btree (upwpmeter);
CREATE INDEX idx_t_upwp_upwpprpol ON t_upwp USING btree (upwpprpol);
CREATE INDEX idx_t_upwp_upwpdry ON t_upwp USING btree (upwpdry);
CREATE INDEX idx_t_upwp_upwpclos ON t_upwp USING btree (upwpclos);
CREATE INDEX idx_t_upwp_upwputype ON t_upwp USING btree (upwputype);
CREATE INDEX idx_t_upwp_fk_aqfcode ON t_upwp USING btree (fk_aqfcode);

COMMENT ON TABLE t_upwp IS 'Specific characterisitics of infrastructure of WP type';
COMMENT ON COLUMN t_upwp.upiscode IS 'National identification code of the water point (WP001)';
COMMENT ON COLUMN t_upwp.upwpcodex IS 'Hydraulic inventory code of the infrastructure (WP003)';
COMMENT ON COLUMN t_upwp.upwptype IS 'Type of water point (WP007)';
COMMENT ON COLUMN t_upwp.upwpbench IS 'Benchmarking of the water point by the users (WP017)';
COMMENT ON COLUMN t_upwp.upwpmainuse IS 'Principal use of the water point (WP009)';
COMMENT ON COLUMN t_upwp.upwpantiq IS 'Anti-quagmire disposal (WP018)';
COMMENT ON COLUMN t_upwp.upwpdiam IS 'Diameter of the well / water source (mm) (WP019/UWS008)';
COMMENT ON COLUMN t_upwp.upwpdepth IS 'Depth of the well / water source (WP020/UWS009)';
COMMENT ON COLUMN t_upwp.upwpcoph IS 'Height of the coping of the well (WP021)';
COMMENT ON COLUMN t_upwp.upwpcatch IS 'Catchment area (m²) (WP022)';
COMMENT ON COLUMN t_upwp.upwprain IS 'Rainfall (mm/year) (WP023)';
COMMENT ON COLUMN t_upwp.upwpnotfr IS 'Reasons for non functionality (WP052)';
COMMENT ON COLUMN t_upwp.upwpnotfd IS 'Duration of non functionality (WP053)';
COMMENT ON COLUMN t_upwp.upwdlfdate IS 'Date of last failure (WP054)';
COMMENT ON COLUMN t_upwp.upwpasscw IS 'Assessment of civil works on the water point (WP055)';
COMMENT ON COLUMN t_upwp.upwpasspl IS 'Assessment of plumbering on the water point (WP056)';
COMMENT ON COLUMN t_upwp.upwpasshy IS 'Assessment of hygiene around the water point (WP057)';
COMMENT ON COLUMN t_upwp.upwpfence IS 'Fencing of the water point (WP58)';
COMMENT ON COLUMN t_upwp.upwpwturb IS 'Turbidity of water (WP059)';
COMMENT ON COLUMN t_upwp.upwpwtast IS 'Taste of water (WP060)';
COMMENT ON COLUMN t_upwp.upwpwodor IS 'Odor of water (WP061)';
COMMENT ON COLUMN t_upwp.upwptreat IS 'System of water treatment on the water point (WP062)';
COMMENT ON COLUMN t_upwp.upwptreatf IS 'Functionality of the system of water treatment (WP063)';
COMMENT ON COLUMN t_upwp.upwppaymd IS 'Modalities for the payment of water on the water point (WP064)';
COMMENT ON COLUMN t_upwp.upwppayseas IS 'Seasonability of the payment of water on the water point(WP065)';
COMMENT ON COLUMN t_upwp.upwphptype IS 'Brand/Type of the handpump (WP066)';
COMMENT ON COLUMN t_upwp.upwphpfunc IS 'Functionality of the handpump (WP067)';
COMMENT ON COLUMN t_upwp.upwphpyear IS 'Year of installation of the handpump (WP068)';
COMMENT ON COLUMN t_upwp.upwphpiwho IS 'Implementing agency of the handpump (WP069)';
COMMENT ON COLUMN t_upwp.upwphpcraf IS 'Identification code for the repairer craftmant of the handpump (WP070)';
COMMENT ON COLUMN t_upwp.upwpmeter IS 'Functionality of the water metering system (WP071)';
COMMENT ON COLUMN t_upwp.upwpyield IS 'Measured or estimated yield (m3/h) (WP072)';
COMMENT ON COLUMN t_upwp.upwpprero IS 'Protection against erosion (WP073)';
COMMENT ON COLUMN t_upwp.upwpprpol IS 'Protection area against contamination by latrines (WP074)';
COMMENT ON COLUMN t_upwp.upwpdry IS 'Drying up of the water point (WP075)';
COMMENT ON COLUMN t_upwp.upwpclos IS 'Closing system on the well (WP076)';
COMMENT ON COLUMN t_upwp.upwpabsq IS 'Water abstraction system on the well (WP077)';
COMMENT ON COLUMN t_upwp.upwpwlev IS 'Water level in the well (WP078/UWS007)';
COMMENT ON COLUMN t_upwp.upwpeprice IS 'Main water sale price (if water is sold) in country currency (WP079)';
COMMENT ON COLUMN t_upwp.upwpesaleu IS 'Sale unit (WP080)';
COMMENT ON COLUMN t_upwp.upwputype IS 'Type of source (UWS005)';
COMMENT ON COLUMN t_upwp.upwpuyelm IS 'Maximum authorized abstraction yield (m3/h) (UWS010)';
COMMENT ON COLUMN t_upwp.upwpudepin IS 'Recommended depth of installation (m) (UWS011)';
COMMENT ON COLUMN t_upwp.fk_aqfcode IS 'Catched aquifer (UWS012)';
COMMENT ON COLUMN t_upwp.upwpudraw IS 'Borehole drawings. Link to a downlodable pdf file of the borehole drawings (UWS015)';



--
-- Table: t_wqm
--
DROP TABLE IF EXISTS t_wqm CASCADE;
CREATE TABLE t_wqm
(
	wqmcode serial NOT NULL,
	wqmsamplecode character varying(20),
	wqmreference character varying,
	wqmoperator character varying,
	wqmdate date DEFAULT ('now'::text)::date,
	wqmconformity boolean,
	wqmcomment character varying,

	CONSTRAINT pk_t_wqmcode PRIMARY KEY (wqmcode),
	CONSTRAINT fk_t_wqm_wqmsamplecode_t_upis FOREIGN KEY (wqmsamplecode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

COMMENT ON TABLE t_wqm IS 'Contains the reference of the water quality reports related to onsite controls made on underground water sources or water (distribution) points';
COMMENT ON COLUMN t_wqm.wqmcode IS 'Primary key. Report code';
COMMENT ON COLUMN t_wqm.wqmsamplecode IS 'Sample point code (a value of [PWS001] or [WP001] table)';
COMMENT ON COLUMN t_wqm.wqmreference IS 'Reference (source) ot the control (Ex: "water quality campaign, year 2009")';
COMMENT ON COLUMN t_wqm.wqmoperator IS 'Identification of operator';
COMMENT ON COLUMN t_wqm.wqmdate IS 'Date of the sample';
COMMENT ON COLUMN t_wqm.wqmconformity IS 'Water quality conformity (Conclusion of the report about quality conformity)';
COMMENT ON COLUMN t_wqm.wqmcomment IS 'Comment on the results';


--
-- Table: t_wqmdat
--
DROP TABLE IF EXISTS t_wqmdat CASCADE;
CREATE TABLE t_wqmdat
(
	wqmdatcode serial NOT NULL,
	wqmdatreportcode integer,
	wqmdatparameter character varying (2),
	wqmdatvalue real,
	wqmdatconformity boolean,

	CONSTRAINT pk_t_wqmdatcode PRIMARY KEY (wqmdatcode),
	CONSTRAINT fk_t_wqmdat_wqmdatreportcode_t_wqm FOREIGN KEY (wqmdatreportcode) REFERENCES t_wqm (wqmcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_wqmdat_wqmdatparameter_lex_wqd003 FOREIGN KEY (wqmdatparameter) REFERENCES lex_wqd003 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT

);

COMMENT ON TABLE t_wqmdat IS 'Contains the reference of the water quality reports related to onsite controls made on underground water sources or water (distribution) points';
COMMENT ON COLUMN t_wqmdat.wqmdatcode IS 'Primary key. Measurement code';
COMMENT ON COLUMN t_wqmdat.wqmdatreportcode IS 'Report code';
COMMENT ON COLUMN t_wqmdat.wqmdatparameter IS 'Parameter measured and measurement unit';
COMMENT ON COLUMN t_wqmdat.wqmdatvalue IS 'Measured value';
COMMENT ON COLUMN t_wqmdat.wqmdatconformity IS 'Conformity';


--
-- Table: t_upwpn
--
DROP TABLE IF EXISTS t_upwpn CASCADE;
CREATE TABLE t_upwpn
(
	wpnadlocode character varying(20) NOT NULL,
	upwpntype character varying(2) NOT NULL,
	upwpnok integer,
	upwpnoknot integer,
	upwpntot integer,
	CONSTRAINT pk_t_upwpn PRIMARY KEY (wpnadlocode, upwpntype),
	CONSTRAINT fk_t_upwpn_wpnadlocode_t_adloc FOREIGN KEY (wpnadlocode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwpn_upwpntype_lex_wp007 FOREIGN KEY (upwpntype) REFERENCES lex_wp007 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE UNIQUE INDEX idx_t_upwpn_wpnadlocode_upwpntype ON t_upwpn USING btree (wpnadlocode, upwpntype);
CREATE INDEX idx_t_upwpn_wpnadlocode ON t_upwpn USING btree (wpnadlocode);
CREATE INDEX idx_t_upwpn_upwpntype ON t_upwpn USING btree (upwpntype);

COMMENT ON TABLE t_upwpn IS 'Number of functional and non functional water points by category for each locality (LOC_WP)';
COMMENT ON COLUMN t_upwpn.wpnadlocode IS 'Locality code (LWP001)';
COMMENT ON COLUMN t_upwpn.upwpntype IS 'Type of water point from WP007 (LWP002)';
COMMENT ON COLUMN t_upwpn.upwpnok IS 'Number of functional water points in the category (LWP003)';
COMMENT ON COLUMN t_upwpn.upwpnoknot IS 'Number of non-functional water points in the category (LWP004)';
COMMENT ON COLUMN t_upwpn.upwpntot IS 'Total number of water points in the category (LWP005)';



--
-- Table: t_upwpn
--
DROP TABLE IF EXISTS t_upsfn CASCADE;
CREATE TABLE t_upsfn
(
	sfnadlocode character varying(20) NOT NULL,
	upsfntype character varying(2) NOT NULL,
	upsfnok integer,
	upsfnoknot integer,
	upsfntot integer,
	CONSTRAINT pk_t_upsfn PRIMARY KEY (sfnadlocode, upsfntype),
	CONSTRAINT fk_t_upsfn_sfnadlocode_t_adloc FOREIGN KEY (sfnadlocode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_upwpn_upwpntype_lex_sf006 FOREIGN KEY (upsfntype) REFERENCES lex_sf006 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE UNIQUE INDEX idx_t_upsfn_sfnadlocode_upsfntype ON t_upsfn USING btree (sfnadlocode, upsfntype);
CREATE INDEX idx_t_upsfn_sfnadlocode ON t_upsfn USING btree (sfnadlocode);
CREATE INDEX idx_t_upsfn_upsfntype ON t_upsfn USING btree (upsfntype);

COMMENT ON TABLE t_upsfn IS 'Number of functional and non functional water points by category for each locality (LOC_SP)';
COMMENT ON COLUMN t_upsfn.sfnadlocode IS 'Locality code (LSP001)';
COMMENT ON COLUMN t_upsfn.upsfntype IS 'Type of water point from WP007 (LSP002)';
COMMENT ON COLUMN t_upsfn.upsfnok IS 'Number of functional water points in the category (LSP003)';
COMMENT ON COLUMN t_upsfn.upsfnoknot IS 'Number of non-functional water points in the category (LSP004)';
COMMENT ON COLUMN t_upsfn.upsfntot IS 'Total number of water points in the category (LSP005)';


--
-- Table: t_upis_comments
--

CREATE TABLE t_upis_comments
(
	id serial NOT NULL,
	code character varying(20),
	"user" character varying(20),
	user_name character varying(100),
	user_email character varying(100),
	submit_date date DEFAULT ('now'::text)::date,
	rating integer,
	"comment" text,
	ip_address character varying(16),
	is_public boolean,
	is_removed boolean,
	CONSTRAINT pk_upis_comments PRIMARY KEY (id),
	CONSTRAINT fk_t_upis_comments_code_t_upis FOREIGN KEY (code) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_upis_comments_fk_code ON t_upis_comments USING btree (code);

COMMENT ON TABLE t_upis_comments IS 'User comments on waterpoints';
COMMENT ON COLUMN t_upis_comments.id IS 'Identifier for the comment';
COMMENT ON COLUMN t_upis_comments.code IS 'Identifier for the commented waterpoint';
COMMENT ON COLUMN t_upis_comments."user" IS 'Identifier for the user who posted the comment';
COMMENT ON COLUMN t_upis_comments.submit_date IS 'Date of the post';
COMMENT ON COLUMN t_upis_comments.rating IS 'Rating given by user';
COMMENT ON COLUMN t_upis_comments."comment" IS 'The comment';
COMMENT ON COLUMN t_upis_comments.ip_address IS 'IP address of the commenter';
COMMENT ON COLUMN t_upis_comments.is_public IS '1 if comment is public, 0 for restricted';
COMMENT ON COLUMN t_upis_comments.is_removed IS '1 if comment is removed, 0 for available';



--
-- Table: t_uppws
--
DROP TABLE IF EXISTS t_uppws CASCADE;
CREATE TABLE t_uppws
(
	upiscode character varying(20) NOT NULL,
	uppwsowntype character varying(2),
	uppwsctrar character varying(2),
	uppwspop integer,
	uppwslonbr integer,
	uppwsstcpdi double precision,
	uppwsminhd double precision,
	uppwsstcp double precision,
	uppwslentot double precision,
	uppwstapnbr integer,
	uppwsconnbr integer,
	uppwscapnbr integer,
	uppwscatnbr integer,
	uppwswdpnbr integer,
	uppwsfunc character varying(2),
	uppwsabdm double precision,
	uppwscapday double precision,
	uppwscapdr double precision,
	uppwsenerc character varying(2),
	uppwsextent character varying(2),
	uppwsschlen integer,
	uppwsabstpe character varying(2),
	uppwschldis boolean,
	uppwstpsys boolean,
	uppwspricept double precision,
	uppwspricedt double precision,
	uppwspriceos double precision,
	uppwspriceun character varying(10),
	CONSTRAINT pk_t_uppws PRIMARY KEY (upiscode),
	CONSTRAINT fk_t_uppws_upiscode_t_upis FOREIGN KEY (upiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT fk_t_uppws_uppwsowntype_lex_pws005 FOREIGN KEY (uppwsowntype) REFERENCES lex_pws005 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppws_uppwsctrar_lex_pws007 FOREIGN KEY (uppwsctrar) REFERENCES lex_pws007 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppws_uppwsfunc_lex_pws020 FOREIGN KEY (uppwsfunc) REFERENCES lex_pws020 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppws_uppwsenerc_lex_pws024 FOREIGN KEY (uppwsenerc) REFERENCES lex_pws024 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppws_uppwsextent_lex_pws025 FOREIGN KEY (uppwsextent) REFERENCES lex_pws025 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppws_uppwsabstpe_lex_pws027 FOREIGN KEY (uppwsabstpe) REFERENCES lex_pws027 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_uppws_upiscode ON t_uppws USING btree(upiscode);
CREATE INDEX idx_t_uppws_uppwsowntype ON t_uppws USING btree(uppwsowntype);
CREATE INDEX idx_t_uppws_uppwsctrar ON t_uppws USING btree(uppwsctrar);
CREATE INDEX idx_t_uppws_uppwsfunc ON t_uppws USING btree(uppwsfunc);
CREATE INDEX idx_t_uppws_uppwsenerc ON t_uppws USING btree(uppwsenerc);
CREATE INDEX idx_t_uppws_uppwsextent ON t_uppws USING btree(uppwsextent);
CREATE INDEX idx_t_uppws_uppwsabstpe ON t_uppws USING btree(uppwsabstpe);

COMMENT ON TABLE t_uppws IS 'Specific characterisitics of Piped Water Scheme (PWS)';
COMMENT ON COLUMN t_uppws.upiscode IS 'National identification code of the scheme (PWS001)';
COMMENT ON COLUMN t_uppws.uppwsowntype IS 'Ownership of the scheme (PWS005)';
COMMENT ON COLUMN t_uppws.uppwsctrar IS 'Contracting arrangement (PWS007)';
COMMENT ON COLUMN t_uppws.uppwspop IS 'Total population served by the scheme (PWS008)';
COMMENT ON COLUMN t_uppws.uppwslonbr IS 'Number of localities served by the scheme (PWS009)';
COMMENT ON COLUMN t_uppws.uppwsstcpdi IS 'Total water storage capacity for distribution (m3) (PWS011)';
COMMENT ON COLUMN t_uppws.uppwsminhd IS 'Minimum head on the scheme (m) (PWS012)';
COMMENT ON COLUMN t_uppws.uppwsstcp IS 'Total water storage capacity (m3) (PWS013)';
COMMENT ON COLUMN t_uppws.uppwslentot IS 'Total length of pipes (m) (PWS014)';
COMMENT ON COLUMN t_uppws.uppwstapnbr IS 'Number of public taps (PWS015)';
COMMENT ON COLUMN t_uppws.uppwsconnbr IS 'Number of connections (PWS016)';
COMMENT ON COLUMN t_uppws.uppwscapnbr IS 'Number of posts for carts (PWS017)';
COMMENT ON COLUMN t_uppws.uppwscatnbr IS 'Number of cattle troughs (PWS018)';
COMMENT ON COLUMN t_uppws.uppwswdpnbr IS 'Number of other water delivery points (PWS019)';
COMMENT ON COLUMN t_uppws.uppwsfunc IS 'Functionality of the scheme (code PWS020)';
COMMENT ON COLUMN t_uppws.uppwsabdm IS 'Mean daily water abstraction (m3/day) (PWS021)';
COMMENT ON COLUMN t_uppws.uppwscapday IS 'Nominal daily capacity (m3/day) (PWS022)';
COMMENT ON COLUMN t_uppws.uppwscapdr IS 'Rate of use of the daily capacity % (PWS023)';
COMMENT ON COLUMN t_uppws.uppwsenerc IS 'Energy source(s) configuration (PWS024)';
COMMENT ON COLUMN t_uppws.uppwsextent IS 'Extent of piped scheme (PWS025)';
COMMENT ON COLUMN t_uppws.uppwsschlen IS 'Total scheme length (m) (PWS026)';
COMMENT ON COLUMN t_uppws.uppwsabstpe IS 'Type of water resource abstracted (PWS027)';
COMMENT ON COLUMN t_uppws.uppwschldis IS 'Integrated chlorination disposal (PWS028)';
COMMENT ON COLUMN t_uppws.uppwstpsys IS 'Water treatment system (PWS029)';
COMMENT ON COLUMN t_uppws.uppwspricept IS 'Water sale price at public taps (per m3, in national currency) (PWS030)';
COMMENT ON COLUMN t_uppws.uppwspricedt IS 'Water sale price at domestic taps (per m3, in national currency) (PWS031)';
COMMENT ON COLUMN t_uppws.uppwspriceos IS 'Other sale price (in national currency) (PWS032)';
COMMENT ON COLUMN t_uppws.uppwspriceun IS 'Price unit for other sale price (PWS033)';



--
-- Table: t_uppwsen
--
DROP TABLE IF EXISTS t_uppwsen CASCADE;
CREATE TABLE t_uppwsen
(
	uppencode serial NOT NULL,
	enisupiscode character varying(20),
	uppensrc character varying(2),
	uppenet character varying(9),
	uppenrp double precision,
	uppenup double precision,
	uppenstart date,
	CONSTRAINT pk_t_uppwsen PRIMARY KEY (uppencode),
	CONSTRAINT fk_t_uppwsen_enisupiscode_t_upis FOREIGN KEY (enisupiscode) REFERENCES t_upis(upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwsen_uppensrc_lex_pwe003 FOREIGN KEY (uppensrc) REFERENCES lex_pwe003 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwsen_uppenet_t_cat_item FOREIGN KEY (uppenet) REFERENCES t_cat_item(id_item) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);
CREATE INDEX idx_t_uppwsen_enisupiscode ON t_uppwsen USING btree(enisupiscode);

COMMENT ON TABLE t_uppwsen IS 'This table describes the sources of energy on each piped water (PWS_EN)';
COMMENT ON COLUMN t_uppwsen.uppencode IS 'Energy source identification code (PWE001)';
COMMENT ON COLUMN t_uppwsen.enisupiscode IS 'National identification code of the scheme (PWE002->PWS001)';
COMMENT ON COLUMN t_uppwsen.uppensrc IS 'Category of energy source (PWE003)';
COMMENT ON COLUMN t_uppwsen.uppenet IS 'Brand and type of the energy source (PWE004)';
COMMENT ON COLUMN t_uppwsen.uppenrp IS 'Rated power. Value extracted from catalog table (PWE005)';
COMMENT ON COLUMN t_uppwsen.uppenup IS 'Unit of power. Value extracted from catalog table (PWE006)';
COMMENT ON COLUMN t_uppwsen.uppenstart IS 'Starting date (PWE007)';



--
-- Table: t_uppwspi
--
DROP TABLE IF EXISTS t_uppwspi CASCADE;
CREATE TABLE t_uppwspi
(
	upppicode serial,
	piisupiscode character varying(20),
	upppicp character varying(2),
	upppimdp character varying(9),
	upppilen double precision,
	upppistart date,
	CONSTRAINT pk_t_uppwspi PRIMARY KEY (upppicode),
	CONSTRAINT fk_t_uppwspi_piisupiscode_t_upis FOREIGN KEY (piisupiscode) REFERENCES t_upis(upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspi_upppicp_lex_pwi003 FOREIGN KEY (upppicp) REFERENCES lex_pwi003 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspi_upppimdp_t_cat_item FOREIGN KEY (upppimdp) REFERENCES t_cat_item(id_item) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);
CREATE INDEX idx_t_uppwspi_piisupiscode ON t_uppwspi USING btree(piisupiscode);

COMMENT ON TABLE t_uppwspi IS 'This table describes the piping system on each piped water scheme (PWS_PI)';
COMMENT ON COLUMN t_uppwspi.upppicode IS 'Pipe identification code (PWP001)';
COMMENT ON COLUMN t_uppwspi.piisupiscode IS 'National identification code of the scheme (PWI002->PWS001)';
COMMENT ON COLUMN t_uppwspi.upppicp IS 'Category of pipe (PWI003)';
COMMENT ON COLUMN t_uppwspi.upppimdp IS 'Material / Diameter / Rated pressure. Pointer on the "Pipes" catalog (PWI004)';
COMMENT ON COLUMN t_uppwspi.upppilen IS 'Length (m) (PWI005)';
COMMENT ON COLUMN t_uppwspi.upppistart IS 'Starting date (PWI006)';



--
-- Table: t_uppwspp
--
DROP TABLE IF EXISTS t_uppwspp CASCADE;
CREATE TABLE t_uppwspp
(
	uppppcode serial NOT NULL,
	ppisupiscode character varying(20),
	uppphyi integer,
	uppppws character varying(20),
	upppppump character varying(2),
	uppppbtp character varying(9),
	uppppenr character varying(2),
	uppppoy double precision,
	uppppotmh double precision,
	uppppidp double precision,
	uppppdwtoy double precision,
	uppppstart date,
	CONSTRAINT pk_t_uppwspp PRIMARY KEY (uppppcode),
	CONSTRAINT fk_t_uppwspp_ppisupiscode_t_upis FOREIGN KEY (ppisupiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspp_uppppws_lex_pwp004 FOREIGN KEY (uppppws) REFERENCES lex_pwp004 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspp_upppppump_lex_pwp005 FOREIGN KEY (upppppump) REFERENCES lex_pwp005 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspp_uppppenr_lex_pwp007 FOREIGN KEY (uppppenr) REFERENCES lex_pwp007 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspp_uppppbtp_t_cat_item FOREIGN KEY (uppppbtp) REFERENCES t_cat_item(id_item) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwspp_uppphyi_t_hyentity FOREIGN KEY (uppphyi) REFERENCES t_hyentity(hyecode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_uppwspp_ppisupiscode ON t_uppwspp USING btree(ppisupiscode);

COMMENT ON TABLE t_uppwspp IS 'This table describes the pumping equipment on each piped water scheme (PWS_PP)';
COMMENT ON COLUMN t_uppwspp.uppppcode IS 'Pump identification code (PWP001)';
COMMENT ON COLUMN t_uppwspp.ppisupiscode IS 'National identification code of the scheme (PWP002->PWS001)';
COMMENT ON COLUMN t_uppwspp.uppphyi IS 'Hydraulic inventory code of the water source (PWP003)';
COMMENT ON COLUMN t_uppwspp.uppppws IS 'Category of water source (PWP004)';
COMMENT ON COLUMN t_uppwspp.upppppump IS 'Category of pump (PWP005)';
COMMENT ON COLUMN t_uppwspp.uppppbtp IS 'Brand and type of pump. Pointer on the "Pumps" catalog (PWP006)';
COMMENT ON COLUMN t_uppwspp.uppppenr IS 'Category of energy required for the pump (PWP007)';
COMMENT ON COLUMN t_uppwspp.uppppoy IS 'Operating yield (m3/h) (PWP008)';
COMMENT ON COLUMN t_uppwspp.uppppotmh IS 'Operating total manometric head (PWP009)';
COMMENT ON COLUMN t_uppwspp.uppppidp IS 'Installation depth of the pump (m) (PWP010)';
COMMENT ON COLUMN t_uppwspp.uppppdwtoy IS 'Dynamic water table at operating yield (m) (PWP011)';
COMMENT ON COLUMN t_uppwspp.uppppstart IS 'Starting date (PWP012)';



--
-- Table: t_uppwsst
--
DROP TABLE IF EXISTS t_uppwsst CASCADE;
CREATE TABLE t_uppwsst
(
	uppstcode serial NOT NULL,
	stisupiscode character varying(20),
	uppstcr character varying(2),
	uppstbm character varying(2),
	uppsttc double precision,
	uppsthbt double precision,
	uppstminh double precision,
	uppstx double precision,
	uppsty double precision,
	uppststart date,
	CONSTRAINT pk_t_uppwsst PRIMARY KEY (uppstcode),
	CONSTRAINT fk_t_uppwsst_stisupiscode_t_upis FOREIGN KEY (stisupiscode) REFERENCES t_upis(upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwsst_uppstcr_lex_pwss003 FOREIGN KEY (uppstcr) REFERENCES lex_pwss003 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwsst_uppstbm_lex_pwss004 FOREIGN KEY (uppstbm) REFERENCES lex_pwss004 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

SELECT AddGeometryColumn('public', 't_uppwsst', 'uppstcoordinate', 4326, 'POINT', 2);

CREATE INDEX idx_t_uppwsst_stisupiscode ON t_uppwsst USING btree(stisupiscode);

COMMENT ON TABLE t_uppwsst IS 'This table describes the water reservoirs on each piped water scheme (PWS_ST)';
COMMENT ON COLUMN t_uppwsst.uppstcode IS 'Reservoir identification code (PWS001)';
COMMENT ON COLUMN t_uppwsst.stisupiscode IS 'National identification code of the scheme (PWS002->PWS001)';
COMMENT ON COLUMN t_uppwsst.uppstcr IS 'Category of reservoir (PWS003)';
COMMENT ON COLUMN t_uppwsst.uppstbm IS 'Building material of the tank (PWS004)';
COMMENT ON COLUMN t_uppwsst.uppsttc IS 'Tank capacity (m3) (PWS005)';
COMMENT ON COLUMN t_uppwsst.uppsthbt IS 'Height from ground to the bottom of the tank (m) (PWS006)';
COMMENT ON COLUMN t_uppwsst.uppstminh IS 'Minimum head on the scheme (m) (PWS007)';
COMMENT ON COLUMN t_uppwsst.uppstx IS 'GPS coordinate (longitude) (PWS008)';
COMMENT ON COLUMN t_uppwsst.uppsty IS 'GPS coordinate (latitude) (PWS009)';
COMMENT ON COLUMN t_uppwsst.uppststart IS 'Starting date (PWS010)';



--
-- Table: t_uppwstt
--
DROP TABLE IF EXISTS t_uppwstt CASCADE;
CREATE TABLE t_uppwstt
(
	uppttcode serial NOT NULL,
	ttisupiscode character varying(20),
	upptttype character varying(2),
	uppttdesc character varying,
	uppttequip character varying(2),
	CONSTRAINT pk_t_uppwstt PRIMARY KEY (uppttcode),
	CONSTRAINT fk_t_uppwstt_ttisupiscode_t_upis FOREIGN KEY (ttisupiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwstt_upptttype_lex_pwt002 FOREIGN KEY (upptttype) REFERENCES lex_pwt002 (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_uppwstt_uppttequip_t_cat_item FOREIGN KEY (uppttequip) REFERENCES t_cat_item(id_item) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_uppwstt_ttisupiscode ON t_uppwstt USING btree(ttisupiscode);

COMMENT ON TABLE t_uppwstt IS 'This table describes the water treatment equipment on each piped water scheme (PWS_TT)';
COMMENT ON COLUMN t_uppwstt.uppttcode IS 'Water treatment equipement identification code';
COMMENT ON COLUMN t_uppwstt.ttisupiscode IS 'Piped water scheme identification code (PWT001)';
COMMENT ON COLUMN t_uppwstt.upptttype IS 'Type of water treatment (PWT002)';
COMMENT ON COLUMN t_uppwstt.uppttdesc IS 'Description of the equipment (PWT003)';
COMMENT ON COLUMN t_uppwstt.uppttequip IS 'Identification code of the equipment in the catalog (PWT004)';



-----------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------- LINKED TABLES ---------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------------------------------------------

---
--- Table: tl_adopr (OP_ZON)
---
DROP TABLE IF EXISTS tl_adopr CASCADE;
CREATE TABLE tl_adopr (
	opradopcode integer NOT NULL,
	opradrcode character varying(20) NOT NULL,
	
	CONSTRAINT pk_tl_adopr PRIMARY KEY (opradrcode, opradopcode),
	CONSTRAINT fk_tl_adopr_opradopcode_t_adope FOREIGN KEY (opradopcode) REFERENCES t_adope (adopcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT fk_tl_adopr_opradrcode_t_adreglev FOREIGN KEY (opradrcode) REFERENCES t_adreglev (code) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE
);

COMMENT ON TABLE tl_adopr IS 'Link between projects and subdivision (OP_ZON)';
COMMENT ON COLUMN tl_adopr.opradopcode IS 'Code of the project (OPZ001)';
COMMENT ON COLUMN tl_adopr.opradrcode IS 'Code of the subdivision (OPZ002)';



---
--- Table: tl_adoploc (OP_LOC)
---
DROP TABLE IF EXISTS tl_adoploc CASCADE;
CREATE TABLE tl_adoploc
(
	oploadlocode character varying(20) NOT NULL,
	oploadopcode integer NOT NULL,
	CONSTRAINT pk_tl_adoploc PRIMARY KEY (oploadlocode, oploadopcode),
	CONSTRAINT fk_tl_adoploc_oploadopcode_t_adope FOREIGN KEY (oploadopcode) REFERENCES t_adope (adopcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT fk_tl_adoploc_oploadlocode_t_adloc FOREIGN KEY (oploadlocode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE
);

COMMENT ON TABLE tl_adoploc IS 'Link between projects and localities (OP_LOC)';
COMMENT ON COLUMN tl_adoploc.oploadlocode IS 'Code of the locality (OPL002)';
COMMENT ON COLUMN tl_adoploc.oploadopcode IS 'Code of the project (OPL001)';



--
-- Table: tl_aulocpws (PWS_LOC)
--
DROP TABLE IF EXISTS tl_aulocpws CASCADE;
CREATE TABLE tl_aulocpws
(
	pwsisupiscode character varying(20) NOT NULL,
	loadlocode character varying(20) NOT NULL,
	CONSTRAINT pk_tl_aulocpws PRIMARY KEY (loadlocode, pwsisupiscode),
	CONSTRAINT fk_tl_auloc_pwsisupiscode_t_upis FOREIGN KEY (pwsisupiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_tl_auloc_loadlocode_t_adloc FOREIGN KEY (loadlocode) REFERENCES t_adloc (adlocode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_tl_aulocpws_pwsisupiscode ON tl_aulocpws USING btree (pwsisupiscode);
CREATE INDEX idx_tl_aulocpws_loadlocode ON tl_aulocpws USING btree (loadlocode);

COMMENT ON TABLE tl_aulocpws IS 'This table links each water scheme to the localities served by this scheme (PWS_LOC)';
COMMENT ON COLUMN tl_aulocpws.pwsisupiscode IS 'National identification code of the scheme (PWL001->PWS001)';
COMMENT ON COLUMN tl_aulocpws.loadlocode IS 'Locality code (PWL002->LOC001)';



--
-- Table: tl_uppwswp (PWS_WP)
--
DROP TABLE IF EXISTS tl_uppwswp CASCADE;
CREATE TABLE tl_uppwswp
(
	pwsisupiscode character varying(20) NOT NULL,
	wpisupiscode character varying(20) NOT NULL,
	CONSTRAINT pk_tl_uppwswp PRIMARY KEY (pwsisupiscode, wpisupiscode),
	CONSTRAINT fk_tl_uppwswp_pwsisupiscode_t_upis FOREIGN KEY (pwsisupiscode) REFERENCES t_uppws (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_tl_uppwswp_wpisupiscode_t_upwp FOREIGN KEY (wpisupiscode) REFERENCES t_upwp (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

COMMENT ON TABLE tl_uppwswp IS 'This table links each piped water scheme to its public waterpoints (PWS_WP)';
COMMENT ON COLUMN tl_uppwswp.pwsisupiscode IS 'Piped water scheme code - National identification code of the scheme (PWW001->PWS001)';
COMMENT ON COLUMN tl_uppwswp.wpisupiscode IS 'Water point code - National identification code of the scheme (PWW002->WP001)';



--
-- Table: tl_uhisbas
--
DROP TABLE IF EXISTS tl_uhisbas CASCADE;
CREATE TABLE tl_uhisbas
(
	bashybcode integer NOT NULL,
	isupiscode character varying(20) NOT NULL,
	CONSTRAINT pk_tl_uhisbas PRIMARY KEY (bashybcode, isupiscode),
	CONSTRAINT fk_tl_uhisbas_bashybcode_t_hybasin FOREIGN KEY (bashybcode) REFERENCES t_hybasin (hybcode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_tl_uhisbas_isupiscode_t_upis FOREIGN KEY (isupiscode) REFERENCES t_upis (upiscode) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

COMMENT ON TABLE tl_uhisbas IS 'Table link between infrastructure and basins';
COMMENT ON COLUMN tl_uhisbas.bashybcode IS 'Basin code';
COMMENT ON COLUMN tl_uhisbas.isupiscode IS 'Infrastructure code';




---
--- Table: watsan_cfg
---
DROP TABLE IF EXISTS watsan_cfg CASCADE;
CREATE TABLE watsan_cfg
(
	id character varying NOT NULL,
	numeric_value real,
	string_value character varying,
	"comment" character varying,
	CONSTRAINT pk_watsan_cfg PRIMARY KEY (id)
);

COMMENT ON TABLE watsan_cfg IS 'Watsan configuration variables. Keeps various variables used within computations';
COMMENT ON COLUMN watsan_cfg.id IS 'Primary key';
COMMENT ON COLUMN watsan_cfg.numeric_value IS 'Value if this variable is of numeric type. Then string_value is NULL';
COMMENT ON COLUMN watsan_cfg.string_value IS 'Value if this variable is of string type. Then numeric_value is NULL';
COMMENT ON COLUMN watsan_cfg."comment" IS 'Description of the variable. What it does within the application';


INSERT INTO watsan_cfg (id, numeric_value, string_value, comment) VALUES ('max_persons_ewp', 10, NULL, 'Maximum number of persons served by 1 EWP (Equivalent Water Point)');
INSERT INTO watsan_cfg (id, numeric_value, string_value, comment) VALUES ('dummy', 0, NULL, 'DO NOT DELETE - Internally managed');
