--
-- DDL for technical catalogs table
-- Author(s): Cristian Romanescu <cristian.romanescu@eaudeweb.ro>
-- This file contains the database structure common to both WatSan website and country portal.
-- Please manually keep both structure identical (through update scripts in production etc) because they are synchronizing.
--
-- Designed using a simplified version of Entity-Attribute-Value (EAV) model to represent multiple types of items into a single table. Simplified because
-- we know all possible data types for attributes: string, real, integer, boolean and lexicon. Lexicon is when an attribute value comes from an lexicon.

---
--- Table: t_cat_attribute
---
DROP TABLE IF EXISTS t_cat_attribute CASCADE;
CREATE TABLE t_cat_attribute
(
	id_attribute character varying(9) NOT NULL,
	"name" character varying NOT NULL, 
	is_string boolean NOT NULL DEFAULT FALSE,
	is_integer boolean NOT NULL DEFAULT FALSE,
	is_real boolean NOT NULL DEFAULT FALSE,
	is_bool boolean NOT NULL DEFAULT FALSE,
	is_lexicon boolean NOT NULL DEFAULT FALSE,
	is_url boolean NOT NULL DEFAULT FALSE,
	is_picture boolean NOT NULL DEFAULT FALSE,
	lexicon_klass character varying,
	enabled boolean NOT NULL DEFAULT TRUE,
	reference boolean NOT NULL DEFAULT TRUE,
	created timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
	author character varying NOT NULL,

	CONSTRAINT pk_t_cat_attribute PRIMARY KEY (id_attribute), 
	CONSTRAINT unq_t_cat_attribute_name UNIQUE ("name")
);

CREATE INDEX idx_t_cat_attribute_id_attribute ON t_cat_attribute USING btree (id_attribute);
CREATE INDEX idx_t_cat_attribute_name ON t_cat_attribute USING btree (name);

COMMENT ON TABLE t_cat_attribute IS 'Attributes definition table';
COMMENT ON COLUMN t_cat_attribute.id_attribute IS 'Primary key - attribute ID. Format is REFXXX/LOCXXX, where REF is for reference attributes (website), LOC is for local attributes (locally defined)';
COMMENT ON COLUMN t_cat_attribute.name IS 'Name of the attribute in english';
COMMENT ON COLUMN t_cat_attribute.is_string IS 'Attribute is of type string (other is_ fields are false). Values for this attribute are stored in column t_cat_catalog_data.value_string';
COMMENT ON COLUMN t_cat_attribute.is_integer IS 'Attribute is of type integer (other is_ fields are false). Values for this attribute are stored in table t_cat_catalog_data.value_integer';
COMMENT ON COLUMN t_cat_attribute.is_real IS 'Attribute is of type real (other is_ fields are false). Values for this attribute are stored in table t_cat_catalog_data.value_real';
COMMENT ON COLUMN t_cat_attribute.is_bool IS 'Attribute is of type boolean (other is_ fields are false). Values for this attribute are stored in table t_cat_catalog_data.value_bool';
COMMENT ON COLUMN t_cat_attribute.is_lexicon IS 'Attribute is of type lexicon (other is_ fields are false). Values for this attribute are stored in table t_cat_catalog_data.value_lexicon';
COMMENT ON COLUMN t_cat_attribute.is_url IS 'Attribute is of type URL (other is_ fields are false). Values for this attribute are stored in table t_cat_catalog_data.value_string';
COMMENT ON COLUMN t_cat_attribute.is_picture IS 'Attribute is of type picture (url to a picture) (other is_ fields are false). Values for this attribute are stored in table t_cat_catalog_data.value_string';
COMMENT ON COLUMN t_cat_attribute.lexicon_klass IS 'Used internally by the framework. Lexicon Python class name if attribute is a lexicon type. Specifies ORM class mapping for lexicon';
COMMENT ON COLUMN t_cat_attribute.enabled IS 'Attribute is enabled and displayed in user interface when adding new attributes to a catalog';
COMMENT ON COLUMN t_cat_attribute.reference IS 'Attribute is part of reference attributes. Reference attributes come from main website and cannot be disabled in local portal instance, only in website';
COMMENT ON COLUMN t_cat_attribute.author IS 'Record author';
COMMENT ON COLUMN t_cat_attribute.created IS 'Date when record was created';



---
--- Table: t_cat_catalog
---
DROP TABLE IF EXISTS t_cat_catalog CASCADE;
CREATE TABLE t_cat_catalog
(
	id_catalog character varying(9) NOT NULL,
	name character varying,
	enabled boolean NOT NULL DEFAULT TRUE,
	reference boolean NOT NULL DEFAULT TRUE,
	created timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
	author character varying NOT NULL,

	CONSTRAINT pk_t_cat_catalog PRIMARY KEY (id_catalog),
	CONSTRAINT unq_t_cat_catalog_name UNIQUE ("name")
);

CREATE INDEX idx_t_cat_catalog_id_catalog ON t_cat_catalog USING btree (id_catalog);
CREATE INDEX idx_t_cat_catalog_name ON t_cat_catalog USING btree (name);

COMMENT ON TABLE t_cat_catalog IS 'This table containing existing catalogs';
COMMENT ON COLUMN t_cat_catalog.id_catalog IS 'Primary key - catalog ID. Format is REFXXX/LOCXXX, where REF is for reference catalogs (website), LOC is for local catalogs (locally defined)';
COMMENT ON COLUMN t_cat_catalog.name IS 'Catalog name in english';
COMMENT ON COLUMN t_cat_catalog.enabled IS 'Catalog is enabled and displayed in user interface and used when searching for lookup';
COMMENT ON COLUMN t_cat_catalog.reference IS 'Catalog is part of reference attributes. Reference catalogs come from main website and cannot be edited/disabled in local portal instance, only in website';
COMMENT ON COLUMN t_cat_catalog.created IS 'Date when catalog was created';
COMMENT ON COLUMN t_cat_catalog.author IS 'Catalog author';



---
--- Table: t_cat_catalog_dd
---
DROP TABLE IF EXISTS t_cat_catalog_dd CASCADE;
CREATE TABLE t_cat_catalog_dd
(
	id_catalog character varying(9) NOT NULL,
	id_attribute character varying(9) NOT NULL,
	show_in_overview boolean NOT NULL DEFAULT false,
	CONSTRAINT pk_t_cat_catalog_dd PRIMARY KEY (id_catalog, id_attribute),
	CONSTRAINT unq_t_cat_catalog_dd UNIQUE (id_catalog, id_attribute),
	CONSTRAINT fk_t_cat_catalog_dd_id_catalog FOREIGN KEY (id_catalog) REFERENCES t_cat_catalog (id_catalog) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT,
	CONSTRAINT fk_t_cat_catalog_dd_id_attribute FOREIGN KEY (id_attribute) REFERENCES t_cat_attribute (id_attribute) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_cat_catalog_dd_id_catalog ON t_cat_catalog_dd USING btree (id_catalog);
CREATE INDEX idx_t_cat_catalog_dd_id_attribute ON t_cat_catalog_dd USING btree (id_attribute);

COMMENT ON TABLE t_cat_catalog_dd IS 'Catalog data definition. Defines the attributes for a single catalog of items. Used to build UI that constructs forms to insert data into catalogs';
COMMENT ON COLUMN t_cat_catalog_dd.id_catalog IS 'Primary key. Catalog identification code';
COMMENT ON COLUMN t_cat_catalog_dd.id_attribute IS 'Primary key. Attribute identification code';
COMMENT ON COLUMN t_cat_catalog_dd.show_in_overview IS 'If true, this attribute is displayed in catalog overview HTML table as column. For best results set only relevant attributes (i.e. cubic capacity, power output) to true';


---
--- Table: t_cat_item
---
DROP TABLE IF EXISTS t_cat_item CASCADE;
CREATE TABLE t_cat_item
(
	id_item character varying(9) NOT NULL, 
	id_catalog character varying(9) NOT NULL,
	brand character varying NOT NULL,
	model character varying NOT NULL,
	created timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
	author character varying NOT NULL,

	CONSTRAINT pk_t_cat_item PRIMARY KEY (id_item),
	CONSTRAINT fk_t_cat_item_id_catalog FOREIGN KEY (id_catalog) REFERENCES t_cat_catalog (id_catalog) MATCH SIMPLE ON UPDATE CASCADE ON DELETE RESTRICT
);

CREATE INDEX idx_t_cat_item_id_item ON t_cat_item USING btree (id_item);
CREATE INDEX idx_t_cat_item_id_catalog ON t_cat_item USING btree (id_catalog);
CREATE INDEX idx_t_cat_item_brand ON t_cat_item USING btree (brand);
CREATE INDEX idx_t_cat_item_model ON t_cat_item USING btree (model);

COMMENT ON TABLE t_cat_item IS 'Catalog item definition table';
COMMENT ON COLUMN t_cat_item.id_item IS 'Primary key - item ID. Format is REFXXX/LOCXXX, where REF is for reference catalogs (website), LOC is for local catalogs (locally defined)';
COMMENT ON COLUMN t_cat_item.id_catalog IS 'Catalog where this item belongs to';
COMMENT ON COLUMN t_cat_item.brand IS 'Item brand. Manufacturer - All items share this attribute';
COMMENT ON COLUMN t_cat_item.model IS 'Item model. All items share this attribute';



---
--- Table: t_cat_catalog_data
---
DROP TABLE IF EXISTS t_cat_catalog_data CASCADE;
CREATE TABLE t_cat_catalog_data
(
	id_item character varying(9) NOT NULL,
	id_attribute character varying(9) NOT NULL,
	value_string character varying,
	value_real double precision,
	value_integer integer,
	value_bool boolean,
	value_lexicon character varying(2),
	CONSTRAINT t_cat_catalog_data_pkey PRIMARY KEY (id_item, id_attribute),
	CONSTRAINT unq_t_cat_catalog_data UNIQUE (id_item, id_attribute)
);

CREATE INDEX idx_t_cat_catalog_data_id_item ON t_cat_catalog_data USING btree (id_item);
CREATE INDEX idx_t_cat_catalog_data_id_attribute ON t_cat_catalog_data USING btree (id_attribute);
CREATE INDEX idx_t_cat_catalog_data_value_string ON t_cat_catalog_data USING btree (value_string);
CREATE INDEX idx_t_cat_catalog_data_value_real ON t_cat_catalog_data USING btree (value_real);
CREATE INDEX idx_t_cat_catalog_data_value_integer ON t_cat_catalog_data USING btree (value_integer);
CREATE INDEX idx_t_cat_catalog_data_value_bool ON t_cat_catalog_data USING btree (value_bool);
CREATE INDEX idx_t_cat_catalog_data_value_lexicon ON t_cat_catalog_data USING btree (value_lexicon);

COMMENT ON TABLE t_cat_catalog_data IS 'Catalog data with attribute values for each item';
COMMENT ON COLUMN t_cat_catalog_data.id_item IS 'Primary key. Item identification code';
COMMENT ON COLUMN t_cat_catalog_data.id_attribute IS 'Primary key. Item attribute ID';
COMMENT ON COLUMN t_cat_catalog_data.value_string IS 'Attribute value if attribute is string type';
COMMENT ON COLUMN t_cat_catalog_data.value_real IS 'Attribute value if attribute is real type';
COMMENT ON COLUMN t_cat_catalog_data.value_integer IS 'Attribute value if attribute is integer type';
COMMENT ON COLUMN t_cat_catalog_data.value_bool IS 'Attribute value if attribute is boolean type';
COMMENT ON COLUMN t_cat_catalog_data.value_lexicon IS 'Attribute value if attribute is lexicon type';
