# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cristian Romanescu, Eau De Web
# Andrei Laza, Eau de Web
#Python imports
from formencode import validators
from types import ListType
from ws.common.fixtures import Fixtures
from ws.common.sql.queries.StatisticalData import WatsanStatisticalData
import os, ConfigParser
import sys, traceback, string, urllib
import zipfile
import simplejson as json
from datetime import datetime
import mapscript


#Zope imports
import zLOG
from Globals import InitializeClass
from Products.PageTemplates.PageTemplateFile import PageTemplateFile
from AccessControl import ClassSecurityInfo, getSecurityManager
import Globals
from AccessControl.Permissions import view, view_management_screens
from z3c.sqlalchemy.util import createSAWrapper, registeredWrappers
from persistent.mapping import PersistentMapping

#Naaya imports
from Products.Naaya.NySite import NySite
from Products.Naaya.NyFolder import addNyFolder
from naaya.core.StaticServe import StaticServeFromZip, StaticServeFromFolder
from Products.NaayaCore.FormsTool.NaayaTemplate import NaayaPageTemplateFile
from Products.NaayaCore.EmailTool.EmailPageTemplate import EmailPageTemplateFile

#Watsan imports
from ws.common.sql.admin import update_portal_configuration
from ws.common.sql.mappings.Lexicon import lexicon_list
from ws.common.exceptions import DatabaseException
from ws.common.sql import query, pg_create_database, pg_execute_sql_file, pg_connect,\
    pg_drop_database, pg_database_exists, pg_execute_statement
from ws.common.sql.queries.SiteSearch import SiteSearch
from ws.common.sql.queries import Access as access_query
from ws.common.utilities import valid_url, decode_country_name, unzip_file
from ws.common.utilities.rating.rating import Rating
from ws.common.utilities.paginate import DiggPaginator, EmptyPage, InvalidPage

from ws.common.sync.DictionarySync import DictionarySync
from Products.UserRegistration.UserRegistration import manage_addUserRegistration
from Products.UMap.UMap import UMap

from Products.WSPortal.Access.WSPortalAccess import create_wsportalaccess_object_callback
from Products.WSPortal.Facilities.WSPortalFacilities import create_wsportalfacilities_object_callback
from Products.WSPortal.Programs.WSPortalPrograms import create_wsportalprograms_object_callback
from Products.WSPortal.Resources.WSPortalResources import create_wsportalresources_object_callback
from Products.WSPortal.IWRM.IWRMProduct import create_iwrm_object_callback
from Products.WSPortal.Comments import models
from Products.WSPortal.Comments.Comments import Comment

import forms
import constants

WEBSITE_ID = 'wwebsite' # from Products.WSWebsite.constants import WEBSITE_ID
DEFAULT_USER_SETTINGS = 'watsan_default_users_settings'

EDITABLE_EXT = ['map']

default_settings = {}
default_settings['locality.umap.backgrounds'] = ['google', 'google_hyb', 'google_sat']
default_settings['add_facility.umap.backgrounds'] = ['google_merc', 'google_hyb_merc', 'google_sat_merc']
default_settings['locality.umap.overlays'] = []
default_settings['subdivision.umap.overlays'] = []

email_templates = {
    'feedback': EmailPageTemplateFile('emailpt/feedback.zpt', globals()),
}

manage_addWSPortal_html = PageTemplateFile('zpt/portal_manage_add', globals())
def manage_addWSPortal(parent, country_code='', subdivisions='', languages='', skin='', moderation='', mapserver_url='', gmap_key='', REQUEST=None):
    """
        Create new WSPortal instance inside database.
        Parameters:
            `parent`
                Where to attach this website to
            `id`
                Unique ID of the portal
            `lang`
                Language, default None
            `country_code`
                Country code (ISO 2-letter code)
            `subdivisions`
                Number of subdivisions
            `languages`
                Languages available within portal
            `skin`
                Name of the skin used for portal
            `moderation`
                Type of moderation for user contribution (pre/post posting)
            `REQUEST`
                Zope's HTTP request object
        Returns newly created object or redirects if REQUEST is not None
        Raises exception if any error occurrs during portal creation
    """
    raise NotImplemented
    ## Validation of input parameters
    #country_name = decode_country_name(country_code)
    #zLOG.LOG(__name__, zLOG.INFO, 'Creating new WatSan Country Portal for %s' % country_name)
    #
    #cp = ConfigParser.ConfigParser()
    #cp.read(os.path.join(os.path.dirname(__file__), 'sql', 'setup.ini'))
    #db_superuser = cp.get('database', 'super_username')
    #db_superpass = cp.get('database', 'super_password')
    #host = cp.get('database', 'database_host')
    #sdb = cp.get('database', 'super_database')
    #db_user = cp.get('username')
    #db_pass = cp.get('password')
    #
    ##Fix country name to be database-naming compatible
    #country_code = country_code.lower()
    #db_name = 'watsan_portal_%s' % country_code
    ## ZOPE specific
    #portal_uid = 'country_portal_%s' % (country_code)
    #zLOG.LOG(__name__, zLOG.INFO, '     * Creating Zope website: %s/' % portal_uid)
    #parent._setObject(portal_uid, WSPortal(portal_uid, portal_uid, 'Country portal for %s' % country_name,
    #                                       'en', country_code, db_name, subdivisions, skin,
    #                                       moderation,
    #                                       db_user, db_pass, host, mapserver_url, gmap_key))
    #ob = parent._getOb(portal_uid)
    #ob.createPortalTools()
    #ob.loadDefaultData()
    #for lang in languages:
    #    ob.gl_add_site_language(lang)
    #ob.getLayoutTool().manageLayout('skin', skin)
    #
    ## DATABASE
    #conn = None
    #if not pg_database_exists(db_superuser, db_superpass, host, db_name, sdb):
    #    try:
    #        try:
    #            zLOG.LOG(__name__, zLOG.INFO, '    * Creating PostgreSQL database named %s' % db_name)
    #            pg_create_database(db_superuser, db_superpass, host, db_name, db_user, sdb, zLOG)
    #
    #            conn = pg_connect(db_superuser, db_superpass, host, db_name)
    #            pg_execute_statement(conn, 'GRANT SELECT, UPDATE, INSERT, DELETE ON TABLE geometry_columns TO %s;' % db_user, True, False, zLOG)
    #            pg_execute_statement(conn, 'GRANT SELECT, UPDATE, INSERT, DELETE ON TABLE spatial_ref_sys TO %s;' % db_user, True, True, zLOG)
    #
    #            base_path = cp.get('sql', 'base_path')
    #            files = ['%s/%s' % (base_path, file) for file in cp.get('sql', 'files').split('\n')]
    #            files.append('%s/data/%s.sql' % (base_path, country_code)) # try to load data for that country
    #            for file in files:
    #                if(valid_url(file)):
    #                    url_file = urllib.urlopen(file)
    #                    conn = pg_connect(db_user, db_pass, host, db_name)
    #                    zLOG.LOG(__name__, zLOG.INFO, '    * Executing SQL script: %s' % file)
    #                    pg_execute_sql_file(conn, url_file, True, zLOG)
    #                    url_file.close()
    #                else:
    #                    zLOG.LOG(__name__, zLOG.WARNING, '    * Skipping (not found) file: %s' % file)
    #        except:
    #            pg_drop_database(db_superuser, db_superpass, host, db_name, sdb, zLOG)
    #            type, val, tb = sys.exc_info()
    #            zLOG.LOG(__name__, zLOG.ERROR, 'Error occurred . WatSan country portal creation process encountered a fatal error', error = (type, val, tb), reraise=True)
    #            msg = string.join(traceback.format_exception(type, val, tb), '')
    #            del type, val, tb
    #            raise DatabaseException(msg)
    #    finally:
    #        if conn:
    #            del conn
    #else:
    #    zLOG.LOG(__name__, zLOG.WARNING, 'Database %s already exists for this portal. It will be used as current database' % db_name)
    #
    #if REQUEST is not None:
    #    return parent.manage_main(parent, REQUEST, update_menu=1)
    #
    #zLOG.LOG(__name__, zLOG.INFO, 'Portal was successfully created. URL is: %s' % ob.absolute_url())
    #return ob


class UserSettings(object):


    def __init__(self, username, defaults = {}):
        self.username = username
        self.values = defaults


    def get(self, key, default_value=None):
        if key in self.values:
            return self.values[key]
        return default_value


    def has_setting(self, key):
        return key in self.values.keys()


    def set(self, key, value):
        self.values[key] = value


class WSPortal(NySite):
    """
        WSPortal object, folder-type that contains items described within specs.
        This is the root of an WatSan Country Portal.
    """
    meta_type = 'WatSan Country Portal'
    security = ClassSecurityInfo()

    product_paths = NySite.product_paths + [Globals.package_home(globals())]

    # Specific fields
    country_code = None
    subdivisions = None
    skin = None
    level = None
    moderation = None
    is_wswebsite = False # Used to distinct between WSPortal and WSWebsite
    default_latitude = 10
    default_longitude = 10
    default_zoom = 3

    #Setting this to true enables:
    # - coordinates display in map control
    # - SQL statement output in console
    debug = False

    rating = Rating()

    def __init__(self, id, title, lang, country_code):
        """
        Constructor that builds new WSPortal setting specific properties.
        Parameters:
            `country_code`
                Portal's country code (ISO2L code)
        """
        super(WSPortal, self).__init__(id, id, title, lang)
        self.country_code = country_code
        self.gis_path = os.path.join(CLIENT_HOME, 'gis', country_code)
        self.users_settings = {}
        self._p_changed=1
        self.is_wswebsite = False
        self.watsan_website_url = 'http://watsan2.eaudeweb.ro/wwebsite' #@todo: server_url
        self.labels = PersistentMapping({
                    's1': {
                        'name': "subdivision 1",
                        'plural': "subdivisions",
                    },
                    's2': {
                        'name': "subdivision 2",
                        'plural': "subdivisions",
                    },
                    's3': {
                        'name': "subdivision 3",
                        'plural': "subdivisions",
                    },
                    's4': {
                        'name': "subdivision 4",
                        'plural': "subdivisions",
                    },
                    's5': {
                        'name': "subdivision 5",
                        'plural': "subdivisions",
                    },
                    'locality': {
                        'name': "locality",
                        'plural': "localities",
                    },
                })

    security.declarePrivate('loadDefaultData')
    def loadDefaultData(self, *args, **kwargs):
        """
            Load the initial data into a new created WatSan Country Portal
        """
        super(WSPortal, self).loadDefaultData(*args, **kwargs)

        # Add left portlets
        site = self.getSite()
        portlets_tool = site.getPortletsTool()
        portlets_tool.assign_portlet(folder_path='/', position='left', portlet_id='portal_useful_links', inherit=False)
        portlets_tool.assign_portlet(folder_path='news_articles', position='left', portlet_id='portal_news_articles', inherit=True)

        # Create the structure
        addNyFolder(self, id='access', callback=create_wsportalaccess_object_callback, title='Access')
        addNyFolder(self, id='facilities', callback=create_wsportalfacilities_object_callback, title='Facilities')
        addNyFolder(self, id='programs', callback=create_wsportalprograms_object_callback, title='Ongoing programs')
        addNyFolder(self, id='iwrm', callback=create_iwrm_object_callback, title='IWRM')
        addNyFolder(self, id='resources', callback=create_wsportalresources_object_callback, title='Resources')
        self.resources.loadDefaultData()

        zLOG.LOG(__name__, zLOG.INFO, 'Installing UMap library. GIS path for this portal is: %s' % self.gis_path)
        self._setObject('umap', UMap('umap'))

        #remove Naaya default content
        self.getLayoutTool().manage_delObjects('skin')
        self.getPortletsTool().manage_delObjects(['menunav_links', 'topnav_links'])
        for portlet_id in self.getPortletsTool().get_portlet_ids_for('', 'left'):
            self.getPortletsTool().unassign_portlet('', 'left', portlet_id)
        for portlet_id in self.getPortletsTool().get_portlet_ids_for('', 'center'):
            self.getPortletsTool().unassign_portlet('', 'center', portlet_id)

        #add watsan portal content
        manage_addUserRegistration(self)
        self.loadSkeleton(Globals.package_home(globals()))
        addNyFolder(self, id='news_articles', title='News & articles')
        news_articles_folder = site._getOb('news_articles')
        addNyFolder(news_articles_folder, id='news', title='News', folder_meta_types='Naaya News')
        addNyFolder(news_articles_folder, id='articles', title='Articles', folder_meta_types='Naaya Document')

        #TODO: UNCOMMENT NEXT LINE (Resource management folder removed for the moment)
        #addNyFolder(self, id='iwrm', title='Resource management')
        #TODO: REMOVE NEXT LINE (Hardcoded portal description for the moment)
        self._setLocalPropValue('description', 'en', 'The WatSan Platform enables African countries to rapidly set up a free database '
                'and website for monitoring water and sanitation services. Web portals created on the WatSan '
                'Platform are participatory tools for information dissemination in the water and sanitation '
                'sector at national and regional level. They can be used to generate maps which highlight '
                'the distribution of water and sanitation services.')

        #change mainsections
        self.getPropertiesTool().manageMainTopics(['access', 'facilities', 'programs', 'news_articles', 'resources', 'iwrm'])

        #add watsan portlets
        self.getPortletsTool().assign_portlet('', 'left', 'portlet_useful_links', True)
        self.getPortletsTool().assign_portlet('', 'right', 'portlet_latestuploads_rdf', False)
        self.getPortletsTool().assign_portlet('facilities', 'left', 'portlet_facilities_useful_links', False)

        # Create the CLIENT_HOME/gis/XX folder to keep the GIS
        if not os.path.exists(self.gis_path):
            os.makedirs(self.gis_path)
            zLOG.LOG(__name__, zLOG.INFO, 'Creating GIS location for MapServer data for this portal: %s' % self.gis_path)

        # Create the default user settings and save them
        zLOG.LOG(__name__, zLOG.INFO, 'Installing default user settings')
        self.users_settings[DEFAULT_USER_SETTINGS] = UserSettings(DEFAULT_USER_SETTINGS, default_settings)
        self._p_changed=1

        #recaptcha keys
        self.recaptcha_public_key   = '6LcQ1AsAAAAAAL_DVBgE37oDKLTg26u29PNpQ3ep'
        self.recaptcha_private_key  = '6LcQ1AsAAAAAAFnAgZR2nvEFO95977pT63PLKN-Y'

        # role-permissions mappers
        self.manage_permission('Naaya - Skip Captcha', roles=['Authenticated', 'Editor', 'Program manager', 'Waterpoint manager', 'District/municipal agent', 'WSS services agent', 'Administrator', 'Manager'], acquire=1)
        self.manage_permission('Add Naaya Forum Message', roles=['Authenticated', 'Editor', 'Program manager', 'Waterpoint manager', 'District/municipal agent', 'WSS services agent', 'Administrator', 'Manager'], acquire=1)
        self.manage_permission('Watsan: Give rating', roles=['Authenticated', 'Editor', 'Program manager', 'Waterpoint manager', 'District/municipal agent', 'WSS services agent', 'Administrator', 'Manager'], acquire=1)
        self.manage_permission('Watsan: Report malfunction', roles=['Waterpoint manager', 'District/municipal agent', 'WSS services agent', 'Manager'], acquire=1)
        self.manage_permission('Watsan: Report non inventoried', roles=['Authenticated', 'Editor', 'Program manager', 'Waterpoint manager', 'District/municipal agent', 'WSS services agent', 'Administrator', 'Manager'], acquire=1)
        self.manage_permission('Watsan: Add/Edit Program', roles=['Program manager', 'Manager'], acquire=1)
        self.manage_permission('Watsan: Add/Edit Organisation', roles=['Program manager', 'Manager', 'Editor', 'Administrator'], acquire=1)
        self.manage_permission('Watsan: Edit portal data', roles=['WSS services agent', 'Administrator', 'Manager'], acquire=1)
        #self.manage_permission('Watsan: Post photo', roles=['Authenticated', 'Editor', 'Program manager', 'Waterpoint manager', 'District/municipal agent', 'WSS services agent', 'Administrator', 'Manager'], acquire=1)

        schema_tool = site.getSchemaTool()
        doc_schema = schema_tool.getSchemaForMetatype('Naaya Document')
        doc_schema.addWidget('resourceurl', sortorder=120, widget_type='String', label='Concerned URL', localized=True)

    security.declareProtected(view, 'custom_editor')
    def custom_editor(self, editor_tool, lang, dom_id):

        extra_options = {
            #'content_css': '%s/misc_/NaayaContent/tb-editor.css' % self.absolute_url(),
            'theme_advanced_buttons1':
                'bold,italic,underline,strikethrough,sub,sup,removeformat,separator, \
                bullist,numlist,separator, \
                justifyleft,justifycenter,justifyright,justifyfull,separator, \
                link,unlink,hr,separator, \
                pastetext,pasteword,cleanup,code',
            'theme_advanced_buttons2': '',
            'edit_image_url': '',
            'select_image_url': '',
            'plugins': 'advlink,directionality,preview,paste,inlinepopups',
        }
        return editor_tool.render(dom_id, lang, image_support=False, extra_options=extra_options)

    
    def get_db_session(self):
        """
        Retrive managed database connection
        Return:
            SQLAlchemy database session
        """
        wrapper = None
        if self.db_name in registeredWrappers.keys():
            wrapper = registeredWrappers[self.db_name]
        else:
            wrapper = createSAWrapper('postgresql://%s:%s@%s/%s' \
                                      % (self.db_username, self.db_password, self.db_host, self.db_name),
                                      name=self.db_name,
                                      engine_options = {'echo' : self.debug, 'encoding' : 'utf-8'})
        return wrapper.session
    
    
    def _delete_wrapper(self):
        """
        Delete the Z3C.SQLAlchemy registered wrapper (created by get_db_session)
        """
        if self.db_name in registeredWrappers.keys():
            del registeredWrappers[self.db_name]
            self._p_changed = 1


    security.declarePublic('get_user_settings')
    def get_user_settings(self, username=None):
        """
        Retrieve user settings customized.
        Parameters:
            `p_username`
                Specific user name to retrieve settings for. User must exists, otherwise exception is raised.
        Return:
            Object of type UserSettings or None if none found
        """
        if not username in self.users_settings.keys():
            return None
        return self.users_settings[username]


    security.declarePublic('get_setting')
    def get_setting(self, key, format=None ,p_username=None, fail=False):
        """
        Retrieve single
        Parameters:
            `format`
                Format of the returned value. Default is None. Possible values
                supported are: 'json'
            `p_username`
                Specific user name. If none specified, checks for
                REQUEST.AUTHENTICATED_USER. If none, retrieves the default
                saved settings.
            `fail`
                If fail is set then it doesn't try autopatcher and returns None
        Return:
            Value associated with key. None if no setting found (even default).
        """
        ret = None
        username = p_username
        if not username:
            if self.REQUEST and self.REQUEST.AUTHENTICATED_USER:
                username = self.REQUEST.AUTHENTICATED_USER.getUserName()
        user_settings = self.get_user_settings(username)
        if user_settings and user_settings.has_setting(key):
            ret = user_settings.get(key)
        else:
            default_settings = self.get_user_settings(DEFAULT_USER_SETTINGS)
            if default_settings.has_setting(key):
                ret = default_settings.get(key)
            else:
                zLOG.LOG(__name__, zLOG.WARNING, 'Cannot retrieve setting %s. Looked in user settings (%s) and default settings. Please update the system (patches of WSPortal). Returning None.' % (key, p_username))
                if not fail:
                    #Try one more time, maybe autopatcher fixed the missing value
                    self.user_settings_autopatcher(username)
                    ret = self.get_setting(key, format, p_username, fail=True)
        if format == 'json':
            ret = json.dumps(ret)
        zLOG.LOG(__name__, zLOG.DEBUG, 'WSPortal::get_setting(): %s=%s' % (key, ret))
        return ret


    def set_setting(self, key, value, p_username=None):
        """
        Set new user setting.
        Parameters:
            `key`
                Setting key to configure
            `value`
                Value associated with this key
            `p_username`
                If None defaults to DEFAULT_USER_SETTINGS and changes the defaults
        Return:
            Nothing
        """
        username = DEFAULT_USER_SETTINGS
        if p_username:
            username = p_username

        user_settings = self.get_user_settings(username)
        if not user_settings:
            zLOG.LOG(__name__, zLOG.INFO, 'Creating new customized user settings for (%s)' % username)
            self.users_settings[username] = UserSettings(username, {key : value})
            self._p_changed=1
        else:
            user_settings.set(key, value)
            self._p_changed=1
        zLOG.LOG(__name__, zLOG.INFO, 'WSPortal::set_setting(): %s=%s' % (key, value))


    def user_settings_autopatcher(self, username=None):
        """
        Provides on-the-fly patcher for user_settings. This way when we insert
        new settings it automatically applies the patch into ZODB objects,
        without requiring to run an update.
        """
        patches = { #Add eacch new property here
            'locality.umap.backgrounds' : ['google', 'google_hyb', 'google_sat'],
            'locality.umap.overlays' : [],
            'subdivision.umap.overlays' : [],
            'add_facility.umap.backgrounds' : ['google_merc', 'google_hyb_merc', 'google_sat_merc']
        }
        zLOG.LOG(__name__, zLOG.INFO, 'Running autopatcher for default user settings and current user (%s)' % username)
        user_settings = self.get_user_settings(username)
        default_settings = self.get_user_settings(DEFAULT_USER_SETTINGS)
        new_patches = []
        for key in patches.keys():
            if user_settings and not user_settings.has_setting(key):
                user_settings.set(key, patches[key])
                new_patches.append(key)
            if default_settings and not default_settings.has_setting(key):
                default_settings.set(key, patches[key])
                if key not in new_patches:
                    new_patches.append(key)
        if new_patches:
            zLOG.LOG(__name__, zLOG.INFO, '    * WSPortal::user_settings patched with new settings: (%s)' % ','.join(new_patches))
            self._p_changed=1


    _feedback = NaayaPageTemplateFile('zpt/feedback', globals(), 'ws_portal_feedback')
    security.declareProtected(view, 'feedback')
    def feedback(self, REQUEST):
        """ Sends an email to the administrator of the portal from any portal page """

        errors = success = None
        if REQUEST.form.has_key('btn-send-feedback'):
            form_values = dict(REQUEST.form)

            #get the user credentials
            if self.isAnonymousUser():
                form_values['user'] = ''
            else:
                form_values['user'] = REQUEST.AUTHENTICATED_USER.getUserName()

            #validate form fields
            try:
                form = forms.FeedbackForm.to_python(form_values)
            except validators.Invalid, e:
                errors = e
            else:
                form_values['page_url'] = REQUEST['referer']
                form_values['site_title'] = self.site_title
                self.send_notification(template='feedback', **form_values)
                success = True
            return self._feedback(REQUEST, errors=errors, success=success, referer=REQUEST['referer'])
        return self._feedback(REQUEST)

    security.declarePrivate('_get_template')
    def _get_template(self, name):
        template = self._getOb('emailpt_%s' % name, None)
        if template is not None:
            return template.render_email

        template = email_templates.get(name, None)
        if template is not None:
            return template.render_email

        raise ValueError('template for %r not found' % name)

    security.declarePrivate('send_notification')
    def send_notification(self, template, **kwargs):
        """ """
        portal = self.getSite()
        email_tool = portal.getEmailTool()

        message = {'url': self.absolute_url()}
        for k, v in kwargs.items():
            message[k] = v

        if kwargs['user']:
            auth_tool = self.getAuthenticationTool()
            u = auth_tool.getUser(kwargs['user'])
            if u and u.email:
                message['user_email'] = u.email
            else:
                message['user_email'] = ''

            full_name = auth_tool.getUserFullName(u)
            if full_name:
                message['user_name'] = full_name
            else:
                message['user_name'] = ''

        template = self._get_template(template)
        mail_data = template(**message)
        email_tool.sendEmail(p_content = mail_data['body_text'],
                             p_to = email_tool.administrator_email,
                             p_from = email_tool._get_from_address(),
                             p_subject = mail_data['subject'])

    def captcha_errors(self, contact_word, REQUEST):
        if self.checkPermissionSkipCaptcha():
            return None
        if not self.recaptcha_is_present():
            return None
        return self.validateCaptcha(contact_word, REQUEST)

    security.declareProtected(view_management_screens, 'reset_user_settings')
    def reset_user_settings(self):
        """ Resets the user settings to their default """
        self.users_settings = {}
        self.users_settings[DEFAULT_USER_SETTINGS] = UserSettings(DEFAULT_USER_SETTINGS, default_settings)
        self._p_changed=1
        return 'OK! Done, boss!'

    #backwards compatibility
    security.declareProtected(view, 'requestrole_html')
    def requestrole_html(self, REQUEST=None):
        """ """
        return REQUEST.RESPONSE.redirect(self.accounts.absolute_url() + '/signup')

    security.declareProtected(view, 'forgotpassword_html')
    def forgotpassword_html(self, REQUEST=None):
        """ """
        return REQUEST.RESPONSE.redirect(self.accounts.absolute_url() + '/forgot_password')

    security.declareProtected(view, 'feedback_html')
    feedback_html = feedback

    #------------------------------------------------

    security.declareProtected(view, 'get_localities_names')
    def get_localities_names(self, parent=None, order='adloname'):
        """ """
        localities = access_query.get_localities(self.get_db_session(), parent, order)
        return [loc.adloname.encode('utf-8') for loc in localities.all() if loc.adloname]

    security.declareProtected(view, 'do_search')
    def do_search(self, q=None, lookup = []):
        """ Website search functionality """
        session = self.get_db_session()
        search_ob = SiteSearch(session)
        search_ob.do_search(q, lookup)
        return search_ob
    
    security.declareProtected(constants.GIVE_RATING, 'addRating')
    def addRating(self):
        """ dummy method """
        pass

    security.declarePublic('html_highlight')
    def html_highlight(self, search_ob, statement):
        """ Shortcut """
        return search_ob.html_highlight(statement)

    ### getters ###
    security.declarePublic('url_subdivision')
    def url_subdivision(self, id):
        return '%s/subdivision?id=%s' % (self.access.absolute_url(), id)

    security.declarePublic('url_locality')
    def url_locality(self, locality):
        code = locality
        if hasattr(locality, 'adlocode'):
            code = locality.adlocode
        return '%s/locality?id=%s' % (self.access.absolute_url(), code)

    security.declarePublic('url_access')
    def url_access(self):
        return self.access.absolute_url()

    security.declarePublic('url_facilities')
    def url_facilities(self):
        return self.facilities.absolute_url()

    security.declarePublic('url_sanitation')
    def url_sanitation(self, id):
        return '%s/sanitations/view?id=%s' % (self.facilities.absolute_url(), id)

    security.declarePublic('url_pipedwaterscheme')
    def url_pipedwaterscheme(self, id):
        return '%s/pipedwaterschemes/view?id=%s' % (self.facilities.absolute_url(), id)

    security.declarePublic('url_waterpoint')
    def url_waterpoint(self, id):
        return '%s/waterpoints/view?id=%s' % (self.facilities.absolute_url(), id)

    security.declarePublic('url_organisation')
    def url_organisation(self, id):
        return '%s/directory/view?id=%s' % (self.resources.absolute_url(), id)

    security.declarePublic('url_resources')
    def url_resources(self):
        return '%s/directory' % (self.resources.absolute_url())

    security.declarePublic('url_catalogs')
    def url_catalogs(self, code, type=''):
        return '%s/%s/?code=%s' % (self.resources.technical_catalogs.absolute_url(), type, code)


    def get_photos_gallery(self):
        return self.resources.multimedia.photos


    security.declarePublic('get_default_coordinates')
    def default_coordinates(self):
        """
        Retrieve the portal's default coordinates
        Return:
            Tuple containing (latitude=y, longitude=x)
        """
        return {
                    'longitude' : self.default_longitude,
                    'latitude' : self.default_latitude,
                    'x' : self.default_longitude,
                    'y' : self.default_latitude,
                    'zoom' : self.default_zoom
                }


    def is_portal(self):
        """ Specifies if this Zope object is of type WSPortal. Always returns True
        """
        return True

    def get_google_map_default_key(self):
        """ """
        return self.gmap_key.get(self.REQUEST.HTTP_X_FORWARDED_SERVER, '')

    def get_google_maps_url(self):
        """ """
        return """
        <script type="text/javascript">
document.write([
  '<script src="http://maps.google.com/maps?file=api&amp;v=2&amp;key=', 
    %s
  [window.location.host],
  '" type="text/javascript"><\/script>'
].join(''));
</script>
""" % self.rstk.json_dumps(self.gmap_key.data)


    ############################
    # Administration interface #
    ############################
    # Manage statistical generation for existing data
    _statistics = NaayaPageTemplateFile('zpt/admin/statistics', globals(), 'ws_admin_statistics')
    security.declareProtected(view_management_screens, 'statistics')
    def statistics(self, REQUEST):
        """ Statistics management """
        stats = WatsanStatisticalData(self.get_db_session())
        if REQUEST.form.has_key('btn-compute-locality'):
            stats.compute_locality_statistics()

        elif REQUEST.form.has_key('btn-compute-subdivision'):
            stats.compute_subdivision_statistics()

        elif REQUEST.form.has_key('btn-compute-rank'):
            stats.compute_localities_national_rank()
            stats.compute_subdivision_national_rank()

        elif REQUEST.form.has_key('btn-compute-statistics'):
            stats.compute_locality_statistics()
            stats.compute_subdivision_statistics()
            stats.compute_localities_national_rank()
            stats.compute_subdivision_national_rank()

        count_locality_stats = stats.count_locality_statistics()
        count_localities = stats.count_localities()
        count_subdivision_stats = stats.count_locality_statistics()
        count_subdivisions = stats.count_localities()

        valid_loc = count_localities == count_locality_stats
        valid_subdiv = count_subdivisions == count_subdivision_stats

        return self._statistics(REQUEST,
                                valid_loc = valid_loc,
                                valid_subdiv = valid_subdiv,
                                valid_rank = stats.valid_rank())

    # Generate test data into the database
    _fixtures = NaayaPageTemplateFile('zpt/admin/fixtures', globals(), 'ws_admin_fixtures')
    security.declareProtected(view_management_screens, 'fixtures')
    def fixtures(self, REQUEST):
        """ Fixture management """
        fixture = Fixtures(self.get_db_session(), self.default_latitude - 1.5, self.default_longitude - 1.5, self.default_latitude + 1.5, self.default_longitude + 1.5)
        if REQUEST.form.has_key('btn-add-fixtures'):
            if REQUEST.form.has_key('tu'):
                fixture.generate_demo_users(self)
            if REQUEST.form.has_key('wp'):
                fixture.generate_water_points(1000)
            if REQUEST.form.has_key('sf'):
                fixture.generate_sanitation_facilities(1000)
            if REQUEST.form.has_key('pws'):
                fixture.generate_piped_water_schemes(1000)
            if REQUEST.form.has_key('wpn'):
                fixture.generate_wpn()
            if REQUEST.form.has_key('sfn'):
                fixture.generate_sfn()
            if REQUEST.form.has_key('orga'):
                fixture.generate_organisations(30)
            if REQUEST.form.has_key('op'):
                fixture.generate_programs(self, 20)
            if REQUEST.form.has_key('country'):
                fixture.generate_country_statistics(self.country_code)
            if REQUEST.form.has_key('country_history'):
                fixture.generate_country_historic_access(self.country_code)
            if REQUEST.form.has_key('sf_access'):
                fixture.generate_sf_access()
            if REQUEST.form.has_key('iwrm'):
                fixture.generate_iwrm()
        return self._fixtures(REQUEST, tu_count = len(self.getAuthenticationTool().getUsers()),
            wp_count = fixture.wp_count(), sf_count = fixture.sf_count(),
            orga_count = fixture.orga_count(), op_count = fixture.op_count(),
            loc_count = fixture.loc_count(), subdiv_count = fixture.subdiv_count(),
            wpn_count = fixture.wpn_count(), sfn_count = fixture.sfn_count(),
            pws_count = fixture.pws_count(),
            country_historic_count = fixture.country_historic_count()
        )


    _dictionaries = NaayaPageTemplateFile('zpt/admin/dictionaries', globals(), 'ws_portal_admin_dictionaries')
    security.declareProtected(view_management_screens, 'dictionaries')
    def dictionaries(self, REQUEST):
        """ Dictionaries management """
        default_url = '%s/dictionary_sync?format=json' % self.watsan_website_url
        url_dictionary_sync = REQUEST.get('url_dictionary_sync', default_url)

        lexicon_keys = []
        lexicon_descriptions = {}

        for key in lexicon_list.keys():
            lexicon_keys.append(key)
            lexicon_descriptions[key] = lexicon_list[key]['table_name']
        lexicon_keys.sort()

        rows = None
        lexicon = None
        if REQUEST.form.has_key('btn-show-lexicon'):
            lexicon = REQUEST.form.get('lexicon', None)
            if lexicon:
                session = self.get_db_session()
                klass = lexicon_list[lexicon]['table_class']
                rows = session.query(klass).order_by('code').all()

        sync = False
        sync_success = False
        report = None
        if REQUEST.form.has_key('btn-sync'):
            sync = True
            dc = DictionarySync(self.get_db_session(), url_dictionary_sync)
            sync_success = dc.sync_dictonaries()
            report = dc.report

        return self._dictionaries(REQUEST, lexicon_keys=lexicon_keys, rows=rows,
                              lexicon_descriptions=lexicon_descriptions, lexicon=lexicon,
                              sync=sync, report=report, sync_success = sync_success,
                              url_dictionary_sync=url_dictionary_sync)


    security.declarePublic('search')
    search = NaayaPageTemplateFile('zpt/search', globals(), 'ws_search')

    setup = StaticServeFromFolder('setup', globals(), cache=False)
    wat_media = StaticServeFromFolder('media', globals(), cache=False)

    autocomplete = StaticServeFromZip('jquery-autocomplete', 'media/js/jquery.autocomplete.zip', globals())

    _configuration = NaayaPageTemplateFile('zpt/admin/configuration', globals(), 'ws_admin_configuration')
    security.declareProtected(view_management_screens, 'configuration')
    def configuration(self, REQUEST):
        """ WSPortal configuration page """
        error = None
        success = False

        if REQUEST.form.get('btn-save-configuration'):
            try:
                update_portal_configuration(self, REQUEST)
                success = True
            except validators.Invalid, e:
                error = e
        return self._configuration(REQUEST, error=error, success=success)

    _comment = NaayaPageTemplateFile('zpt/admin/comment', globals(), 'ws_admin_comment_details')
    security.declareProtected(view_management_screens, 'comment')
    def comment(self, REQUEST):
        """ WSPortal administration area: comment details page """
        success = False
        session = self.get_db_session()
        id = REQUEST.get('id', '')
        section = REQUEST.get('id', '')

        if section == 'ar':
            comment_model = models.RegionComment
        elif section == 'al':
            comment_model = models.LocComment
        else:
            comment_model = models.UpisComment

        comments_tool = Comment(model = comment_model, code = None,
                                session = session, context = self)
        comment = comments_tool.get_comment(id)
        if REQUEST.form.has_key('btn-manage-comment'):
            id = self.utConvertToList(REQUEST.get('id', []))
            action = REQUEST.form.get('action', '')
            if action == 'approve':
                comments_tool.moderate_comments(id, approve=True)
                success = True
            elif action == 'pending':
                comments_tool.moderate_comments(id, approve=False)
                success = True
            elif action == 'delete':
                comments_tool.trash_comments(id)
                return REQUEST.RESPONSE.redirect('%s/comments' % self.absolute_url())
        return self._comment(REQUEST, comment = comment, section = section, success = success)

    _comments = NaayaPageTemplateFile('zpt/admin/comments', globals(), 'ws_admin_comments')
    security.declareProtected(view_management_screens, 'comments')
    def comments(self, REQUEST):
        """ WSPortal comments moderation page """

        session = self.get_db_session()
        section = REQUEST.get('section', 'fc')
        status = REQUEST.get('status', 'approved')
        date = REQUEST.get('date', '')
        
        if section == 'ar':
            comment_model = models.RegionComment
        elif section == 'al':
            comment_model = models.LocComment
        else:
            comment_model = models.UpisComment

        if status == 'approved':
            approved = True
        else:
            approved = False

        comments_tool = Comment(model = comment_model, code = None,
                                session = session, context = self)

        facilities_comments = comments_tool.count_unapproved_by_model(models.UpisComment)
        region_comments = comments_tool.count_unapproved_by_model(models.RegionComment)
        localities_comments = comments_tool.count_unapproved_by_model(models.LocComment)
        
        if REQUEST.form.has_key('btn-manage-comments'):
            ids = self.utConvertToList(REQUEST.get('ids', []))
            action = REQUEST.form.get('action', '')
            if action == 'approve':
                comments_tool.moderate_comments(ids, approve=True)
            elif action == 'pending':
                comments_tool.moderate_comments(ids, approve=False)
            elif action == 'delete':
                comments_tool.trash_comments(ids)

        items = comments_tool.get_all_comments(approved = approved,
                                            sort_on = REQUEST.get('sort', 'submit_date'),
                                            sort_order = REQUEST.get('order', False))

        paginator = DiggPaginator(items, 20, body=5, padding=2, orphans=5)   #Show 20 documents per page
        
        # Make sure page request is an int. If not, deliver first page.
        try:
            page = int(REQUEST.get('page', '1'))
        except ValueError:
            page = 1
        
        # If page request (9999) is out of range, deliver last page of results.
        try:
            items = paginator.page(page)
        except (EmptyPage, InvalidPage):
            items = paginator.page(paginator.num_pages)

        return self._comments(REQUEST, comments=items,
                            facilities_comments = facilities_comments,
                            region_comments = region_comments,
                            localities_comments = localities_comments)

    _labels = NaayaPageTemplateFile('zpt/admin/labels', globals(), 'ws_admin_labels')
    security.declareProtected(view_management_screens, 'labels_html')
    def labels_html(self, REQUEST):
        """ WSPortal labels management page """
        if REQUEST.form.get('btn-save-labels'):
            self.labels['s1']['name'] = REQUEST.get('s1_name', '')
            self.labels['s1']['plural'] = REQUEST.get('s1_plural', '')

            self.labels['s2']['name'] = REQUEST.get('s2_name', '')
            self.labels['s2']['plural'] = REQUEST.get('s2_plural', '')

            self.labels['s3']['name'] = REQUEST.get('s3_name', '')
            self.labels['s3']['plural'] = REQUEST.get('s3_plural', '')

            self.labels['s4']['name'] = REQUEST.get('s4_name', '')
            self.labels['s4']['plural'] = REQUEST.get('s4_plural', '')

            self.labels['s5']['name'] = REQUEST.get('s5_name', '')
            self.labels['s5']['plural'] = REQUEST.get('s5_plural', '')

            self.labels['locality']['name'] = REQUEST.get('locality_name', '')
            self.labels['locality']['plural'] = REQUEST.get('locality_plural', '')
            self.labels._p_changed = 1
        return self._labels(REQUEST, labels = self.labels)

    _google_keys = NaayaPageTemplateFile('zpt/admin/google_keys', globals(), 'wwebsite_admin_configuration')
    security.declareProtected(view_management_screens, 'google_keys')
    def google_keys(self, REQUEST):
        """ Portal gogole keys administration page """
        if REQUEST.has_key('btn-add-key'):
            domain = REQUEST.get('domain', '')
            gmap_key = REQUEST.get('gmap_key', '')
            if domain and gmap_key:
                self.gmap_key[domain] = gmap_key
                self.gmap_key._p_changed = 1
        if REQUEST.has_key('btn-del-key'):
            ids = REQUEST.get('ids', [])
            if ids:
                for id in ids:
                    del self.gmap_key[id]
                self.gmap_key._p_changed = 1
        return self._google_keys(REQUEST)

    _gis_files = NaayaPageTemplateFile('zpt/admin/gis_files', globals(), 'ws_admin_configuration')
    security.declareProtected(view_management_screens, 'gis_files')
    def gis_files(self, REQUEST):
        """ Manage GIS data """
        message_id = None
        if REQUEST.form.has_key('btn-delete-file'):
            filename = REQUEST.get('filename', None)
            if filename:
                if isinstance(filename, ListType):
                    for name in filename:
                        f = os.path.join(self.gis_path, name)
                        zLOG.LOG(__name__, zLOG.INFO, 'WSPortal::gis_files(): Removing file %s' % f)
                        os.remove(f)
                else:
                    f = os.path.join(self.gis_path, filename)
                    zLOG.LOG(__name__, zLOG.INFO, 'WSPortal::gis_files(): Removing file %s' % f)
                    os.remove(f)
                    message_id = 'deleted'

        if REQUEST.form.has_key('btn-upload-file'):
            file = REQUEST.get('file', None)
            if file:
                dest_path = os.path.join(self.gis_path, file.filename)
                if not os.path.exists(dest_path):
                    try:
                        try:
                            f = open(dest_path, 'w')
                            zLOG.LOG(__name__, zLOG.INFO, 'WSPortal::gis_files(): Uploading file to %s' % dest_path)
                            f.write(file.read())
                            f.close()
                            message_id = 'uploaded'
                        except:
                            type, val, tb = sys.exc_info()
                            zLOG.LOG(__name__, zLOG.ERROR, 'WSPortal::gis_files(): Unable to upload file to %s' % dest_path, error = (type, val, tb))
                    finally:
                        file.close()

                    if zipfile.is_zipfile(dest_path):
                        f = open(dest_path)
                        try:
                            unzip_file(f, self.gis_path, False, zLOG=zLOG)
                            message_id = 'unzipped'
                        except:
                            type, val, tb = sys.exc_info()
                            zLOG.LOG(__name__, zLOG.ERROR, 'WSPortal::gis_files(): Unable to upload file to %s' % dest_path, error = (type, val, tb))
                        os.remove(dest_path)
                else:
                    message_id = 'exists'


        gis_files = filter(lambda x: not os.path.isdir(os.path.join(self.gis_path, x)), os.listdir(self.gis_path))
        gis_stat_files = {}
        for f in gis_files:
            gis_stat_files[f] = {}
            stats = os.stat(os.path.join(self.gis_path, f))
            gis_stat_files[f]['size'] = stats.st_size
            gis_stat_files[f]['last_edit'] = datetime.fromtimestamp(stats.st_mtime).strftime('%Y-%m-%d %H:%M')
            gis_stat_files[f]['editable'] = f[-3:] in EDITABLE_EXT

        return self._gis_files(REQUEST, gis_files=gis_files, gis_stat_files=gis_stat_files, message_id=message_id)



    _gis_edit = NaayaPageTemplateFile('zpt/admin/gis_edit', globals(), 'ws_admin_configuration')
    security.declareProtected(view_management_screens, 'gis_edit')
    def gis_edit(self, REQUEST):
        """ Edit GIS file """
        message_id = None
        content = None
        valid = False
        layers = []
        filename = REQUEST.get('filename', None)
        if REQUEST.form.has_key('btn-save-file'):
            content = REQUEST.get('content', None)
            if content:
                f = open(os.path.join(self.gis_path, filename), 'w')
                f.write(content)
                f.close()
                message_id = 'saved'
        if filename:
            path = os.path.join(self.gis_path, filename)
            if os.path.exists(path):
                f = open(path)
                content = f.read()
                f.close()

            try:
                map_filepath = '%s/%s' % (self.gis_path, filename)
                mapscript.mapObj(map_filepath)
                mobj = mapscript.mapObj(os.path.join(self.gis_path, map_filepath))
                for i in range(mobj.numlayers):
                    layers.append(mobj.getLayer(i))
                valid = True
            except:
                type, val, tb = sys.exc_info()
                zLOG.LOG(__name__, zLOG.ERROR, 'WSPortal::gis_edit() Error while validating (%s). File is not valid' % (map_filepath), error = (type, val, tb), reraise=False)
                message_id = string.join(traceback.format_exception(type, val, tb), '')
        else:
            message_id = 'not_exists'

        return self._gis_edit(REQUEST, message_id=message_id, content=content, valid=valid, filename=filename, layers=layers)


    _ewp = NaayaPageTemplateFile('zpt/admin/ewp', globals(), 'ws_admin_ewp')
    security.declareProtected(view_management_screens, 'ewp')
    def ewp(self, REQUEST):
        """
        This page allows editing the EWP parameters for the portal
        """
        message_id = None

        session = self.get_db_session()
        klass = query.get_lexicon_klass('LexWP007')
        rows = query.get_lexicon_objects(session, klass)

        if REQUEST.form.has_key('btn-save-ewp'):
            message_id = 'updated'
            for row in rows:
                new_value = REQUEST.get(row.code, None)
                try:
                    numeric_new_value = float(new_value)
                except:
                    message_id = 'error'
                    pass # Nothing to do, non-numeric value
                else:
                    row.ewp = numeric_new_value
                    session.merge(row)
            session.flush()
            rows = query.get_lexicon_objects(session, klass)


        return self._ewp(REQUEST, rows=rows, message_id=message_id)


    _paginator_navigation = NaayaPageTemplateFile('zpt/paginator_navigation', globals(), 'ws_paginator')
    def paginator_navigation(self, page_ob, filter_url, sort_order='', currpage='1', sort_on='', REQUEST=None):
        """ This template makes the DiggPaginator reusable, avoiding copy/paste of 30 lines of code in all pages that require pagination.
        """
        return self._paginator_navigation(REQUEST, page_ob=page_ob, filter_url=filter_url, currpage=currpage, sort_order=sort_order, sort_on=sort_on)


    _paginator_tfoot = NaayaPageTemplateFile('zpt/paginator_tfoot', globals(), 'ws_paginator')
    def paginator_tfoot(self, page_ob, colspan, REQUEST=None):
        """ This template makes the DiggPaginator reusable, avoiding copy/paste of tfoot table in all paginated tables of results.
        """
        return self._paginator_tfoot(REQUEST, page_ob=page_ob, colspan=colspan)


    _sortable_column = NaayaPageTemplateFile('zpt/sortable_column', globals(), 'ws_paginator')
    def sortable_column(self, column_title, column_code, filter_url, sort_on, sort_order, currpage, REQUEST=None):
        """ This template renders one sortable table column header.
        """
        return self._sortable_column(REQUEST, column_title=column_title, column_code=column_code, filter_url=filter_url,
                                     sort_on=sort_on, sort_order=sort_order, currpage=currpage)

    #AVAILABLE PERMISSIONS
    def checkPermissionReportNonInventoried(self):
        return getSecurityManager().checkPermission(constants.REPORT_NON_INVENTORIED, self)

    def checkPermissionReportMalfunction(self):
        return getSecurityManager().checkPermission(constants.REPORT_MALFUNCTION, self)

    def checkPermissionPostPhoto(self):
        return getSecurityManager().checkPermission(constants.POST_PHOTO, self)

    def checkPermissionManageProgram(self):
        return getSecurityManager().checkPermission(constants.MANAGE_PROGRAM, self)

    def checkPermissionManageOrganisation(self):
        return getSecurityManager().checkPermission(constants.MANAGE_ORGANISATION, self)

    def checkPermissionGiveRating(self):
        return getSecurityManager().checkPermission(constants.GIVE_RATING, self)

    def checkPermissionEditData(self):
        return getSecurityManager().checkPermission(constants.EDIT_DATA, self)

InitializeClass(WSPortal)
