#Python imports
from formencode import validators

#Zope imports
from AccessControl.SecurityInfo import ClassSecurityInfo
from Globals import InitializeClass

#Naaya imports
from Products.NaayaCore.FormsTool.NaayaTemplate import NaayaPageTemplateFile
from Products.Naaya.NyFolder import NyFolder, addNyFolder
from Products.NaayaCore.EmailTool.EmailPageTemplate import EmailPageTemplateFile

#Watsan imports
from Products.WSPortal import constants
import forms
from ws.common.sql.queries import Access as access_query
from ws.common import get_current_year
from ws.common.sql import update
from ws.common.sql import query
from ws.common.statistics.FacilitiesAccessStatistics import FacilitiesAccessStatistics
from ws.common.utilities.paginate import DiggPaginator, EmptyPage, InvalidPage
from ws.common.utilities.photo_upload.PhotoUpload import PhotoUpload
from Waterpoints import create_waterpoints_object_callback
from SanitationFacilities import create_sanitations_object_callback
from PipedWaterSchemes import create_pipedwaterschemes_object_callback



def create_wsportalfacilities_object_callback(parent, id, contributor):
    ob = WSPortalFacilities(id, contributor)
    parent.gl_add_languages(ob)
    parent._setObject(id, ob)
    ob = parent._getOb(id)
    ob.after_setObject()
    ob.loadDefaultData()
    return ob

email_templates = {
    'report_waterpoint': EmailPageTemplateFile('emailpt/report_waterpoint.zpt', globals()),
    'report_sanitation': EmailPageTemplateFile('emailpt/report_sanitation.zpt', globals()),
}


class WSPortalFacilities(NyFolder):
    """ "Facilities" section of the portal """

    security = ClassSecurityInfo()

    security.declarePrivate('loadDefaultData')
    def loadDefaultData(self, *args, **kwargs):
        """ Load the initial data """
        # Add left portlets
        site = self.getSite()
        site_base_url = site.absolute_url(1)
        location = self.absolute_url(1)
        if site_base_url and location.startswith(site_base_url):
            location = location[len(site_base_url)+1:]

        portlets_tool = site.getPortletsTool()
        addNyFolder(self, id='waterpoints', callback=create_waterpoints_object_callback, title='Water points', folder_meta_types =['Naaya Folder'])
        addNyFolder(self, id='sanitations', callback=create_sanitations_object_callback, title='Sanitation facilities', folder_meta_types =['Naaya Folder'])
        addNyFolder(self, id='pipedwaterschemes', callback=create_pipedwaterschemes_object_callback, title='Piped water schemes', folder_meta_types =['Naaya Folder'])


    security.declarePublic('portlet_filter_data')
    def portlet_filter_data(self, REQUEST):
        """ Data necessary to render filter navigation portlet """
        ret = {
               'subdivisions' : [], 
               'localities' : [], 
        }
        session = self.get_db_session()
        id_subdivision = REQUEST.get('id_subdivision', None)
        ret_subdivisions = []

        if id_subdivision:
            max_level = access_query.get_subdivision_max(session)
            subdivision = access_query.get_subdivision(session, id_subdivision)
            level = subdivision.level
            ob = subdivision

            if level < max_level:
                dd = {}
                dd['level'] = level + 1
                dd['subdivisions'] = access_query.get_subdivision_list(session, ob.code)
                dd['selected'] = None
                ret_subdivisions.append(dd)

            while(ob is not None):
                dd = {}
                dd['level'] = ob.level
                dd['subdivisions'] = access_query.get_subdivision_list(session, ob.parent_code)
                dd['selected'] = ob.id
                ret_subdivisions.append(dd)
                ob = access_query.get_subdivision(session, ob.parent_code)

            ret_subdivisions.reverse()
            if level == max_level:
                dd = {}
                dd['level'] = None
                dd['localities'] = access_query.get_localities(session, subdivision)
                dd['selected'] = id
                ret['localities'] = dd

        else:
            dd = {}
            dd['level'] = 1
            dd['subdivisions'] = access_query.get_subdivisions_by_level(session, 1)
            dd['selected'] = None
            ret_subdivisions.append(dd)

        ret['subdivisions'] = ret_subdivisions
        return ret


    security.declarePrivate('_get_template')
    def _get_template(self, name):
        template = self._getOb('emailpt_%s' % name, None)
        if template is not None:
            return template.render_email
    
        template = email_templates.get(name, None)
        if template is not None:
            return template.render_email
    
        raise ValueError('template for %r not found' % name)

    security.declarePublic('captcha_errors')
    def captcha_errors(self, contact_word, REQUEST):
        if self.checkPermissionSkipCaptcha():
            return None
        if not self.recaptcha_is_present():
            return None
        return self.validateCaptcha(contact_word, REQUEST)

    security.declarePublic('get_lexicon_objects')
    def get_lexicon_objects(self, class_name):
        klass = query.get_lexicon_klass(class_name)
        return query.get_lexicon_objects(self.get_db_session(), klass)

    security.declarePrivate('get_facility_id')
    def get_facility_id(self, type, REQUEST):
        if type == 'wp':
            return REQUEST.form.get('id', '')
        elif type == 'sf':
            return REQUEST.form.get('id', '')

    security.declareProtected(constants.EDIT_DATA, 'list_pending_waterpoints_localities')
    def list_pending_waterpoints_localities(self):
        """ Retrieve the list of water points localities """
        localities = query.get_pending_waterpoints_localities(self.get_db_session())
        # TODO: in ZPT call ob_tuple/0 ob_tuple/1
        return [(loc.adlocode, loc.adloname) for loc in localities if loc]

    security.declareProtected(constants.EDIT_DATA, 'list_pending_sanitation_facilities_localities')
    def list_pending_sanitation_facilities_localities(self):
        """ Retrieve the list of water points localities """
        localities = query.get_pending_sanitation_facilities_localities(self.get_db_session())
        # TODO: in ZPT call ob_tuple/0 ob_tuple/1
        return [(loc.adlocode, loc.adloname) for loc in localities if loc]

    security.declareProtected(constants.EDIT_DATA, 'pending')
    def pending(self, REQUEST):
        """ Databases pending listing """
        session = self.get_db_session()
        type = REQUEST.get('type', 'waterpoints')
        locality = REQUEST.get('locality', '')
        sort_on = REQUEST.get('sort', '')
        sort_order = REQUEST.get('order', True)


        if type == 'waterpoints':
            records = query.get_pending_water_points(session, locality, sort_on, sort_order)
            localities=self.list_pending_waterpoints_localities()
        else: #sanitations
            records = query.get_pending_sanitation_facilities(session, locality, sort_on, sort_order)
            localities=self.list_pending_sanitation_facilities_localities()


        if REQUEST.has_key('btnDelete'):
            selected_items = REQUEST.get('items', [])
            items = [record.upiscode for record in records.all() if record.upiscode in selected_items]
            return self._pending(REQUEST, records=items, localities=[])

        elif REQUEST.has_key('btnConfirmDelete'):
            redirect_url = REQUEST.form.get('destination', '')
            for item in self.utConvertToList(REQUEST.form.get('items', [])):
                if type == 'waterpoints':
                    photo_gallery = PhotoUpload(gallery = 'resources/multimedia/photos/waterpoints',
                                                context=self,
                                                album_id = item,
                                                album_title = '')
                    update.delete_waterpoint(session, item)
                else:
                    photo_gallery = PhotoUpload(gallery = 'resources/multimedia/photos/facilities',
                                                context=self,
                                                album_id = item,
                                                album_title = '')
                    update.delete_sanitation_facility(session, item)
                photo_gallery.delete_album()

            return REQUEST.RESPONSE.redirect('%s&del=ok' % redirect_url)

        else:
            paginator = DiggPaginator(records, 20, body=5, padding=2, orphans=5)   #Show 20 documents per page

            # Make sure page request is an int. If not, deliver first page.
            try:
                page = int(REQUEST.get('page', '1'))
            except ValueError:
                page = 1

            # If page request (9999) is out of range, deliver last page of results.
            try:
                items = paginator.page(page)
            except (EmptyPage, InvalidPage):
                items = paginator.page(paginator.num_pages)

            return self._pending(REQUEST, records=items, localities=localities)

    security.declarePublic('index_html')
    def index_html(self, REQUEST):
        """ Index page for facilities section """
        year = get_current_year()
        statistics = FacilitiesAccessStatistics(self.get_db_session())
        return self._index_html(REQUEST, year=year, statistics=statistics)

    security.declareProtected(constants.REPORT_MALFUNCTION, 'report_non_functioning')
    def report_non_functioning(self, code, template, REQUEST):
        """ Report a non functioning point """
        
        errors = success = None
        form_values = dict(REQUEST.form)

        #get the user credentials
        if self.isAnonymousUser():
            form_values['user'] = ''
        else:
            form_values['user'] = REQUEST.AUTHENTICATED_USER.getUserName()

        #validate form fields
        try:
            form = forms.NonFunctioningForm.to_python(form_values)
        except validators.Invalid, e:
            errors = e
        else:
            form_values['site_title'] = self.site_title
            form_values['code'] = code
            if template == 'report_sanitation':
                form_values['url'] = '%s/sanitations/view' % self.absolute_url()
            else:
                form_values['url'] = '%s/waterpoints/view' % self.absolute_url()
            self.send_notification(template, **form_values)
            success = True
        return errors, success

    security.declarePrivate('send_notification')
    def send_notification(self, template, **kwargs):
        """ """
        portal = self.getSite()
        email_tool = portal.getEmailTool()
    
        message = {}
        for k, v in kwargs.items():
            message[k] = v
    
        if kwargs['user']:
            auth_tool = self.getAuthenticationTool()
            u = auth_tool.getUser(kwargs['user'])
            if u and u.email:
                message['user_email'] = u.email
            else:
                message['user_email'] = ''
    
            full_name = auth_tool.getUserFullName(u)
            if full_name:
                message['user_name'] = full_name
            else:
                message['user_name'] = ''
    
        template = self._get_template(template)
        mail_data = template(**message)
        email_tool.sendEmail(p_content = mail_data['body_text'],
                             p_to = email_tool.administrator_email,
                             p_from = email_tool._get_from_address(),
                             p_subject = mail_data['subject'])


    
    _index_html = NaayaPageTemplateFile('zpt/country', globals(), 'ws_facilities_country')
    _pending =  NaayaPageTemplateFile('zpt/pending', globals(), 'ws_facilities_pending')
    finish = NaayaPageTemplateFile('zpt/finish', globals(), 'ws_finish')
    template_facilities_html = NaayaPageTemplateFile('zpt/template_facilities', globals(), 'ws_facilities_template')

InitializeClass(WSPortalFacilities)
