import datetime
from sqlalchemy.sql import desc, asc
from sqlalchemy.sql.expression import or_
from formencode import validators
import zLOG

from Products.NaayaCore.EmailTool.EmailPageTemplate import EmailPageTemplateFile
from Products.NaayaCore.FormsTool.NaayaTemplate import NaayaPageTemplateFile
from forms import CommentForm

email_templates = {
    'notification': EmailPageTemplateFile('emailpt/notification.zpt', globals()),
    'moderation': EmailPageTemplateFile('emailpt/moderation.zpt', globals()),
}

class Comment(object):
    """ Generic comments tool class.
        Defines methods for comments management.
    """
    
    def __init__(self, model, code, session, context):
        self.model = model
        self.code = code
        self.session = session
        self.context = context
        self.errors = None
        self.success = False

    def get_comments(self):
        comments = self.session.query(self.model).\
                    filter(self.model.code == self.code).\
                    filter(self.model.is_public == True).\
                    filter(or_(self.model.is_removed==False, self.model.is_removed==None)).all()
        for comment in comments:
            self.get_comment_userinfo(comment)
        return comments

    def get_all_comments(self, approved, sort_on, sort_order):
        query =  self.session.query(self.model).\
                            filter(self.model.is_public==approved).\
                            filter(or_(self.model.is_removed==False, self.model.is_removed==None))
        if sort_on:
            column = getattr(self.model, sort_on)
            if sort_order:
                query = query.order_by(asc(column))
            else:
                query = query.order_by(desc(column))
        return query

    def count_unapproved_by_model(self, model):
        return  self.session.query(model).\
                            filter(model.is_public==False).\
                            filter(or_(model.is_removed==False, model.is_removed==None)).count()

    def get_comment(self, id):
        return self.session.query(self.model).\
                    filter(self.model.id == id).\
                    filter(or_(self.model.is_removed==False, self.model.is_removed==None)).one()

    def get_comment_userinfo(self, comment):
        """
        Get a dictionary that pulls together information about the poster
        safely for both authenticated and non-authenticated comments.
     
        This dict will have ``name`` and ``email`` fields.
        """
        comment.userinfo = {
            "name"  : comment.user_name,
            "email" : comment.user_email,
            }
        if comment.user:
            auth_tool = self.context.getAuthenticationTool()
            u = auth_tool.getUser(comment.user)
            if u and u.email:
                comment.userinfo["email"] = u.email

            full_name = auth_tool.getUserFullName(u)
            if full_name:
                comment.userinfo["name"] = full_name
        return comment.userinfo

    def delete_comments(self, ids):
        self.session.query(self.model).\
            filter(self.model.id.in_(ids)).\
            filter(self.model.is_removed==True).\
            delete(synchronize_session=False)

    def trash_comments(self, ids):
        self.session.query(self.model).filter(self.model.id.in_(ids)).update({self.model.is_removed: True}, synchronize_session=False)

    def moderate_comments(self, ids, approve):
        self.session.query(self.model).filter(self.model.id.in_(ids)).update({self.model.is_public: approve}, synchronize_session=False)

    def add_comment(self, REQUEST, **kwargs):
        form_values = dict(REQUEST.form)
        #get the user credentials
        if self.context.isAnonymousUser():
            form_values['user'] = ''
        else:
            form_values['user'] = REQUEST.AUTHENTICATED_USER.getUserName()

        #validate form fields
        try:
            form = CommentForm.to_python(form_values)
        except validators.Invalid, e:
            self.errors = e
        else:
            form_values['code'] = self.code
            form_values['ip_address'] = REQUEST.REMOTE_ADDR
            if int(self.context.moderation):
                form_values['is_public'] = False
            else:
                form_values['is_public'] = True #check permission

            form_values['submit_date'] = datetime.datetime.now()
            if form_values['user']:
                auth_tool = self.context.getAuthenticationTool()
                u = auth_tool.getUser(form_values['user'])
                if u and u.email:
                    form_values['user_email'] = u.email
                else:
                    form_values['user_email'] = ''
        
                full_name = auth_tool.getUserFullName(u)
                if full_name:
                    form_values['user_name'] = full_name
                else:
                    form_values['user_name'] = ''

            del form_values['id']
            comment = self.model(**form_values)
            self.session.add(comment)

            #send notification
            form_values['page_url'] = REQUEST.HTTP_REFERER
            form_values['site_title'] = self.context.site_title
            form_values['approve_url'] = '%s/comments' % self.context.getSitePath()
            if int(self.context.moderation):
                self.success = True
                self._send_notification(template='moderation', **form_values)
            else:
                self._send_notification(template='notification', **form_values)

    def get_rating(self):
        comments = self.get_comments()
        num_comments = len(comments)
        if num_comments:
            s = sum([c.rating for c in comments if c.rating])
            average = float(s) / num_comments
            ret = int(average + 0.5)
            return ret

    def edit_comment(self, id_comment, **kwargs):
        comment = self.session.query(self.model).filter_by(id = id).one()
        for k, v in kwargs.iteritems():
            setattr(comment, k, v)
        self.session.add(comment)

    def list_comments(self):
        options = {
            'comments': self.get_comments(),
            'errors': self.errors,
            'code': self.code,
            'success': self.success,
        }
        return self._list_comments.__of__(self.context)(**options)

    _list_comments = NaayaPageTemplateFile('zpt/list_comments', globals(), 'comments_list_comments')

    # Notifications
    def _get_template(self, context, name):
        template = context._getOb('emailpt_%s' % name, None)
        if template is not None:
            return template.render_email

        template = email_templates.get(name, None)
        if template is not None:
            return template.render_email

        raise ValueError('template for %r not found' % name)

    def _send_notification(self, template, **kwargs):
        """ """
        portal = self.context.getSite()
        email_tool = portal.getEmailTool()

        message = {'url': self.context.absolute_url()}
        for k, v in kwargs.items():
            message[k] = v

        if kwargs['user']:
            auth_tool = portal.getAuthenticationTool()
            u = auth_tool.getUser(kwargs['user'])
            if u and u.email:
                message['user_email'] = u.email
            else:
                message['user_email'] = ''
        
            full_name = auth_tool.getUserFullName(u)
            if full_name:
                message['user_name'] = full_name
            else:
                message['user_name'] = ''

        template = self._get_template(self.context, template)
        mail_data = template(**message)
        email_tool.sendEmail(p_content = mail_data['body_text'],
                             p_to = email_tool.administrator_email,
                             p_from = email_tool._get_from_address(),
                             p_subject = mail_data['subject'])