'''
Created on Apr 2, 2010

@author: cristiroma
'''

#Python imports
import simplejson as json

#Zope imports
from OFS.Folder import Folder
from AccessControl.SecurityInfo import ClassSecurityInfo
from Globals import InitializeClass

from ws.common.sql.queries import Program as program_queries
from ws.common import gis
from ws.common.sql.queries import Access as access_query
from ws.common.sql.mappings.Subdivision import Subdivision


class Customizations(Folder):
    """
    This class is constructed - and generated data is conform to GIS data
    from WATSAN_INSTANCE/var/watstan-instance/gis/COUNTRY_CODE/*.map files
    (level1.map, level2.map, COUNTRY_CODE.map, etc.).

    It is the server side component that handles requests from ZPT page
    when constructing pages that have a map component.
    Each UMap map displayes various GIS layers. These methods returns the required
    layers, as necessary.
    """
    security = ClassSecurityInfo()


    def __init__(self, id):
        self.id = id

    security.declarePublic('gis_get_mapoptions')
    def gis_get_mapoptions(self, as_json=True):
        """ """
        return """{
            projection: new OpenLayers.Projection("EPSG:900913"),
            displayProjection: new OpenLayers.Projection("EPSG:4326"),
            units: "m",
            numZoomLevels: 21,
            maxResolution: 156543.0339,
            maxExtent: new OpenLayers.Bounds(-20037508, -20037508, 20037508, 20037508.34)
            }"""

    def gis_get_min_mapoptions(self, as_json=True):
        """ """
        return """{
            units: "m",
            numZoomLevels: 21
        }"""


######## Shortcuts #############################################################
    def mapfile(self):
        """
        Retrieve the absolute path to the GIS mapfile for this portal.
        Usually the mapfile is located in /var of the instance to be editable.
        """
        return '%s/%s.map' % (self.getSite().gis_path, self.getSite().country_code)


    def layer_country_wp(self):
        """
        Layer with all water points from country.
        """
        _trans = self.getPortalTranslations()
        layer = gis.MapLayerMapserver(_trans('Water points'), layout={'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'WaterPoints' }
        return layer.to_umap()


    def layer_locality_wp(self, id):
        """
            Layer with all water points from specific locality.
            Parameters:
                id
                    Id of the locality
        """
        _trans = self.getPortalTranslations()
        layer = gis.MapLayerMapserver(_trans('Water points'), layout={'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'WaterPoints' }
        return layer.to_umap()


    def layer_country_sf(self):
        """
            Layer with all sanitation facilities from country
        """
        _trans = self.getPortalTranslations()
        layer = gis.MapLayerMapserver(_trans('Sanitation facilities'), layout={'singleTile': 'true'})
        layer.cfg = { 'MAP' : self.mapfile(), 'layers' : 'country_sf' }
        return layer.to_umap()


    def layer_country_localities(self):
        """
            Layer with all localities from country
        """
        _trans = self.getPortalTranslations()
        layer = gis.MapLayerMapserver(_trans('Localities'), layout={'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'locality' }
        return layer.to_umap()


    def layer_country_subdivision(self, name=None, level=1, filter=None, raw=False, visibility=False):
        """
            Retrieve the layer with all regions drawn on the specified level.
            Also can draw only specific regions based on filter parameter.
            Parameters:
                name
                    Text/Name of the layer as will appear on UI. If not specified, this method will try to build one.
                level
                    Level to be drawn (from 1 - 5). Some countries may have only up to level 3 data. Default 1.
                filter
                    Do not draw all subdivisions, but only those with ID specified on this filter (array)
                raw
                    If true returns raw layer object instead or umap specific construction (layer.to_umap()). Default False.
        """
        _trans = self.getPortalTranslations()
        if not name:
            lname = self.labels.get('s%s' % level, '')
            name = lname.get('name').capitalize()
        layer = gis.MapLayerMapserver(_trans(name), layout={'visibility' : str(visibility).lower(), 'singleTile': 'true'})
        layer_name = 'adreglevel%s' % level
        if filter:
            str_filter = layer.ut_regex_or_filter(filter)
            layer.add_filter('adrcodes', str_filter)
        else:
            layer.add_filter('adrcodes', '.*')

        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : layer_name }
        if raw:
            return layer
        return layer.to_umap()


    def custom_overlays(self, setting):
        """
        TODO: To be described & fully implemented.
        Allows extension of pre-defined layers with additional ones defined directly from the CMS UI,
        without requiring intervention on the code. Layers defined below are for now considered 'hardcoded'
        and cannot be modified. This means that using this method we could edit the MAPFILE online from CMS (http://server/portal/umap/configure)
        and add new layers (autodetected by framework using mapserver API). See umap.cofigure for more information.
        Return
            Additional overlays. For now an empty array.
        """
        ret = []
        _trans = self.getPortalTranslations()
        custom_overlays = self.get_setting(setting)
        for overlay in custom_overlays:
            layer = gis.MapLayerMapserver(_trans(overlay), layout={'singleTile': 'true'})
            layer.cfg = { 'MAP' : '%s/custom.map' % self.getSite().gis_path, 'layers' : overlay }
            ret.append(layer.to_umap())
        return ret

################################################################################


    security.declarePublic('gis_get_backgrounds')
    def gis_get_backgrounds(self, as_json=True, use_mercator=False):
        """
        This method returns the backgrounds used by the map component throughout entire pages.
        For now all pages have the same backgrounds (3 Google: Normal/Satellite/Hybrid).
        This could be extended to return additional or less layers for each map.
        Parameters:
            as_json
                Return JSON object directly. This is usually useful when called directly from template (ZPT).
            use_mercator
                Define the layers in mercator coordinates.
                This is usefuly when we edit the map (map has an editable vector layer).
                See OpenLayers framework for more information.
        Return
            The layer as python dictionary object (umap compatible) or JSON (umap compatible).
            Umap compatible means that data returned can be parsed by the UMap JS framework.
        """
        ret = []
        google = gis.MapLayerGoogle(cfg={'sphericalMercator' : use_mercator}, layout={'visibility' : True})
        google_sat = gis.MapLayerGoogleSatellite(cfg={'sphericalMercator' : use_mercator}, layout={'visibility' : True})
        google_hyb = gis.MapLayerGoogleHybrid(cfg={'sphericalMercator' : use_mercator}, layout={'visibility' : True})

        ret.append(google.to_umap())
        ret.append(google_sat.to_umap())
        ret.append(google_hyb.to_umap())
        if as_json:
            return json.dumps(ret)
        return ret


    def get_locality_overlays(self, id=None, as_json=True):
        """
        Create the overlays that are built into the locality page map.
        Parameters:
            id
                ID of the locality
            as_json
                Return JSON object directly. This is usually useful when called directly from template (ZPT).

        Return
            Array of umap compatible overlays.
        """
        ret = []
        _trans = self.getPortalTranslations()
        sn_map = '%s/%s.map' % (self.getSite().gis_path, self.getSite().country_code)

        if id:
            all_wp = gis.MapLayerMapserver(_trans('All water points'), layout={'visibility' : True, 'singleTile': 'true'})
            all_wp.cfg = { 'MAP' : '%s' % sn_map, 'layers' : 'subdivision_wp' }
            all_wp.add_filter('filter', "upisadlocode='%s'" % (id))
            ret.append(all_wp.to_umap())

            all_sf = gis.MapLayerMapserver(_trans('All sanitation facilities'), layout={'visibility' : True, 'singleTile': 'true'})
            all_sf.cfg = { 'MAP' : '%s' % sn_map, 'layers' : 'subdivision_sf' }
            all_sf.add_filter('filter', "upisadlocode='%s'" % (id))
            ret.append(all_sf.to_umap())

            public_taps = gis.MapLayerMapserver(_trans('Public taps'), layout={'visibility' : False, 'singleTile': 'true'})
            public_taps.cfg = { 'MAP' : '%s' % sn_map, 'layers' : 'subdivision_wp' }
            public_taps.add_filter('filter', "upisloctype='01' and upisadlocode='%s'" % (id))
            ret.append(public_taps.to_umap())

            handpumps = gis.MapLayerMapserver(_trans('Hand pumps'), layout={'visibility' : False, 'singleTile': 'true'})
            handpumps.cfg = { 'MAP' : '%s' % sn_map, 'layers' : 'subdivision_wp' }
            handpumps.add_filter('filter', "upwptype='02' and upisadlocode='%s'" % (id))
            ret.append(handpumps.to_umap())

            modernwells = gis.MapLayerMapserver(_trans('Modern wells'), layout={'visibility' : False, 'singleTile': 'true'})
            modernwells.cfg = { 'MAP' : '%s' % sn_map, 'layers' : 'subdivision_wp' }
            modernwells.add_filter('filter', "upwptype='04' and upisadlocode='%s'" % (id))
            ret.append(modernwells.to_umap())
        ret.extend(self.custom_overlays('locality.umap.overlays'))
        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_subdivision_overlays(self, id, level, as_json=True):
        """
        Create the overlays that are built into the subdivision page map.
        Parameters:
            id
                ID of the subdivision
            as_json
            Return JSON object directly. This is usually useful when called directly from template (ZPT).
        Return
            Array of umap compatible overlays.
        """
        ret = []
        _trans = self.getPortalTranslations()

        if id:
            session = self.get_db_session()

            subdiv = access_query.get_subdivision(session, id)
            subdiv_layer = self.layer_country_subdivision(subdiv.name, level, [subdiv.code], raw=True)
            subdiv_layer.layout['visibility'] = True
            ret.append(subdiv_layer.to_umap())

            max_level = access_query.get_subdivision_max(session)
            if level < max_level:
                children = access_query.get_subdivision_list(session, id)
                subdiv_layer = self.layer_country_subdivision(_trans('Subdivisions'), level + 1, [subdiv.code for subdiv in children], raw=True)
                subdiv_layer.layout['visibility'] = False
                ret.append(subdiv_layer.to_umap())

            localities = gis.MapLayerMapserver(_trans('Localities'), layout={'visibility' : False, 'singleTile': 'true'})
            localities.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_localities' }
            localities.add_filter('filter', "level%s='%s'" % (level, id))
            ret.append(localities.to_umap())

            all_wp = gis.MapLayerMapserver(_trans('All water points'), layout={'visibility' : False, 'singleTile': 'true'})
            all_wp.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_wp' }
            all_wp.add_filter('filter', "upisadr%scode='%s'" % (level, id))
            ret.append(all_wp.to_umap())

            all_sf = gis.MapLayerMapserver(_trans('All sanitation facilities'), layout={'visibility' : False, 'singleTile': 'true'})
            all_sf.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_sf' }
            all_sf.add_filter('filter', "upisadr%scode='%s'" % (level, id))
            ret.append(all_sf.to_umap())

            public_taps = gis.MapLayerMapserver(_trans('Public taps'), layout={'visibility' : False, 'singleTile': 'true'})
            public_taps.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_wp' }
            public_taps.add_filter('filter', "upisloctype='01' and upisadr%scode='%s'" % (level, id))
            ret.append(public_taps.to_umap())

            handpumps = gis.MapLayerMapserver(_trans('Hand pumps'), layout={'visibility' : False, 'singleTile': 'true'})
            handpumps.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_wp' }
            handpumps.add_filter('filter', "upwptype='02' and upisadr%scode='%s'" % (level, id))
            ret.append(handpumps.to_umap())

            modernwells = gis.MapLayerMapserver(_trans('Modern wells'), layout={'visibility' : False, 'singleTile': 'true'})
            modernwells.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_wp' }
            modernwells.add_filter('filter', "upwptype='04' and upisadr%scode='%s'" % (level, id))
            ret.append(modernwells.to_umap())
        else:
            ret.append(self.layer_country_subdivision(None, level))


        ret.extend(self.custom_overlays('subdivision.umap.overlays'))
        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_program_overlays(self, id, as_json=True):
        """
        Create the overlays that are built into the program page map.
        Parameters:
            id
                ID of the program
            as_json
            Return JSON object directly. This is usually useful when called directly from template (ZPT).
        Return
            Array of umap compatible overlays.
        """
        ret = []
        _trans = self.getPortalTranslations()

        subdiv = gis.MapLayerMapserver(_trans('Covered regions'), layout={'visibility' : True, 'singleTile': 'true'})
        subdivisions = program_queries.get_program_subdivisions_gis(self.get_db_session(), id)
        str_filter = subdiv.ut_OR_filter('ID', subdivisions, 'code')
        subdiv.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'project_subdivisions' }
        subdiv.add_filter('code', str_filter)
        ret.append(subdiv.to_umap())


        subdiv = gis.MapLayerMapserver(_trans('Covered localities'), layout={'visibility' : True, 'singleTile': 'true'})
        subdiv.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'project_localities' }
        subdiv.add_filter('project', id)
        ret.append(subdiv.to_umap())

        ret.extend(self.custom_overlays('locality.umap.overlays'))
        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_programs_map_overlays(self, programs, colors=['0 0 255', '0 0 0', '255 0 0'], as_json=True):
        """
        Create the overlays that are built into the program page map.
        Parameters:
            `programs`
                List of maximum 3 programs.
            `as_json`
                Return JSON object directly. This is usually useful when called directly from template (ZPT).
        Return
            Array of umap compatible overlays.
        """
        ret = []
        i = 0
        _trans = self.getPortalTranslations()
        for program in programs:
            color = colors[i]
            subdiv_title = '#%s - %s' % ((i+1), _trans('Regions'))
            subdiv = gis.MapLayerMapserver(subdiv_title, layout={'visibility' : True, 'singleTile': 'true'})
            subdivisions = program_queries.get_program_subdivisions_gis(self.get_db_session(), program.adopcode)
            str_filter = subdiv.ut_OR_filter('ID', subdivisions, 'code')
            subdiv.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'project_subdivisions' }
            subdiv.add_filter('code', str_filter)

            subdiv.add_filter('map.layer[project_subdivisions].class[0].style[0]', 'SYMBOL diagonal COLOR %s SIZE 3 OUTLINECOLOR 0 0 0' % color)
            ret.append(subdiv.to_umap())

            loc_title = '#%s - %s' % ((i+1), _trans('Localities'))
            loc = gis.MapLayerMapserver(loc_title, layout={'visibility' : True, 'singleTile': 'true'})
            loc.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'project_localities' }
            loc.add_filter('project', program.adopcode)

            loc.add_filter('map.layer[project_localities].class[0].style[0]', 'SYMBOL circle SIZE 6 OUTLINECOLOR 0 0 0 COLOR %s' % color)
            ret.append(loc.to_umap())
            i += 1

        if as_json:
            ret = json.dumps(ret)
        return ret

    def get_waterpoints_map_overlays(self, id_subdivision=None, id_locality=None, id_functionality=None, as_json=True):
        """
        Create the overlays that are built into the waterpoints map page.
        Parameters:
            id_locality
                Filter by view_gis_wp.upisadlocode column from view
            id_functionality
                Filter by view_gis_wp.upisobj
        """
        ret = []
        _trans = self.getPortalTranslations()
        filter = "1=1"
        if id_locality:
            filter = filter + " and upisadlocode='%s'" % id_locality
        elif id_subdivision:
            session = self.get_db_session()
            subdivision = session.query(Subdivision).filter_by(code=id_subdivision).one()
            filter = filter + " and upisadr%scode='%s'" % (subdivision.level, id_subdivision)
        if id_functionality:
            filter = filter + " and upisobj='%s'" % id_functionality

        subdiv = gis.MapLayerMapserver(_trans('Search engine results'), layout={'visibility' : True, 'displayInLayerSwitcher' : False, 'singleTile': 'true'})
        subdiv.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_wp' }
        subdiv.add_filter('filter', filter)
        ret.append(subdiv.to_umap())

        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_piped_water_schemes_map_overlays(self, id_subdivision=None, id_locality=None, id_functionality=None, as_json=True):
        """
        Create the overlays that are built into the piped water schemes map page.
        Parameters:
            id_locality
                Filter by view_gis_pws.upisadlocode column from view
            id_functionality
                Filter by view_gis_pws.uppwsfunc
        """
        ret = []
        _trans = self.getPortalTranslations()
        filter = "1=1"
        if id_locality:
            filter = filter + " and upisadlocode='%s'" % id_locality
        elif id_subdivision:
            session = self.get_db_session()
            subdivision = session.query(Subdivision).filter_by(code=id_subdivision).one()
            filter = filter + " and upisadr%scode='%s'" % (subdivision.level, id_subdivision)
        if id_functionality:
            filter = filter + " and uppwsfunc='%s'" % id_functionality

        subdiv = gis.MapLayerMapserver(_trans('Search engine results'), layout={'visibility' : True, 'displayInLayerSwitcher' : False, 'singleTile': 'true'})
        subdiv.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_pws' }
        subdiv.add_filter('filter', filter)
        ret.append(subdiv.to_umap())

        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_piped_water_schemes_index_overlays(self, id, as_json=True):
        """
        Create the overlays that are built into the piped water schemes map page.
        Parameters:
            `id`
                ID of the water scheme
            id_functionality
                Filter by view_gis_pws.uppwsfunc
        """
        ret = []
        _trans = self.getPortalTranslations()

        wp = gis.MapLayerMapserver(_trans('Water points'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        wp.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'pws_waterpoints' }
        wp.add_filter('filter', "pwsisupiscode='%s'" % id)
        ret.append(wp.to_umap())

        loc = gis.MapLayerMapserver(_trans('Served localities'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        loc.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'pws_localities' }
        loc.add_filter('filter', "pwsisupiscode='%s'" % id)
        ret.append(loc.to_umap())

        reservoirs = gis.MapLayerMapserver(_trans('Water reservoirs'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        reservoirs.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'pws_reservoirs' }
        reservoirs.add_filter('filter', "stisupiscode='%s'" % id)
        ret.append(reservoirs.to_umap())

        pipes = gis.MapLayerMapserver(_trans('Pipe layout'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        pipes.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'pws_piping' }
        pipes.add_filter('pws_id', id)
        ret.append(pipes.to_umap())


        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_sanitations_map_overlays(self, id_subdivision=None, id_locality=None, id_functionality=None, as_json=True):
        """
        Create the overlays that are built into the sanitation facilities map page.
        Parameters:
            id_locality
                Filter by view_gis_sf.upisadlocode column from view
            id_functionality
                Filter by view_gis_sf.upisobj
        """
        ret = []
        _trans = self.getPortalTranslations()

        filter = "1=1"
        if id_locality:
            filter = filter + " and upisadlocode='%s'" % id_locality
        elif id_subdivision:
            session = self.get_db_session()
            subdivision = session.query(Subdivision).filter_by(code=id_subdivision).one()
            filter = filter + " and upisadr%scode='%s'" % (subdivision.level, id_subdivision)
        if id_functionality:
            filter = filter + " and upisobj='%s'" % id_functionality

        subdiv = gis.MapLayerMapserver(_trans('Search engine results'), layout={'visibility' : True, 'displayInLayerSwitcher' : False, 'singleTile': 'true'})
        subdiv.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'subdivision_sf' }
        subdiv.add_filter('filter', filter)
        ret.append(subdiv.to_umap())

        if as_json:
            ret = json.dumps(ret)
        return ret


    def get_iwrm_index_overlays(self, as_json=True):
        """
        Create the overlays that are built into the IWRM map from index page.
        """
        ret = []
        _trans = self.getPortalTranslations()

        layer = gis.MapLayerMapserver(_trans('Monitoring stations'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'iwrm_mostation' }
        ret.append(layer.to_umap())

        layer = gis.MapLayerMapserver(_trans('Underground water sources'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'iwrm_uws' }
        ret.append(layer.to_umap())

        # Shapefile based layers
        layer = gis.MapLayerMapserver(_trans('Basins'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'iwrm_basin' }
        ret.append(layer.to_umap())

        layer = gis.MapLayerMapserver(_trans('Aquifers'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'iwrm_aquifer' }
        ret.append(layer.to_umap())

        layer = gis.MapLayerMapserver(_trans('Rivers'), layout={'visibility' : True, 'displayInLayerSwitcher' : True, 'singleTile': 'true'})
        layer.cfg = { 'MAP' : '%s' % self.mapfile(), 'layers' : 'iwrm_river' }
        ret.append(layer.to_umap())

        if as_json:
            ret = json.dumps(ret)
        return ret


InitializeClass(Customizations)