# -*- coding: utf-8 -*-
from urlparse import urlparse
from httplib import HTTPConnection
from ws.common.exceptions import ValidationException
from types import ListType
from ws.common.sql.mappings.WatsanCfg import WatsanCfg
import zipfile
import os

DICT_COUNTRIES_EN = {
    'AO' : 'Angola',
    'BF' : 'Burkina Faso',
    'BI' : 'Burundi',
    'BJ' : 'Benin',
    'BW' : 'Botswana',
    'CD' : 'Congo, Democratic Republic of',
    'CF' : 'Central African Republic',
    'CG' : 'Congo',
    'CI' : 'Ivory Coast',
    'CM' : 'Cameroun',
    'CV' : 'Cape Verde',
    'DJ' : 'Djibouti, Republic of',
    'EH' : 'Western Sahara',
    'ER' : 'Eritrea',
    'ET' : 'Ethiopia',
    'GA' : 'Gabon',
    'GH' : 'Ghana',
    'GM' : 'Gambia',
    'GN' : 'Guinea',
    'GQ' : 'Equatorial Guinea',
    'GW' : 'Guinea-Bissau',
    'KE' : 'Kenya',
    'KM' : 'Comoros',
    'LR' : 'Liberia',
    'LS' : 'Lesotho',
    'MG' : 'Madagascar',
    'ML' : 'Mali',
    'MR' : 'Mauritania',
    'MU' : 'Mauritius',
    'MW' : 'Malawi',
    'MZ' : 'Mozambique',
    'NA' : 'Namibia',
    'NE' : 'Niger',
    'NG' : 'Nigeria',
    'RE' : 'Reunion',
    'RW' : 'Rwanda',
    'SC' : 'Seychelles',
    'SD' : 'Sudan',
    'SH' : 'Saint Helena',
    'SL' : 'Sierra Leone',
    'SN' : 'Senegal',
    'SO' : 'Somalia',
    'ST' : 'São Tomé and Príncipe',
    'SZ' : 'Swaziland',
    'TD' : 'Chad',
    'TG' : 'Togo',
    'TZ' : 'Tanzania',
    'UG' : 'Uganda',
    'YT' : 'Mayotte',
    'ZA' : 'South Africa',
    'ZM' : 'Zambia',
    'ZW' : 'Zimbabwe'
}

def valid_url(url):
    """
    Verify if an URL exists
    Parameters:
        `url`
            URL to verify
    Return:
        True if URL is valid (HTTP 200), False otherwise
    """
    ret = False
    try:
        up = urlparse(url)
        conn = HTTPConnection(up[1])
        conn.request('HEAD', up[2])
        answ = conn.getresponse()
        ret = answ.status == 200
    finally:
        try:
            conn.close()
        except:
            pass
    return ret


def country_list(lang='en'):
    """
    Retrieve the hard-coded list of countries
    Return:
        List of country as dictionary {code:name}
    """
    return DICT_COUNTRIES_EN


def decode_country_name(country_code, lang='en'):
    """
    Try to guess country code from its ISO 3166-2 code
    Parameters:
    `country_code`
        Country code, example: BJ
    `lang`
        Language, example en
    Return:
        country_code if no decoding found, otherwise the country name in 
    specified language. Currently implemented languages are: en.
    """
    if not country_code:
        return None

    cc = country_code.upper()
    if lang == 'en' and cc in DICT_COUNTRIES_EN.keys():
        return DICT_COUNTRIES_EN[cc]
    return country_code


def unzip_file(file, target_dir, override=False, zLOG=None):
    """
    Unzip the content of the given file to specified directory
    Parameters:
        `file` 
            file object containing the zip archive
        `target_dir`
            Directory where file will be unzipped
        `override`
            If True, files are silently overriden. If False, ValidationException will occur
    Return:
        Nothing
    """
    zf = zipfile.ZipFile(file)
    if not override:
        for i, name in enumerate(zf.namelist()):
            final_path = os.path.join(target_dir, name)
            if os.path.exists(final_path):
                raise ValidationException('unzip_file(): File %s already exists and override is not allowed. Please remove it first.' % name)

    zLOG.LOG(__name__, zLOG.INFO, 'unzip_file(): Unzipping %s: ' % file.name)
    for i, name in enumerate(zf.namelist()):
        if zLOG:
            zLOG.LOG(__name__, zLOG.INFO, '    * Deflating #%s: %s' % (i, name))
        ffile = open(os.path.join(target_dir, name), 'wb')
        ffile.write(zf.read(name))
        ffile.flush()
        ffile.close()

def url_get(REQUEST, ommit=[]):
    ret = []
    for param in REQUEST.form.keys():
        if not param in ommit:
            val = REQUEST.form[param]
            if isinstance(val, ListType):
                for ival in val:
                    ret.append('%s=%s' % (param, ival))
            else:
                ret.append('%s=%s' % (param, val))
    return '&'.join(ret)


def req_field(form, name, def_val = None):
    if form.has_key(name):
        return form[name]
    return def_val


def dummy_insert(session):
    # Do a dummy commit to force SQLA to COMMIT underlying transaction and not ROLLBACK it.
    # Seems that SQL statements are dropped if it doesn't detect a 
    # modification trougth session.merge/update/add etc.
    ob = session.query(WatsanCfg).filter_by(id='dummy').one()
    ob.numeric_value = 10
    session.merge(ob)
