import urllib, sys, traceback, string
import simplejson

from ws.common.sql.mappings import Lexicon as lex_models


class SynchronisationReport(object):
    """
    This class contains the summary report for an update session. It reports the results in HTML format.
    """

    def __init__(self):
        self.unknown_dictionaries = []
        self.new_values = {}
        self.messages = []

    def add_unknown_dic(self, name):
        self.unknown_dictionaries.append(name)

    def add_new_value(self, dic_name, ob):
        if not dic_name in self.new_values.keys():
            self.new_values[dic_name] = []
        self.new_values[dic_name].append(ob)

    def add_message(self, message):
        self.messages.append(message)

    def clear(self):
        self.unknown_dictionaries = []
        self.new_values = {}
        self.messages = []

    def html(self):
        ret = []
        ret.append("<h2>Synchronisation report</h2>")
        ret.append("<h3>Messages</h3>")
        ret.append("<pre>")
        for row in self.messages:
            ret.append("%s" % row)
        ret.append("</pre>")
        if self.unknown_dictionaries:
            ret.append("<h3>Unknown dictionaries</h3>")
            ret.append("<ul>")
            for row in self.unknown_dictionaries:
                ret.append("<li>%s</li>" % row)
            ret.append("</ul>")

        if self.new_values:
            ret.append("<h3>New imported values</h3>")
            for key in self.new_values.keys():
                ret.append('<table class="datatable"><caption>%s (%s)</caption><tr><th>Key</th><th>Value</th></tr>' % (lex_models.lexicon_list[key]['table_name'], key))
                for ob in self.new_values[key]:
                    ret.append("<tr><td>%s</td><td>%s</td></tr>" % (ob.code, ob.name))
                ret.append("</table>")
        return '\n'.join(ret)


class DictionarySync(object):
    """
    This class handles the incoming synchronization data from the WatSan Website. It takes the data in JSON format and synchronises the local database with new values.
    Note that values from local database are never deleted.
    If codes are added in local installation, updates from remote portal will be ignored. This happens when someone manually enters data directly into the database.
    It creates and maintains internally an report that can be examied at the end of the synchronisation.

    >>> ds = DictionarySync(session, 'http://watsan2.eaudeweb.ro/dictionary_sync?format=json')
    >>> ds.sync_dictionaries()
    >>> report = ds.report
    >>> report.new_values
    """

    def __init__(self, session, url_wwebsite):
        """
        Constructor
        Parameters
            `session`
                Database session for local portal database
            `url_wwebsite`
                URL to the WatSan Website entry point where data is harvested from
        """
        self.session = session
        self.url_wwebsite = url_wwebsite
        self.report = SynchronisationReport()


    def sync_dictonaries(self):
        """
        Retrieve the JSON object with dictionary data and update the local portal with appropriate values
        """
        self.report.clear()
        ret = False
        file = None
        f = None
        json_data = {}

        self.report.add_message('[INFO]Loading new dictionary data from: %s' % self.url_wwebsite)
        something_updated = False

        try:
            file = urllib.urlopen(self.url_wwebsite)
            f = file.read()
            file.close()
        except:
            type, val, tb = sys.exc_info()
            message = string.join(traceback.format_exception(type, val, tb), '')
            self.report.add_message('[ERROR]Error while loading dictionary data: %s' % message)
        else:
            try:
                self.report.add_message('[INFO]Parsing dictionary data ...')
                json_data = simplejson.loads(f)
            except:
                type, val, tb = sys.exc_info()
                message = string.join(traceback.format_exception(type, val, tb), '')
                self.report.add_message('[ERROR]Error while parsing dictionary data: %s' % message)
            else:
                for key in json_data:
                    if key in lex_models.lexicon_list.keys():
                        values = json_data[key]
                        if self.sync_dictionary(key, values):
                            something_updated = True
                    else:
                        self.report.add_message('[WARNING] The "%s" dictionary is not present in local installation. You must update the portal to latest version to take advantage of new dictionaries.' % key )
        if not something_updated:
            self.report.add_message('[INFO]Everything was up-to-date')
        else:
            self.report.add_message('[INFO]Some dictionaries were updated')
        self.report.add_message('[INFO]Finished')
        return ret

    def sync_dictionary(self, klass_name, values={}):
        """
        Synchronise a single dictionary. Internally used
        Parameters:
            `klass_name`
                Name of the Python class that is being synchronized (must be one of keys from lex_models.lexicon_list - if not warning is raised)
            `values`
                Remote dictionary content that is checked agains local version. value is {'01' : 'value 1', '02' : 'value 2' etc. }
        Return
            True if dictionary was updated or False if it was up-to-date
        """
        klass = lex_models.lexicon_list[klass_name]['table_class']
        dict_description = lex_models.lexicon_list[klass_name]['table_name']
        updated = False
        for key in values.keys():
            check = self.session.query(klass).filter_by(code=key).count()
            if not check:
                updated = True
                dict_description = lex_models.lexicon_list[klass_name]
                value = values[key]
                ob = klass(code=key, name=value, name2=value)
                # Handle special case of LexWP007 where we have the EWP (Equivalent Water Point) value. Default to 1.
                if klass_name == 'LexWP007':
                    ob.ewp = 1
                self.session.add(ob)
                self.report.add_message('[NEW]Inserting new value "%s" - "%s" into dictionary "%s" (%s)' % (key, value, klass_name, dict_description))
                self.report.add_new_value(klass_name, ob)
        return updated


class DictionarySyncEndpoint(object):
    """ This endpoint generates the necessary data and provides it to the clients.
    """

    def __init__(self, session):
        """
        Constructor
        Parameters
            `session`
                Database connection to Website database
        """
        self.session = session

    def as_json(self):
        """
        Output data as JSON-encoded structure.
        Format of the output is:
        {
            'table_class' : {
                'key' : 'value',
            }
        }
        """
        json_data = {}

        for key in lex_models.lexicon_list.keys():
            lexicon = lex_models.lexicon_list[key]
            klass = lexicon['table_class']
            dict_data = self.session.query(klass).order_by('code').all()
            json_data[key] = {}
            for ob in dict_data:
                json_data[key][ob.code] = ob.name

        ret = simplejson.dumps(json_data)
        return ret