'''
Created on Jul 29, 2010

@author: cristiroma
'''
import simplejson, urllib
import sys, traceback, string
from datetime import datetime
from ws.common.sql.mappings import Catalogs as cat_models
from sqlalchemy.sql.expression import and_


class SynchronisationReport(object):
    """
    This class contains the summary report for an update session. It reports the results in HTML format.
    """

    def __init__(self):
        self.new_attributes = []
        self.new_catalogs = []
        self.new_products = []
        self.messages = []

    def add_message(self, message):
        self.messages.append(message)

    def clear(self):
        self.new_attributes = []
        self.new_catalogs = []
        self.new_products = []
        self.messages = []

    def html(self):
        ret = []
        ret.append("<h2>Synchronisation report</h2>")
        ret.append("<h3>Messages</h3>")
        ret.append("<pre>")
        for row in self.messages:
            ret.append("%s" % row)
        ret.append("</pre>")
        if self.new_attributes:
            ret.append("<h3>New attributes</h3>")
            ret.append("<ul>")
            for row in self.new_attributes:
                ret.append("<li>%s</li>" % row)
            ret.append("</ul>")
        return '\n'.join(ret)


class CatalogSync(object):
    """
    This class handles the incoming synchronization data from the WatSan Website. It takes the data in JSON format and synchronises the local database with new values.
    Note that reference catalogs are updated with those from website, local objects are not touched.
    It creates and maintains internally an report that can be examied at the end of the synchronisation.

    >>> ds = DictionarySync(session, 'http://watsan2.eaudeweb.ro/catalogs_sync?format=json')
    >>> ds.sync_catalogs()
    >>> report = ds.report
    >>> report.new_values
        ...
    """

    def __init__(self, session, url_wwebsite):
        """
        Constructor
        Parameters
            `session`
                Database session for local portal database
            `url_wwebsite`
                URL to the WatSan Website entry point where data is harvested from
        """
        self.session = session
        self.url_wwebsite = url_wwebsite
        self.report = SynchronisationReport()


    def sync_catalogs(self):
        """
        Retrieve the JSON object with dictionary data and update the local portal with appropriate values
        """
        self.report.clear()
        ret = False
        errors = False

        file = None
        f = None
        json_data = {}

        self.report.add_message('[INFO]Loading new dictionary data from: %s' % self.url_wwebsite)

        try:
            file = urllib.urlopen(self.url_wwebsite)
            f = file.read()
            file.close()
        except:
            type, val, tb = sys.exc_info()
            message = string.join(traceback.format_exception(type, val, tb), '')
            self.report.add_message('[ERROR]Error while loading dictionary data: %s' % message)
            errors = True
        else:
            try:
                self.report.add_message('[INFO]Parsing dictionary data ...')
                json_data = simplejson.loads(f)
            except:
                type, val, tb = sys.exc_info()
                message = string.join(traceback.format_exception(type, val, tb), '')
                self.report.add_message('[ERROR]Error while parsing JSON dictionary data: %s' % message)
                errors = True
            else:
                try:
                    ret = True
                    self._sync_attributes(json_data)
                    self._sync_catalogs(json_data)
                    self._sync_catalogs_dd(json_data)
                    self._sync_catalogs_items(json_data)
                    self._sync_catalogs_data(json_data)
                except:
                    type, val, tb = sys.exc_info()
                    message = string.join(traceback.format_exception(type, val, tb), '')
                    self.report.add_message('[ERROR]Error during database synchronization: %s' % message)
                    errors = True
        if errors:
            self.report.add_message('[INFO]Catalog synchronization finished with errors. See report above for error messages')
        else:
            self.report.add_message('[INFO]Catalog synchronization finished')
        return ret


    def _sync_catalogs_data(self, json_data):
        self.report.add_message('[INFO] ---------- Synchronizing products data ----------')
        data = json_data['catalog_data']
        distinct_items = []
        for row in data:
            if row['id_item'] not in distinct_items:
                distinct_items.append(row['id_item'])
        #Delete all existing items matching id_item
        self.session.query(cat_models.CatCatalogData).filter(cat_models.CatCatalogData.id_item.in_(distinct_items)).delete(synchronize_session='fetch')
        for row in data:
            ob = cat_models.CatCatalogData()
            ob.id_item = row['id_item']
            ob.id_attribute = row['id_attribute']
            ob.value_string = row['value_string']
            ob.value_real = row['value_real']
            ob.value_integer = row['value_integer']
            ob.value_bool = row['value_bool']
            ob.value_lexicon = row['value_lexicon']
            self.session.add(ob)
        self.report.add_message('[INFO]Total data: %s' % len(data))
        self.report.add_message('[INFO]Done with products data...')



    def _sync_catalogs_items(self, json_data):
        count_new = count_updated = count_unchanged = 0
        items = json_data['items']
        self.report.add_message('[INFO] ---------- Synchronizing catalog products ----------')
        for ob in items:
            objects = self.session.query(cat_models.CatItem).filter_by(id_item=ob['id_item']).all()
            if objects:
                is_new = False
                row = objects[0]
            else:
                is_new = True
                row = cat_models.CatItem()
                count_new += 1

            if not is_new:
                if row.is_updated(ob):
                    count_updated += 1
                else:
                    count_unchanged += 1

            row.brand = ob['brand']
            row.model = ob['model']
            if ob['created']:
                row.created = datetime.fromtimestamp(float(ob['created']))
            else:
                row.created = None
            row.author = ob['author']

            if is_new:
                row.id_item = ob['id_item']
                row.id_catalog = ob['id_catalog']
                self.session.add(row)
            else:
                self.session.merge(row)

        self.report.add_message('[INFO]Total products: %s' % len(items))
        self.report.add_message('[INFO]New: %s' % count_new)
        self.report.add_message('[INFO]Updated: %s' % count_updated)
        self.report.add_message('[INFO]Unchanged: %s' % count_unchanged)
        self.report.add_message('[INFO]Done with products...')



    def _sync_catalogs_dd(self, json_data):
        """ Synchronize catalogs data definition. Catalog DD cannot be deleted, only updated or added new
        """
        self.report.add_message('[INFO] ---------- Synchronizing data definition ----------')
        dd = json_data['catalogs_dd']
        count_new = count_updated = count_unchanged = 0

        for ob in dd:
            objects = self.session.query(cat_models.CatCatalogDD).filter(
                            and_(cat_models.CatCatalogDD.id_catalog == ob['id_catalog'], 
                                 cat_models.CatCatalogDD.id_attribute==ob['id_attribute'])).all()
            if objects:
                row = objects[0]
                if row.show_in_overview != ob['show_in_overview']:
                    count_updated += 1
                    row.show_in_overview = ob['show_in_overview']
                    self.session.merge(row)
                else:
                    count_unchanged += 1
            else:
                count_new += 1
                row = cat_models.CatCatalogDD()
                row.id_catalog = ob['id_catalog']
                row.id_attribute = ob['id_attribute']
                row.show_in_overview = ob['show_in_overview']
                self.session.add(row)
        self.report.add_message('[INFO]New: %s' % count_new)
        self.report.add_message('[INFO]Updated: %s' % count_updated)
        self.report.add_message('[INFO]Unchanged: %s' % count_unchanged)
        self.report.add_message('[INFO]Done with data definition')


    def _sync_catalogs(self, json_data):
        count_new = count_updated = count_unchanged = 0
        catalogs = json_data['catalogs']
        self.report.add_message('[INFO] ---------- Synchronizing catalogs ----------')
        for ob in catalogs:
            objects = self.session.query(cat_models.CatCatalog).filter_by(id_catalog=ob['id_catalog']).all()
            if objects:
                is_new = False
                row = objects[0]
            else:
                is_new = True
                row = cat_models.CatCatalog()
                count_new += 1

            if not is_new:
                if row.is_updated(ob):
                    count_updated += 1
                else:
                    count_unchanged += 1

            row.name = ob['name']
            row.enabled = ob['enabled']
            row.reference = ob['reference']
            if ob['created']:
                row.created = datetime.fromtimestamp(float(ob['created']))
            else:
                row.created = None
            row.author = ob['author']

            if is_new:
                row.id_catalog = ob['id_catalog']
                self.session.add(row)
            else:
                self.session.merge(row)

        self.report.add_message('[INFO]Total catalogs: %s' % len(catalogs))
        self.report.add_message('[INFO]New: %s' % count_new)
        self.report.add_message('[INFO]Updated: %s' % count_updated)
        self.report.add_message('[INFO]Unchanged: %s' % count_unchanged)
        self.report.add_message('[INFO]Done with catalogs...')


    def _sync_attributes(self, json_data):
        count_new = count_updated = count_unchanged = 0
        attributes = json_data['attributes']
        self.report.add_message('[INFO] ---------- Synchronizing attributes ----------')
        for ob in attributes:
            objects = self.session.query(cat_models.CatAttribute).filter_by(id_attribute=ob['id_attribute']).all()
            if objects:
                is_new = False
                row = objects[0]
            else:
                is_new = True
                row = cat_models.CatAttribute()
                count_new += 1

            if not is_new:
                if row.is_updated(ob):
                    count_updated += 1
                else:
                    count_unchanged += 1
            row.name = ob['name']
            row.is_string = ob['is_string']
            row.is_integer = ob['is_integer']
            row.is_real = ob['is_real']
            row.is_bool = ob['is_bool']
            row.is_lexicon = ob['is_lexicon']
            row.is_url = ob['is_url']
            row.is_picture = ob['is_picture']
            row.lexicon_klass = ob['lexicon_klass']
            row.enabled = ob['enabled']
            row.reference = ob['reference']
            if ob['created']:
                row.created = datetime.fromtimestamp(float(ob['created']))
            else:
                row.created = None
            row.author = ob['author']

            if is_new:
                row.id_attribute = ob['id_attribute']
                self.session.add(row)
            else:
                self.session.merge(row)

        self.report.add_message('[INFO]Total attributes: %s' % len(attributes))
        self.report.add_message('[INFO]New: %s' % count_new)
        self.report.add_message('[INFO]Updated: %s' % count_updated)
        self.report.add_message('[INFO]Unchanged: %s' % count_unchanged)
        self.report.add_message('[INFO]Done with attributes...')

class CatalogSyncEndpoint(object):
    """ This endpoint generates the necessary data and provides it to the clients.
    """

    def __init__(self, session):
        """
        Constructor
        Parameters
            `session`
                Database connection to Website database
        """
        self.session = session

    def as_json(self):
        """
        Output data as JSON-encoded structure.
        Format of the output is:
        {
            'table_class' : {
                'key' : 'value',
            }
        }
        """
        json_data = {'attributes' : [], 'catalogs' : [], 'catalogs_dd' : [], 'items' : [], 'catalog_data' : []}

        for ob in self.session.query(cat_models.CatAttribute).all():
            json_data['attributes'].append(ob.as_json())

        for ob in self.session.query(cat_models.CatCatalog).all():
            json_data['catalogs'].append(ob.as_json())

        for ob in self.session.query(cat_models.CatCatalogDD).all():
            json_data['catalogs_dd'].append(ob.as_json())

        for ob in self.session.query(cat_models.CatItem).all():
            json_data['items'].append(ob.as_json())

        for ob in self.session.query(cat_models.CatCatalogData).all():
            json_data['catalog_data'].append(ob.as_json())

        return simplejson.dumps(json_data)
