# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cristian Romanescu, Eau De Web
# 
# Access context functionality - Locality level
from sqlalchemy.sql import func
from sqlalchemy.sql.expression import or_, not_, and_
from ws.common.sql.mappings import Waterpoint as wp_models
from ws.common.sql.mappings import SanitationFacility as sf_models
from ws.common.sql.mappings import Lexicon as lex_models
from ws.common.sql.mappings.SanitationFacility import SanitationFacilityView
from ws.common.sql.queries import Access as access_queries

class AccessLocalityStats(object):
    """
    Retrieve statistics for locality (see terms of ref. pg. 36)
    """


    def __init__(self, session, locality):
        self.session = session
        self.locality = locality


    def get_water_points(self):
        ret = {
            'public_taps' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
            'hand_pumps' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
            'modern_wells' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
            'other_wp' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
            'total_public_wp' :
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },


            'private_connections' : 
                {'functional' : 'See ticket #356)', 'outoforder' : 'See ticket #356)', 'total' : 'See ticket #356)' },
            'equivalent_wp' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
        }
        session = self.session
        locality = self.locality

        others_exclude = [lex_models.WaterPointType.PUBLIC_TAP, lex_models.WaterPointType.SELF_STANDING,
                   lex_models.WaterPointType.HANDPUMP_ON_BOREHOLE, lex_models.WaterPointType.HANDPUMP_ON_MODERN_WELL,
                   lex_models.WaterPointType.MODERN_WELL, lex_models.WaterPointType.TANK_WELL]

        lex_wp007 = session.query(lex_models.LexWP007).all()
        dict_wp007 = {}
        other_codes = []
        for ob in lex_wp007:
            dict_wp007[ob.code] = ob.ewp
            if ob.code not in others_exclude:
                other_codes.append(ob.code)

        # public taps
        qr = self._locality_statistics_wpn(session, lex_models.WaterPointType.PUBLIC_TAP)
        ret['public_taps']['functional'] += qr[0]
        ret['public_taps']['outoforder'] += qr[1]
        ret['public_taps']['total'] += qr[2]
        ewp = dict_wp007[lex_models.WaterPointType.PUBLIC_TAP]
        ret['equivalent_wp']['functional'] += qr[0] * ewp
        ret['equivalent_wp']['outoforder'] += qr[1] * ewp
        ret['equivalent_wp']['total'] += qr[2] * ewp

        qr = self._locality_statistics_wpn(session, lex_models.WaterPointType.SELF_STANDING)
        ret['public_taps']['functional'] += qr[0]
        ret['public_taps']['outoforder'] += qr[1]
        ret['public_taps']['total'] += qr[2]
        ewp = dict_wp007[lex_models.WaterPointType.SELF_STANDING]
        ret['equivalent_wp']['functional'] += qr[0] * ewp
        ret['equivalent_wp']['outoforder'] += qr[1] * ewp
        ret['equivalent_wp']['total'] += qr[2] * ewp

        # hand pumps
        qr = self._locality_statistics_wpn(session, lex_models.WaterPointType.HANDPUMP_ON_BOREHOLE)
        ret['hand_pumps']['functional'] += qr[0]
        ret['hand_pumps']['outoforder'] += qr[1]
        ret['hand_pumps']['total'] += qr[2]
        ewp = dict_wp007[lex_models.WaterPointType.HANDPUMP_ON_BOREHOLE]
        ret['equivalent_wp']['functional'] += qr[0] * ewp
        ret['equivalent_wp']['outoforder'] += qr[1] * ewp
        ret['equivalent_wp']['total'] += qr[2] * ewp

        qr = self._locality_statistics_wpn(session, lex_models.WaterPointType.HANDPUMP_ON_MODERN_WELL)
        ret['hand_pumps']['functional'] += qr[0]
        ret['hand_pumps']['outoforder'] += qr[1]
        ret['hand_pumps']['total'] += qr[2]
        ewp = dict_wp007[lex_models.WaterPointType.HANDPUMP_ON_MODERN_WELL]
        ret['equivalent_wp']['functional'] += qr[0] * ewp
        ret['equivalent_wp']['outoforder'] += qr[1] * ewp
        ret['equivalent_wp']['total'] += qr[2] * ewp

        # modern wells
        qr = self._locality_statistics_wpn(session, lex_models.WaterPointType.MODERN_WELL)
        ret['modern_wells']['functional'] += qr[0]
        ret['modern_wells']['outoforder'] += qr[1]
        ret['modern_wells']['total'] += qr[2]
        ewp = dict_wp007[lex_models.WaterPointType.MODERN_WELL]
        ret['equivalent_wp']['functional'] += qr[0] * ewp
        ret['equivalent_wp']['outoforder'] += qr[1] * ewp
        ret['equivalent_wp']['total'] += qr[2] * ewp

        qr = self._locality_statistics_wpn(session, lex_models.WaterPointType.TANK_WELL)
        ret['modern_wells']['functional'] += qr[0]
        ret['modern_wells']['outoforder'] += qr[1]
        ret['modern_wells']['total'] += qr[2]
        ewp = dict_wp007[lex_models.WaterPointType.TANK_WELL]
        ret['equivalent_wp']['functional'] += qr[0] * ewp
        ret['equivalent_wp']['outoforder'] += qr[1] * ewp
        ret['equivalent_wp']['total'] += qr[2] * ewp

        # others
        for code in other_codes:
            qr = self._locality_statistics_wpn(session, code)
            ret['other_wp']['functional'] += qr[0]
            ret['other_wp']['outoforder'] += qr[1]
            ret['other_wp']['total'] += qr[2]
            ewp = dict_wp007[code]
            ret['equivalent_wp']['functional'] += qr[0] * ewp
            ret['equivalent_wp']['outoforder'] += qr[1] * ewp
            ret['equivalent_wp']['total'] += qr[2] * ewp

        ret['total_public_wp']['functional'] = ret['public_taps']['functional'] + ret['hand_pumps']['functional'] \
                 + ret['modern_wells']['functional'] + ret['other_wp']['functional']
        ret['total_public_wp']['outoforder'] = ret['public_taps']['outoforder'] + ret['hand_pumps']['outoforder'] \
                 + ret['modern_wells']['outoforder'] + ret['other_wp']['outoforder']
        ret['total_public_wp']['total'] = ret['public_taps']['total'] + ret['hand_pumps']['total'] \
                 + ret['modern_wells']['total'] + ret['other_wp']['total']

        return ret


    def get_sanitation_facilities(self):
        ret = {
            'public_toilets' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
            'individual' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
            'total' : 
                {'functional' : 0, 'outoforder' : 0, 'total' : 0 },
        }
        session = self.session
        locality = self.locality

        qr = self._locality_statistics_sfn(session,
            [lex_models.SanitationFacilityType.PUBLIC_TOILET_FLUSH_LATRINES,
             lex_models.SanitationFacilityType.PUBLIC_TOILET_VIP_LATRINES,
             lex_models.SanitationFacilityType.PUBLIC_TOILET_VIP_SEPTIC_TANK_LATRINES,
             lex_models.SanitationFacilityType.PUBLIC_TOILET_TRADITIONAL_LATRINES,
             lex_models.SanitationFacilityType.PUBLIC_TOILET_ON_COLLECTIVE,
             lex_models.SanitationFacilityType.PUBLIC_TOILET_ON_SEMICOLLECTIVE])
        ret['public_toilets']['functional'] = qr[0]
        ret['public_toilets']['outoforder'] = qr[1]
        ret['public_toilets']['total'] = qr[2]

        qr = self._locality_statistics_sfn(session,
            [lex_models.SanitationFacilityType.SANPLAT_LATRINES,
             lex_models.SanitationFacilityType.ECOSAN_LATRINES_LATRINES,
             lex_models.SanitationFacilityType.FLUSH_LATRINES,
             lex_models.SanitationFacilityType.VIP_LATRINES,
             lex_models.SanitationFacilityType.SEPTIC_TANK_LATRINES])
        ret['individual']['functional'] = qr[0]
        ret['individual']['outoforder'] = qr[1]
        ret['individual']['total'] = qr[2]

        ret['total']['functional'] = ret['public_toilets']['functional'] + ret['individual']['functional']
        ret['total']['outoforder'] = ret['public_toilets']['outoforder'] + ret['individual']['outoforder']
        ret['total']['total'] = ret['public_toilets']['total'] + ret['individual']['total']

        return ret
    
    
    def _locality_statistics_wpn(self, session, upwpntype):
        qr = session.query(func.sum(wp_models.Upwpn.upwpnok),
                               func.sum(wp_models.Upwpn.upwpnoknot),
                               func.sum(wp_models.Upwpn.upwpntot)) \
                    .filter(and_(wp_models.Upwpn.wpnadlocode == self.locality.adlocode,
                                 wp_models.Upwpn.upwpntype == upwpntype
                                )
                        ).one()
        ret = []
        for val in qr:
            if val:
                ret.append(val)
            else:
                ret.append(0)
        return ret


    def _locality_statistics_sfn(self, session, upsfntypes):
        qr = session.query(func.sum(sf_models.Upsfn.upsfnok),
                               func.sum(sf_models.Upsfn.upsfnoknot),
                               func.sum(sf_models.Upsfn.upsfntot)) \
                    .filter(and_(sf_models.Upsfn.sfnadlocode == self.locality.adlocode,
                                 sf_models.Upsfn.upsfntype.in_(upsfntypes)
                                )
                        ).one()
        ret = []
        for val in qr:
            if val:
                ret.append(val)
            else:
                ret.append(0)
        return ret
