# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cristian Romanescu, Eau De Web
#
# Database common queries
from sqlalchemy.sql import desc, asc

from ws.common.sql.mappings.Lexicon import lexicon_list
from ws.common.sql.mappings.Upis import Upis
from ws.common.sql.mappings.Waterpoint import WaterpointView, WaterpointPendingView
from ws.common.sql.mappings.SanitationFacility import SanitationFacilityView, SanitationFacilityPendingView

from sqlalchemy.sql.expression import or_
from ws.common.sql.mappings.Subdivision import Subdivision


def get_lexicons():
    """
    Retrieve a mapping from lexicon names to classes.
    """
    ret = {}
    for d in lexicon_list:
        ret[d['table_code']] = (d['table_code'], d['table_name'], d['table_class'])
    return ret


def get_lexicons_klass():
    """
    Retrieve a mapping from lexicon names to classes.
    """
    ret = {}
    for d in lexicon_list:
        ret[d['table_code'].lower()] = d['table_class']
    return ret


def get_lexicon_klass(name):
    """
    Retrieve mapping class for a single lexicon
    """
    return lexicon_list[name]['table_class']


def get_lexicon_objects(session, cl):
    """
    Retrieve a list of rows for a given lexicon class.
    """
    return session.query(cl).order_by('name').all()


def get_lexicon_objects_by_classname(session, class_name):
    """
    Retrieve a list of rows for a given lexicon class.
    """
    if class_name in lexicon_list.keys():
        klass = lexicon_list[class_name]['table_class']
        return get_lexicon_objects(session, klass)


def get_lexicon_row(session, cl, code):
    """
    Retrieve a row for a given lexicon class.
    """
    ret = None
    try:
        ret = session.query(cl).filter(cl.code == code).one()
    except:
        pass
    return ret


def get_lexicon_row_by_classname(session, class_name, code):
    """
    Retrieve a row for a given lexicon class.
    """
    if class_name in lexicon_list.keys():
        klass = lexicon_list[class_name]['table_class']
        return get_lexicon_row(session, klass, code)


def get_upis(session, upiscode):
    """
    Retrieve the upis data.
    """
    return session.query(Upis).filter(Upis.upiscode == upiscode).one()


def get_sanitation(session, upiscode):
    """
    Retrieve the sanitation facility specific data.
    """
    return session.query(SanitationFacilityView).filter(SanitationFacilityView.upiscode == upiscode).one()


def list_table_data(session, klass, sort=None):
    qs = session.query(klass)
    if sort:
        try:
            column = getattr(klass, sort)
            qs = qs.order_by(column)
        except:
            pass
    return qs.all()


def get_waterpoints_localities(session):
    """
    Retrieves the list of available localities from WaterpointView
    """
    return session.query(WaterpointView.adlocode, WaterpointView.adloname).distinct().order_by(WaterpointView.adloname).all()

def get_waterpoints_functionalities(session):
    """
    Retrieves the list of available functionalities from WaterpointView
    """
    return session.query(WaterpointView.upisobj_code, WaterpointView.upisobj_name).distinct().order_by(WaterpointView.upisobj_name).all()

def get_sanitations_localities(session):
    """
    Retrieves the list of available localities from SanitationFacilityView
    """
    return session.query(SanitationFacilityView.adlocode, SanitationFacilityView.adloname).distinct().order_by(SanitationFacilityView.adloname).all()

def get_sanitations_functionalities(session):
    """
    Retrieves the list of available functionalities from WaterpointView
    """
    return session.query(SanitationFacilityView.upisobj_code, SanitationFacilityView.upisobj_name).distinct().order_by(SanitationFacilityView.upisobj_name).all()

def filter_water_points(session, text, id_subdivision, id_locality, functionality, sort_on, sort_order):
    """
    Filters the list of water points from WaterpointView for a give locality and functionality
    """
    query = session.query(WaterpointView)
    if text:
        filter_like = '%%%s%%' % text.strip()
        query = query.filter(or_(WaterpointView.upiscode.ilike(filter_like), WaterpointView.upisname.ilike(filter_like)))
    if id_locality:
        query = query.filter(WaterpointView.adlocode==id_locality)
    elif id_subdivision:
        subdivision = session.query(Subdivision).filter_by(code=id_subdivision).one()
        column = getattr(WaterpointView, 'upisadr%scode' % subdivision.level)
        query = query.filter(column == id_subdivision)

    if functionality:
        query = query.filter(WaterpointView.upisobj_code==functionality)
    if sort_on:
        column = getattr(WaterpointView, sort_on)
        if sort_order:
            query = query.order_by(asc(column))
        else:
            query = query.order_by(desc(column))
    return query

def get_pending_waterpoints_localities(session):
    """
    Retrieves the list of available localities from WaterpointPendingView
    """
    return session.query(WaterpointPendingView.adlocode, WaterpointPendingView.adloname).distinct().order_by(WaterpointPendingView.adloname).all()

def get_pending_water_points(session, locality, sort_on, sort_order):
    """
    Get the list of pending water points from WaterpointPendingView for a give locality
    """
    query = session.query(WaterpointPendingView)
    if locality:
        query = query.filter(WaterpointPendingView.adlocode==locality)
    if sort_on:
        column = getattr(WaterpointPendingView, sort_on)
    else:
        column = getattr(WaterpointPendingView, 'upisupwho')
    if sort_order:
        query = query.order_by(asc(column))
    else:
        query = query.order_by(desc(column))
    return query


def get_pending_sanitation_facilities_localities(session):
    """
    Retrieves the list of available localities from SanitationFacilityPendingView
    """
    return session.query(SanitationFacilityPendingView.adlocode, SanitationFacilityPendingView.adloname).distinct().\
           order_by(SanitationFacilityPendingView.adloname).all()

def get_pending_sanitation_facilities(session, locality, sort_on, sort_order):
    """
    Get the list of pending sanitation facilities from SanitationFacilityPendingView for a give locality
    """
    query = session.query(SanitationFacilityPendingView)
    if locality:
        query = query.filter(SanitationFacilityPendingView.adlocode==locality)
    if sort_on:
        column = getattr(SanitationFacilityPendingView, sort_on)
    else:
        column = getattr(SanitationFacilityPendingView, 'upisupwho')
    if sort_order:
        query = query.order_by(asc(column))
    else:
        query = query.order_by(desc(column))
    return query

def get_pending_water_point_by_id(session, id):
    """
    Get the one pending water point from WaterpointPendingView for a given id
    """
    return session.query(WaterpointPendingView).filter_by(upiscode=id).one()

def get_pending_sanitation_facility_by_id(session, id):
    """
    Get the one pending sanitation facility from SanitationFacilityPendingView for a given id
    """
    return session.query(SanitationFacilityPendingView).filter_by(upiscode=id).one()

def filter_sanitations(session, text, id_subdivision, id_locality, functionality, sort_on, sort_order):
    """
    Filters the list of sanitation points from SanitationFacilityView for a give locality and functionality
    """
    query = session.query(SanitationFacilityView)
    if text:
        filter_like = '%%%s%%' % text.strip()
        query = query.filter(or_(SanitationFacilityView.upiscode.ilike(filter_like), SanitationFacilityView.upisname.ilike(filter_like)))
    if id_locality:
        query = query.filter_by(adlocode=id_locality)
    elif id_subdivision:
        subdivision = session.query(Subdivision).filter_by(code=id_subdivision).one()
        column = getattr(SanitationFacilityView, 'upisadr%scode' % subdivision.level)
        query = query.filter(column == id_subdivision)

    if functionality:
        query = query.filter_by(upisobj_code=functionality)
    if sort_on:
        column = getattr(SanitationFacilityView, sort_on)
        if sort_order:
            query = query.order_by(asc(column))
        else:
            query = query.order_by(desc(column))
    return query