from sqlalchemy.sql import asc
from sqlalchemy.sql.expression import or_, and_, not_, func, desc

from ws.common.sql.mappings import Program as program_models
from ws.common.sql.mappings import Subdivision as subdivision_models
from ws.common.sql.mappings.Locality import Locality
from ws.common.sql.mappings.Subdivision import Subdivision


def filter_programs(session, text='', locality='', subdivision='', sort_on='', sort_order='', level=3):
    """
    Filters the list of ongoing programs from Program
    """
    query = session.query(program_models.Program)
    if text:
        filter_like = '%%%s%%' % text.strip()
        query = query.filter(or_(program_models.Program.adopname.ilike(filter_like), program_models.Program.adopacron.ilike(filter_like)))
    if locality:
        query = query.join(program_models.Tl_adoploc).filter_by(oploadlocode=locality)
    if subdivision:
        query = session.query(program_models.Program).join(program_models.Tl_adopr).filter_by(opradrcode=subdivision)
    if sort_on:
        column = getattr(program_models.Program, sort_on)
        if sort_order:
            query = query.order_by(asc(column))
        else:
            query = query.order_by(desc(column))
    return query


def list_programs(session, filter_out=[]):
    """ Retrieve the list of programs and filter out
    """
    return session.query(program_models.Program).filter(not_(program_models.Program.adopcode.in_(filter_out))).order_by(program_models.Program.adopname).all()


def latest_programs(session, limit=3):
    """ Retrieve the lateste programs added in to the database
    """
    return session.query(program_models.Program).order_by(desc(program_models.Program.adopdate)).limit(limit).all()


def get_program(session, code):
    """
    Retrieve the program object by it's code.
    """
    return session.query(program_models.Program).filter(program_models.Program.adopcode == code).one()


def get_budget(session, code):
    """
    Retrieve the programs' funding records by it's code.
    """
    return session.query(program_models.Budget).filter(program_models.Budget.opfadopcode == code).all()


def remove_budget(session, program_code, organisation_code):
    """ Remove budget entry """
    ob = session.query(program_models.Budget) \
        .filter(and_(
                program_models.Budget.opfadopcode==program_code,
                program_models.Budget.adopfcode==organisation_code)
        ).one()
    session.delete(ob)


def get_program_subdivisions_gis(session, code):
    """
    GIS query to retrieve the subdivisions where the program takes place
    Parameters:
        session
            Database session
        id
            Unique ID of the program
    """
    return session.query(program_models.View_gis_tl_adopr).filter_by(opradopcode=code).all()


def get_program_localities(session, code):
    """
    Retrieves the list of localities attached to specified program
    """
    return session.query(Locality).distinct().join(program_models.Tl_adoploc).filter_by(oploadopcode=code).order_by(Locality.adloname).all()


def get_program_subdivisions(session, code):
    """
    Retrieve the subdivisions where the program takes place
    SELECT * FROM tl_adopr WHERE opradopcode = 'PROJECT_ID'

    Parameters:
        session
            Database session
        code
            Unique code of the program
    """
    return session.query(Subdivision).distinct().join(program_models.Tl_adopr).filter_by(opradopcode=code).order_by(Subdivision.name).all()


def get_subdivision_list(session, exclude=[]):
    """ Retrieve the list of subdivisions for combo box in edit coverage, excluding those already in use """
    level = session.query(func.max(Subdivision.level)).scalar()
    return session.query(Subdivision).filter(and_(Subdivision.level == level, not_(Subdivision.code.in_(exclude)))).order_by('name').all()


def get_localities_list(session, subdivisions=[], excluded=[]):
    """ Retrieve the list of localities found in subdivisions affected by the program """
    return session.query(Locality).filter(
                and_(Locality.loadrcode.in_(subdivisions),
                     not_(Locality.adlocode.in_(excluded))
                )
            ).order_by('adloname').all()


def get_progress_data(session, code, date=''):
    """
    Retrieve the program progress data records by it's code.
    """
    query = session.query(program_models.ProgressData).\
            filter(program_models.ProgressData.opmadopcode == code)
    if date:
        return query.filter(program_models.ProgressData.adopmdate == date).one()
    return query.order_by(program_models.ProgressData.adopmdate).all()


def get_program_localities_in_use(session, subdivision = None):
    """
    Retrieves the list of localities that are affected by at least one program
    """
    if subdivision:
        level = 'level%s' % subdivision.level
        attr = getattr(subdivision_models.SubdivisionLocality, level)
        linked_locs = session.query(program_models.Tl_adoploc.oploadlocode).all()
        subdiv_locs = session.query(subdivision_models.SubdivisionLocality.adlocode).filter(
                            and_(
                                 attr == subdivision.code,
                                 subdivision_models.SubdivisionLocality.adlocode.in_([loc[0] for loc in linked_locs])
                            )).all()

        return session.query(Locality).filter(Locality.adlocode.in_(loc[0] for loc in subdiv_locs)).all()
    else:
        return session.query(Locality).distinct().join(program_models.Tl_adoploc).order_by(Locality.adloname).all()


def get_subdivisions_in_use(session):
    """ Retrieve the list of subdivisions that are attached to a program """
    return session.query(Subdivision).distinct().join(program_models.Tl_adopr).all()


def add_subdivision(session, program_code, subdivision_code):
    """ Attach one subdivision to the program """
    if program_code and subdivision_code:
        ob = program_models.Tl_adopr()
        ob.opradrcode = subdivision_code
        ob.opradopcode = program_code
        ob = session.add(ob)


def validate_add_subdivision(session, program_code, subdivision_code):
    """ Check for duplication """
    obs = session.query(program_models.Tl_adopr).filter(
                    and_(program_models.Tl_adopr.opradrcode == subdivision_code,
                         program_models.Tl_adopr.opradopcode == program_code)
            ).all()
    return  len(obs) == 0


def delete_subdivision(session, program_code, subdivision_code):
    """ Delete one subdivision linked to the program """
    if program_code and subdivision_code:
        ob = session.query(program_models.Tl_adopr).filter(
                    and_(program_models.Tl_adopr.opradrcode==subdivision_code,
                         program_models.Tl_adopr.opradopcode==program_code)
            ).one()
        session.delete(ob)


def validate_delete_subdivision(session, program_code, subdivision_code):
    """ Check for existence of the row in tl_adopr """
    obs = session.query(program_models.Tl_adopr).filter(
                    and_(program_models.Tl_adopr.opradrcode == subdivision_code,
                         program_models.Tl_adopr.opradopcode == program_code)
            ).all()
    return  len(obs) == 1


def add_locality(session, program_code, locality_code):
    """ Attach one subdivision to the program """
    if program_code and locality_code:
        ob = program_models.Tl_adoploc()
        ob.oploadlocode = locality_code
        ob.oploadopcode = program_code
        ob = session.add(ob)


def validate_add_locality(session, program_code, locality_code):
    """ Check for duplication """
    if program_code and locality_code:
        obs = session.query(program_models.Tl_adoploc).filter(
                    and_(program_models.Tl_adoploc.oploadlocode==locality_code,
                         program_models.Tl_adoploc.oploadopcode==program_code)
            ).all()
    return  len(obs) == 0


def delete_locality(session, program_code, locality_code):
    """ Delete one subdivision linked to the program """
    if program_code and locality_code:
        ob = session.query(program_models.Tl_adoploc).filter(
                    and_(program_models.Tl_adoploc.oploadlocode==locality_code,
                         program_models.Tl_adoploc.oploadopcode==program_code)
            ).one()
        session.delete(ob)


def validate_delete_locality(session, program_code, locality_code):
    """ Check for existence of the row in tl_adoploc """
    if program_code and locality_code:
        obs = session.query(program_models.Tl_adoploc).filter(
                    and_(program_models.Tl_adoploc.oploadlocode==locality_code,
                         program_models.Tl_adoploc.oploadopcode==program_code)
            ).all()
    return  len(obs) == 1
