from sqlalchemy.sql.expression import or_, asc, desc
from ws.common.sql.mappings import PWS as models
from ws.common.sql.mappings import Waterpoint as wp_models
from ws.common.sql.mappings.Lexicon import WaterPointType
from ws.common.sql.mappings.Locality import LocalityStats
from ws.common.sql.mappings.Subdivision import Subdivision

def filter(session, query_string, id_subdivision, id_locality, functionality, sort_on, sort_order):
    """
        Filters the list of piped water schemes for a given locality
    """
    query = session.query(models.PWSView)
    if query_string:
        query_string = '%%%s%%' % query_string.strip()
        query = query.filter(or_(models.PWSView.uppwsfunc.ilike(query_string), models.PWSView.upisname.ilike(query_string)))
    if id_locality:
        query = query.filter(models.PWSView.adlocode==id_locality)
    elif id_subdivision:
        subdivision = session.query(Subdivision).filter_by(code=id_subdivision).one()
        column = getattr(models.PWSView, 'upisadr%scode' % subdivision.level)
        query = query.filter(column == id_subdivision)
    if functionality:
        query = query.filter(models.PWSView.uppwsfunc==functionality)
    if sort_on:
        column = getattr(models.PWSView, sort_on)
        if sort_order:
            query = query.order_by(asc(column))
        else:
            query = query.order_by(desc(column))
    return query


def get_piped_water_scheme(session, id):
    """ Retrieve piped water scheme information. Information is retrieved
    from view_pws that contains both t_upis and t_uppws information
    """
    return session.query(models.PWSView).filter_by(upiscode=id).one()


def get_water_points(session, id):
    """ Retrieve the water points used by this scheme
    """
    ret = []
    lst = session.query(models.Tl_PWSWP).filter(models.Tl_PWSWP.pwsisupiscode == id).all()
    for pwswp in lst:
        row = {'code' : '', 'locality_code' : '', 'locality_name' : '', 'population' : '', 'public_taps' : '', 'cattle_troughs' : '', 'other' : ''}
        row['code'] = pwswp.wpisupiscode
        view_wp = session.query(wp_models.WaterpointView).filter(wp_models.WaterpointView.upiscode == pwswp.wpisupiscode).one()
        row['locality_code'] = view_wp.adlocode
        row['locality_name'] = view_wp.adloname

        upwpns = session.query(wp_models.Upwpn).filter(wp_models.Upwpn.wpnadlocode == view_wp.adlocode).all()
        for upwpn_row in upwpns:
            if upwpn_row.upwpntype == WaterPointType.PUBLIC_TAP:
                row['public_taps'] = upwpn_row.upwpntot
            if upwpn_row.upwpntype == WaterPointType.CATTLE_DROUGH:
                row['cattle_troughs'] = upwpn_row.upwpntot
            if upwpn_row.upwpntype == WaterPointType.OTHER:
                row['other'] = upwpn_row.upwpntot

        try:
            loc_stat = session.query(LocalityStats).filter(LocalityStats.adlocode == view_wp.adlocode).one()
            row['population'] = loc_stat.population
        except:
            print 'PipedWaterSchemes::get_water_points(): No locality statistics for %s' % view_wp.adlocode
            pass
        ret.append(row)
    return ret


def get_energy_source(session, id):
    return session.query(models.PWSEnergy).filter(models.PWSEnergy.enisupiscode == id).all()


def get_piping(session, id):
    return session.query(models.PWSPipe).filter(models.PWSPipe.piisupiscode == id).all()


def get_pumping(session, id):
    return session.query(models.PWSPump).filter(models.PWSPump.ppisupiscode == id).all()


def get_reservoirs(session, id):
    return session.query(models.PWSReservoir).filter(models.PWSReservoir.stisupiscode == id).all()


def get_water_treatment(session, id):
    return session.query(models.PWSWaterTreatment).filter(models.PWSWaterTreatment.ttisupiscode == id).all()


def get_piped_water_schemes_localities(session):
    """
    Retrieves the list of available localities from PWSView
    """
    return session.query(models.PWSView.adlocode, models.PWSView.adloname).distinct().order_by(models.PWSView.adloname).all()


def get_piped_water_schemes_functionalities(session):
    """
    Retrieves the list of available functionalities from PWSView
    """
    return session.query(models.PWSView.uppwsfunc, models.PWSView.uppwsfunc_name).distinct().order_by(models.PWSView.uppwsfunc_name).all()
