'''
Created on Jul 21, 2010

@author: cristiroma
'''
import random, calendar, time
from datetime import datetime

from ws.common.sql.mappings.Catalogs import CatAttribute, CatCatalog, CatCatalogDD, CatItem, CatCatalogData
from sqlalchemy.sql.expression import and_, asc, desc, or_, func, cast
from ws.common.utilities.paginate import DiggPaginator, EmptyPage, InvalidPage
from ws.common.sql.query import get_lexicon_row_by_classname
from sqlalchemy.types import Numeric

def generate_id(session, kolumn, reference):
    """ Generate primary key for table
        Parameters:
            `session`
                Database session
            `kolumn`
                SQLAlchemy mapping class' column to select from the maximum ID and increment with 1
            `reference`
                If true, ID prefix is REF (reference catalog) item, otherwise is LOC (local item)
    """
    max = session.query(func.max(cast(func.substring(kolumn, 4), Numeric))).one()
    max = (max[0] or 0) + 1
    if reference:
        return 'REF%.6d' % max
    return 'LOC%.6d' % max


def list_attributes(session):
    """ Retrieve the list of enabled attributes, sorted by name
    """
    return session.query(CatAttribute).filter_by(enabled=True).order_by('name').all()

def list_disabled_attributes(session):
    """ Retrieve the list of disabled attributes, sorted by name
    """
    return session.query(CatAttribute).filter_by(enabled=False).order_by('name').all()


def dict_attributes(session):
    ret = {}
    attributes = session.query(CatAttribute).all()
    for attr in attributes:
        ret[attr.id_attribute] = attr
    return ret

def add_attribute(session, form, reference):
    ob = CatAttribute()
    ob.id_attribute = generate_id(session, CatAttribute.id_attribute, reference)
    ob.reference = reference
    ob.author = form['author']
    ob.created = datetime.now()
    ob.enabled = True
    ob.name = form['name']
    ob.is_string = form['type'] == 'string'
    ob.is_integer = form['type'] == 'integer'
    ob.is_real = form['type'] == 'real'
    ob.is_bool = form['type'] == 'bool'
    ob.is_lexicon = form['type'] == 'lexicon'
    ob.is_url = form['type'] == 'url'
    ob.is_picture = form['type'] == 'picture'
    if form['lexicon']:
        ob.lexicon_klass = form['lexicon']
    session.add(ob)


def enable_attributes(session, ids, enable=True):
    attributes = session.query(CatAttribute).filter(CatAttribute.id_attribute.in_(ids)).all()
    for attr in attributes:
        attr.enabled = enable
        session.merge(attr)


def list_catalogs(session):
    """ Retrieve the list of existing catalogs, sorted by name
    """
    return session.query(CatCatalog).filter_by(enabled=True).order_by('name').all()


def list_disabled_catalogs(session):
    """ Retrieve the list of disabled catalogs, sorted by name
    """
    return session.query(CatCatalog).filter_by(enabled=False).order_by('name').all()


def list_editable_catalogs(session, is_portal):
    """ Retrieve the list of editable catalogs. In portal instances only local catalogs can be edited. 
        On website reference catalogs can be edited too.
    """
    if is_portal:
        return session.query(CatCatalog).filter(and_(CatCatalog.enabled==True, CatCatalog.reference==False)).order_by('name').all()
    return session.query(CatCatalog).filter_by(enabled=True).order_by('name').all()


def add_catalog(session, form, reference):
    """ Insert new catalog and add its attributes into data definition table
    """
    ob = CatCatalog()
    ob.id_catalog = generate_id(session, CatCatalog.id_catalog, reference)
    ob.reference = reference
    ob.author = form['author']
    ob.created = datetime.now()
    ob.enabled = True
    ob.name = form['name']
    ob.created = datetime.now()
    session.add(ob)
    session.flush()

    #Set the attributes
    attributes = form['selected_attribute']
    overview = []
    if 'overview_attribute' in form:
        overview = form['overview_attribute']
    for id in attributes:
        if id:
            ddob = CatCatalogDD()
            ddob.id_catalog = ob.id_catalog
            ddob.id_attribute = id
            ddob.show_in_overview = id in overview
            session.add(ddob)


def enable_catalogs(session, ids, enable=True):
    catalogs = session.query(CatCatalog).filter(CatCatalog.id_catalog.in_(ids)).all()
    for catalog in catalogs:
        catalog.enabled = enable
        session.merge(catalog)


def get_catalog(session, id):
    """ Retrieve one catalog by its ID
    """
    return session.query(CatCatalog).filter_by(id_catalog=id).one()


def get_catalog_attributes(session, id_catalog):
    """ Retrieve list of catalog attributes associated with one catalog
    """
    attributes = [attr.id_attribute for attr in session.query(CatCatalogDD).filter_by(id_catalog=id_catalog).all()]
    return session.query(CatAttribute).filter(CatAttribute.id_attribute.in_(attributes)).order_by('name').all()


def get_catalog_view_data(session, id_catalog, page, sort, order):
    """ Prepare data for page that displays catalog contents (catalog_view_html)
        Parameters:
            `session`
                SQLAlchemy session
            `id_catalog`
                What catalog to extract data from
            `page`
                What page of results to extract from paginator
            `sort`
                What column to sort data (brand/model/ID_ATTRIBUTE)
            `order`
                How to order
                    0,None - Default, 1 - Ascending, null results last 
    """
    ret = {'catalog' : None, 'header' : [], 'rows' : [], 'paginator' : None}
    catalog = session.query(CatCatalog).filter_by(id_catalog=id_catalog).one()
    ret['catalog'] = catalog

    dd = [attr.id_attribute for attr in session.query(CatCatalogDD).filter(and_(CatCatalogDD.id_catalog==id_catalog, CatCatalogDD.show_in_overview==True)).all()]
    headers = session.query(CatAttribute).filter(CatAttribute.id_attribute.in_(dd)).order_by('name').all()
    for header in headers:
        ret['header'].append(header)

    #Filtering and sorting
    products = session.query(CatItem).filter_by(id_catalog=id_catalog)
    if sort:
        try:
            sort = int(sort)
        except:
            # Regular sorting by brand/model
            if sort == 'brand':
                sort = CatItem.brand
            if sort == 'model':
                sort = CatItem.model
        else:
            # Sorting by an attribute
            sort = int(sort)
            attribute = session.query(CatAttribute).filter_by(id_attribute=sort).one()
            products = products.outerjoin(CatCatalogData).filter(or_(CatCatalogData.id_attribute == attribute.id_attribute, CatCatalogData.id_attribute == None))

            # Prepare the sorting column
            if attribute.is_string:
                sort = CatCatalogData.value_string
            if attribute.is_integer:
                sort = CatCatalogData.value_integer
            if attribute.is_real:
                sort = CatCatalogData.value_real
            if attribute.is_bool:
                sort = CatCatalogData.value_bool
            if attribute.is_lexicon:
                sort = CatCatalogData.value_lexicon
        if order:
            products = products.order_by(asc(sort))
        else:
            products = products.order_by(desc(sort))

    paginator = DiggPaginator(products, 10, body=5, padding=2, orphans=5)   #Show 10 items per page


    try:
        paginator = paginator.page(page)
    except (EmptyPage, InvalidPage):
        paginator = paginator.page(paginator.num_pages)
    ret['paginator'] = paginator


    for product in paginator.objects_list:
        row_data = [product.id_item, product.brand, product.model]
        attributes = fill_product_attributes(session, product.id_item)
        for header in headers:
            if header.id_attribute in attributes:
                row_data.append(attributes[header.id_attribute])
            else:
                row_data.append(None)
        ret['rows'].append( row_data )

    return ret


def fill_product_attributes(session, id_item):
    """ Returs an dictionary { id_attribute : value }
    """
    ret = {}
    product_data = session.query(CatCatalogData).filter_by(id_item=id_item).all()
    for record in product_data:
        ret[record.id_attribute] = record.get_value(session)
    return ret


def get_product_data(session, id_item):
    """ Retrive attributes and corresponding values for an item from catalog. Attributes are sorted alphabetically
    """
    ret = []
    item = session.query(CatItem).filter_by(id_item=id_item).one()
    catalog_attributes = session.query(CatAttribute).join(CatCatalogDD).filter(CatCatalogDD.id_catalog == item.id_catalog).order_by('name').all()

    data = session.query(CatCatalogData).filter_by(id_item=id_item).all()

    item_attributes = {}
    for row in data:
        item_attributes[row.id_attribute] = row

    for attribute in catalog_attributes:
        value = None
        if attribute.id_attribute in item_attributes.keys():
            attribute_data = item_attributes[attribute.id_attribute]
            if attribute.is_string or attribute.is_url or attribute.is_picture:
                value = attribute_data.value_string
            if attribute.is_integer:
                value = attribute_data.value_integer
            if attribute.is_real:
                value = attribute_data.value_real
            if attribute.is_bool:
                value = attribute_data.value_bool
            if attribute.is_lexicon:
                lex_row = get_lexicon_row_by_classname(session, attribute.lexicon_klass, attribute_data.value_lexicon)
                if lex_row:
                    value = lex_row.name

        ret.append({'attribute' : attribute, 'value' : value}) 
    return {'item' : item, 'data' : ret}


def add_product(session, form, gallery, reference):
    id_catalog = form['catalog']
    
    attributes = {}
    for element in form:
        try:
            id_attribute = element
        except:
            pass
        else:
            attributes[id_attribute] = form[element]

    catalog = session.query(CatCatalog).filter_by(id_catalog=id_catalog).one()

    item = CatItem()
    item.id_item = generate_id(session, CatItem.id_item, reference)
    item.id_catalog = id_catalog
    item.created = datetime.now()
    item.author = form['author']
    item.brand = form['brand']
    item.model = form['model']
    session.add(item)
    session.flush()

    for id in attributes.keys():
        qr = session.query(CatAttribute).filter_by(id_attribute=id).all()
        if qr:
            attribute = qr[0]
            data = CatCatalogData()
            data.id_item = item.id_item
            data.id_attribute = attribute.id_attribute
            if attributes[id]:
                if attribute.is_string or attribute.is_url:
                    data.value_string = attributes[id]
    
                if attribute.is_integer:
                    data.value_integer = attributes[id]
    
                if attribute.is_real:
                    data.value_real = float(attributes[id])
    
                if attribute.is_bool:
                    data.value_bool = bool(attributes[id])
    
                if attribute.is_lexicon:
                    data.value_lexicon = attributes[id]
    
                if attribute.is_picture:
                    file = attributes[id]
                    if file.filename:
                        id_filename = '%s_%s_%s' % (calendar.timegm(time.gmtime()), random.randint(100, 999), file.filename)
                        # Get/Create the photo album for this technical catalog
                        if not id_catalog in gallery.objectIds():
                            gallery.addNyPhotoFolder(id=str(id_catalog), title=catalog.name)
                        photo_catalog = gallery._getOb(str(id_catalog))

                        id_photo = photo_catalog.addNyPhoto(id=id_filename, title=file.filename, file=file, keywords=[item.brand, item.model])
                        data.value_string = '%s' % photo_catalog._getOb(id_photo).absolute_url()
            if data.is_valid():
                session.add(data)
    session.flush()


def product_search(session, term):
    """ Search in products for term
    """
    if term:
        match = '%%%s%%' % term.strip()
        ret = []
        items = session.query(CatItem).filter(or_(CatItem.brand.ilike(match), CatItem.model.ilike(match))).all()
        ret.extend([item for item in items])
        data_rows = session.query(CatCatalogData).filter(CatCatalogData.value_string.ilike(match)).all()
        ret.extend([data.item_ob for data in data_rows])
        return ret
    return []