'''
Created on Apr 21, 2010

@author: cristiroma
'''
from GChartWrapper import Line, HorizontalBarGroup
from sqlalchemy.sql import func
from sqlalchemy.sql.expression import and_, asc, or_, desc

from ws.common.sql.mappings.Country import Country, CountryStat
from ws.common.sql.mappings.Subdivision import Subdivision, SubdivisionStats,\
    SubdivisionLocality
from ws.common.sql.mappings.Waterpoint import WaterpointView
from ws.common.sql.mappings.SanitationFacility import SanitationFacilityView
from ws.common.sql.mappings.Locality import Locality, LocalityStats, LocalityStatsView

from ws.common.sql.mappings import Waterpoint as wp_models
from ws.common.sql.mappings import SanitationFacility as sf_models
from ws.common.sql.mappings import Lexicon as lex_models
from sqlalchemy.orm.exc import NoResultFound

CHART_COLOR_WATER = '47cdff'
CHART_COLOR_SANITATION = 'ffb889'

def get_country_access_overview(session, country_code):
    """
    Retrieve country access overview
    """
    country = session.query(Country).filter_by(adcncodex=country_code.upper()).one()

    ret =  {
        'water_access_country' : -1,
        'water_access_urban' : -1,
        'water_access_rural' : -1,
        'water_access_population' : -1,
        'sanitation_access_country' : -1,
        'sanitation_access_urban' : -1,
        'sanitation_access_rural' : -1,
        'sanitation_access_population' : -1,
        'year' : -1,
    }
    if country.water_access_urban:
        ret['water_access_urban'] = country.water_access_urban * 100
    if country.water_access_rural:
        ret['water_access_rural'] = country.water_access_rural * 100
    if country.sanitation_access_urban:
        ret['sanitation_access_urban'] = country.sanitation_access_urban * 100
    if country.sanitation_access_rural:
        ret['sanitation_access_rural'] = country.sanitation_access_rural * 100

    urban_pop_served_water = rural_pop_served_water = total_pop_water = 0
    if country.urban_population and country.water_access_urban:
        urban_pop_served_water = country.urban_population * country.water_access_urban
    if country.rural_population and country.water_access_rural:
        rural_pop_served_water = country.rural_population * country.water_access_rural
    if urban_pop_served_water and rural_pop_served_water:
        total_pop_water = int(urban_pop_served_water + rural_pop_served_water)
    ret['water_access_population'] = total_pop_water

    urban_pop_served_san = rural_pop_served_san = total_pop_san = 0
    if country.urban_population and country.sanitation_access_urban:
        urban_pop_served_san = country.urban_population * country.sanitation_access_urban
    if country.rural_population and country.sanitation_access_rural:
        rural_pop_served_san = country.rural_population * country.sanitation_access_rural
    if urban_pop_served_san and rural_pop_served_san:
        total_pop_san = int(urban_pop_served_san + rural_pop_served_san)
    ret['sanitation_access_population'] = total_pop_san

    if total_pop_water and country.total_population:
        ret['water_access_country'] = (total_pop_water * 100) / country.total_population
    if total_pop_san and country.total_population:
        ret['sanitation_access_country'] = (total_pop_san * 100) / country.total_population

    ret['year'] = country.year
    return ret


def get_country_access_history(session):
    ret = {
        'water_access_url' : None,
        'sanitation_access_url' : None,
    }
    CHART_WIDTH = 600
    CHART_HEIGHT = 200


    data = session.query(CountryStat).order_by('year')
    urban_set = []
    rural_set = []
    year_set = []
    for row in data:
        if row.water_access_urban:
            urban_set.append(row.water_access_urban * 100)
        if row.water_access_rural:
            rural_set.append(row.water_access_rural * 100)
        if row.water_access_urban or row.water_access_rural:
            year_set.append(row.year)

    if urban_set and rural_set and year_set:
        G = Line([urban_set, rural_set])
        G.size(CHART_WIDTH, CHART_HEIGHT)
        G.color(CHART_COLOR_WATER, CHART_COLOR_SANITATION)
        G.legend('Urban access', 'Rural & Semi-urban acces')
        G.legend_pos('r')
        G.axes('xy')
        G.axes.label(0, *year_set)
        G.axes.label(1, '25', '50', '75', '100')
        ret['water_access_url'] = G.url

    urban_set = []
    rural_set = []
    year_set = []
    for row in data:
        if row.sanitation_access_urban:
            urban_set.append(row.sanitation_access_urban * 100)
        if row.sanitation_access_rural:
            rural_set.append(row.sanitation_access_rural * 100)
        if row.sanitation_access_urban or row.sanitation_access_rural:
            year_set.append(row.year)


    if urban_set and rural_set and year_set:
        G = Line([urban_set, rural_set])
        G.size(CHART_WIDTH, CHART_HEIGHT)
        G.color(CHART_COLOR_WATER, CHART_COLOR_SANITATION)
        G.legend('Urban access','Rural access')
        G.legend_pos('r')
        G.axes('xy')
        G.axes.label(0, *year_set)
        G.axes.label(1, '25', '50', '75', '100')
        ret['sanitation_access_url'] = G.url

    return ret


def get_subdivision_access(session, level=1, parent=None):
    ret = {
        'chart_url' : None,
        'tabular_data' : []
    }
    subdivisions = session.query(Subdivision).filter(and_(Subdivision.level==level, Subdivision.parent_code==parent)).order_by(asc('name')).all()
    name_set = []
    water_set = []
    sanitation_set = []
    tabular_data = []

    for subdivision in subdivisions:
        water_access = 0
        sanitation_access = 0
        population = 0

        stats = None
        try:
            stats = session.query(SubdivisionStats).filter_by(code=subdivision.code).one()
        except NoResultFound:
            pass
        if stats:
            if stats.water_access:
                water_access = stats.water_access_formatted()
            if stats.sanitation_access:
                sanitation_access = stats.sanitation_access_formatted()
            population = stats.pop_estimated
        name_set.append(subdivision.name.encode('UTF-8'))
        water_set.append(water_access) # In %
        sanitation_set.append(sanitation_access) # In %
        # pass # No statistics for this subdivision
        tabular_data.append({'ob' : subdivision, 'water' : water_access, 'sanitation' : sanitation_access, 'population' : population})
    ret['tabular_data'] = tabular_data

    CHART_WIDTH = 300
    CHART_HEIGHT = 23 * len(name_set) + 20
    if CHART_HEIGHT > 1000: # Google fails for heights > 1000
        CHART_HEIGHT = 1000

    name_set.reverse() # For some strange reason, Google reverses the labels
    G = HorizontalBarGroup([water_set,sanitation_set], encoding='text')
    G.size(CHART_WIDTH, CHART_HEIGHT)
    G.bar(6, 0)
    G.axes('y')
    G.axes.label(0, *name_set)
    G.axes.label(1, '25', '50', '75', '100')
    G.color(CHART_COLOR_WATER, CHART_COLOR_SANITATION)
    G.legend('Water','Sanitation facilities')
    G.legend_pos('r')
    ret['chart_url'] = G.url

    return ret 

def get_locality_statistics(session, code):
    return session.query(LocalityStats).filter_by(adlocode = code).one()

def subdivision_statistics_on_localities(session, localities, type):
    """ Data for tabs in access page """
    result = []
    for locality in localities:
        ret = {'locality' : locality, 'stats' : None, 'rate' : 0, 'pc' : 0, 'pt' : 0, 'hp' : 0, 'mw' : 0}
        ret['locality'] = locality
        locality_stats = None
        try:
            locality_stats = get_locality_statistics(session, locality.adlocode)
        except:
            pass
        ret['stats'] = locality_stats
        if type == lex_models.UpistypeEnum.WP:
            if locality_stats and locality_stats.water_access:
                ret['rate'] = locality_stats.water_access_formatted()

            ret['pc'] = session.query(wp_models.Upwpn.upwpntot).filter(
                and_(wp_models.Upwpn.wpnadlocode == locality.adlocode,
                     wp_models.Upwpn.upwpntype == lex_models.WaterPointType.HOUSEHOLD_CONNECTION
                     )
                ).scalar()
            ret['pt'] = session.query(func.sum(wp_models.Upwpn.upwpntot)).filter(
                and_(wp_models.Upwpn.wpnadlocode == locality.adlocode,
                     wp_models.Upwpn.upwpntype.in_([lex_models.WaterPointType.PUBLIC_TAP, lex_models.WaterPointType.SELF_STANDING])
                     )
                ).scalar()
            ret['hp'] = session.query(func.sum(wp_models.Upwpn.upwpntot)).filter(
                and_(wp_models.Upwpn.wpnadlocode == locality.adlocode,
                     wp_models.Upwpn.upwpntype.in_([lex_models.WaterPointType.HANDPUMP_ON_BOREHOLE, lex_models.WaterPointType.HANDPUMP_ON_MODERN_WELL])
                     )
                ).scalar()
            ret['mw'] = session.query(func.sum(wp_models.Upwpn.upwpntot)).filter(
                and_(wp_models.Upwpn.wpnadlocode == locality.adlocode,
                     wp_models.Upwpn.upwpntype.in_([lex_models.WaterPointType.MODERN_WELL, lex_models.WaterPointType.TANK_WELL])
                     )
                ).scalar()

            result.append(ret)

        elif type == lex_models.UpistypeEnum.SF:
            ret['rate'] = locality.sf_access_formatted()
            ret['pto'] = session.query(func.sum(sf_models.Upsfn.upsfntot)).filter(
                and_(sf_models.Upsfn.sfnadlocode == locality.adlocode,
                     sf_models.Upsfn.upsfntype.in_([
                        lex_models.SanitationFacilityType.PUBLIC_TOILET_FLUSH_LATRINES,
                        lex_models.SanitationFacilityType.PUBLIC_TOILET_VIP_LATRINES,
                        lex_models.SanitationFacilityType.PUBLIC_TOILET_VIP_SEPTIC_TANK_LATRINES,
                        lex_models.SanitationFacilityType.PUBLIC_TOILET_TRADITIONAL_LATRINES,
                        lex_models.SanitationFacilityType.PUBLIC_TOILET_ON_COLLECTIVE,
                        lex_models.SanitationFacilityType.PUBLIC_TOILET_ON_SEMICOLLECTIVE
                        ])
                     )
                ).scalar()
            ret['iss'] = session.query(func.sum(sf_models.Upsfn.upsfntot)).filter(
                and_(sf_models.Upsfn.sfnadlocode == locality.adlocode,
                     sf_models.Upsfn.upsfntype.in_([
                        lex_models.SanitationFacilityType.SANPLAT_LATRINES,
                        lex_models.SanitationFacilityType.ECOSAN_LATRINES_LATRINES,
                        lex_models.SanitationFacilityType.FLUSH_LATRINES,
                        lex_models.SanitationFacilityType.VIP_LATRINES,
                        lex_models.SanitationFacilityType.SEPTIC_TANK_LATRINES,
                        lex_models.SanitationFacilityType.CONNECTION_TO_COLLECTIVE,
                        lex_models.SanitationFacilityType.CONNECTION_TO_SEMICOLLECTIVE
                        ])
                     )
                ).scalar()
            result.append(ret)
    return result


def get_subdivision(session, code):
    """
    Retrieve subdivision object by its code
    `session`
        SQL Alchemy Session instance
    `code`
        Subdivision code
    """
    if code:
        return session.query(Subdivision).filter_by(code=code).one()


def get_subdivision_upis(session, upis_ob):
    """
    Retrieve last level subdivision for 
    """
    for i in range(5, 0, -1):
        attr_value = getattr(upis_ob, 'upisadr%scode' % i)
        if attr_value:
            return session.query(Subdivision).filter(Subdivision.code == attr_value).one()
    return None


def get_subdivision_stats(session, code):
    """
    Retrieve subdivision statistic data object by its code
    `session`
        SQL Alchemy Session instance
    `code`
        Subdivision code
    """
    if code:
        return session.query(SubdivisionStats).filter_by(code=code).one()


def get_subdivision_max(session):
    return session.query(func.max(Subdivision.level)).scalar()


def get_subdivisions_by_level(session, level):
    """ Retrieve the list of subdivisions on specified level """
    return session.query(Subdivision).filter_by(level=level).all()


def get_subdivision_list(session, parent=None):
    """
    Retrieve the list of subdivisions for entire country or for a specific subdivision
    `session`
        SQL Alchemy Session instance
    `parent`
        Parent subdivision code
    Returns list of Subdivision objects
    """
    return session.query(Subdivision).filter(and_(Subdivision.parent_code == parent)).order_by('name').all()


def get_wp_for_subdivision(session, id_subdivision):
    subdivision = get_subdivision(session, id_subdivision)
    attribute = getattr(WaterpointView, 'upisadr%scode' % subdivision.level)
    return session.query(WaterpointView).filter(attribute == id_subdivision)


def get_sf_for_subdivision(session, id_subdivision):
    subdivision = get_subdivision(session, id_subdivision)
    attribute = getattr(SanitationFacilityView, 'upiadr%scode' % subdivision.level)
    return session.query(SanitationFacilityView).filter(attribute == id_subdivision)

def get_waterpoint(session, code):
    """ Retrieve the waterpoint specific data. """
    return session.query(wp_models.WaterpointView).filter(wp_models.WaterpointView.upiscode == code).one()

def get_sanitation(session, code):
    """ Retrieve the sanitation specific data. """
    return session.query(SanitationFacilityView).filter(SanitationFacilityView.upiscode == code).one()

def get_wpcomments(session, code):
    """ Retrieve a list of comments for a given waterpoint code """
    return session.query(wp_models.WPComments).filter(wp_models.WPComments.upiscode == code).all()


def get_locality_by_code(session, locality_code):
    """
    Retrieve the locality object by it's code.
    """
    return session.query(Locality).filter(Locality.adlocode == locality_code).one()

def get_locality_by_name(session, locality_name):
    """
    Retrieve the locality object by it's name.
    """
    return session.query(Locality).filter(Locality.adloname.ilike(locality_name)).one()


def get_subdivision_parents(session, subdivision):
    """
    Retrieve the hierarchical list of parents for a subdivision.
    Parameters:
        `subdivision`
            Subdivision whose parents are calculated
    """
    ret = []
    ob = subdivision
    while ob.parent_code is not None:
        ob = session.query(Subdivision).filter_by(code=ob.parent_code).one()
        ret.append(ob)
    ret.reverse()
    return ret


def get_localities(session, subdivision=None, order=Locality.adloname):
    """
    Retrieve the list of localities located within a subdivision or all localities
    Parameters:
        `session`
            Database session
        `subdivision`
            Restrict to a list of localities from a specific subdivision
        `order`
            Order the results list by one of the Locality attributes (as string)
    Return:
        List of Locality objects
    """
    query = session.query(Locality)
    if subdivision:
        attr_name = 'level%s' % subdivision.level
        attr_val = getattr(SubdivisionLocality, attr_name)
        filter = session.query(SubdivisionLocality).filter(attr_val == subdivision.code).all()
        query = query.filter(Locality.adlocode.in_([s.adlocode for s in filter]))

    if order:
        query = query.order_by(order)

    return query

def get_localities_statistics(session, q, subdv, sort_on, sort_order):
    """ returns localities statistics """
    if q:
        filter_like = '%s%%' % (q)
        query = session.query(LocalityStatsView).filter(LocalityStatsView.adloname.ilike(filter_like))
    else:
        query = session.query(LocalityStatsView)

    if subdv:
        code = getattr(SubdivisionLocality, 'level%s' % subdv.level)
        filter = session.query(SubdivisionLocality).filter(code == subdv.code).all()
        query = query.filter(LocalityStatsView.adlocode.in_([s.adlocode for s in filter]))

    if sort_on:
        column = getattr(LocalityStatsView, sort_on)
        if sort_order:
            query = query.order_by(asc(column))
        else:
            query = query.order_by(desc(column))
    return query       