# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Andrei Laza, Eau De Web
#
# SQL Alchemy mappings for waterpoint tables

from sqlalchemy.schema import Table, MetaData, Column, ForeignKey
from sqlalchemy.types import String, Integer, Float, Date, SmallInteger
from sqlalchemy.orm import mapper, relation
from ws.common.sql.mappings import Lexicon as lex_models
from ws.common.sql.mappings import Organisation as org_models
from ws.common.sql.mappings.Locality import Locality
from ws.common.sql.mappings.Upis import Upis
from ws.common.sql.mappings import IWRM

metadata = MetaData()
upwp_table = Table('t_upwp', metadata,
        Column('upiscode', String, ForeignKey(Upis.upiscode), primary_key=True),        #Code to be created by Min of Health.
        Column('upwpcodex', String),                                                    #Hydraulic inventory code of the infrastructure (WP003)
        Column('upwptype', String, ForeignKey(lex_models.LexWP007.code)),               #Type of water point
        Column('upwpbench', String, ForeignKey(lex_models.LexWP017.code)),              #Benchmarking of the water point by the users
        Column('upwpmainuse', String, ForeignKey(lex_models.LexWP009.code)),            #Principal use of the water point (WP009)
        Column('upwpantiq', String),                                                    #Anti-quagmire disposal          
        Column('upwpdiam', Float),                                                      #Diameter of the well
        Column('upwpdepth', Float),                                                     #Depth of the well
        Column('upwpcoph', Float),                                                      #Height of the coping of the well
        Column('upwpcatch', Float),                                                     #Catchment area (mp)
        Column('upwprain', Integer),                                                    #Rainfall (mm/year)
        Column('upwpnotfr', String, ForeignKey(lex_models.LexWP052.code)),              #Reasons for non functionality
        Column('upwpnotfd', String, ForeignKey(lex_models.LexWP053.code)),              #Duration of non functionality
        Column('upwdlfdate', Date),                                                     #Date of last failure
        Column('upwpasscw', String, ForeignKey(lex_models.LexWP055.code)),              #Assessment of civil works on the water point
        Column('upwpasspl', String, ForeignKey(lex_models.LexWP056.code)),              #Assessment of plumbering on the water point
        Column('upwpasshy', String, ForeignKey(lex_models.LexWP057.code)),              #Assessment of hygiene around the water point
        Column('upwpfence', String, ForeignKey(lex_models.LexWP058.code)),              #Fencing of the water point
        Column('upwpwturb', String, ForeignKey(lex_models.LexWP059.code)),              #Turbidity of water
        Column('upwpwtast', String, ForeignKey(lex_models.LexWP060.code)),              #Taste of water
        Column('upwpwodor', String, ForeignKey(lex_models.LexWP061.code)),              #Odor of water
        Column('upwptreat', String, ForeignKey(lex_models.LexWP062.code)),              #System of water treatment on the water point
        Column('upwptreatf', String, ForeignKey(lex_models.LexWP063.code)),             #Functionality of the system of water treatment
        Column('upwppaymd', String, ForeignKey(lex_models.LexWP064.code)),              #Modalities for the payment of water on the water point
        Column('upwppayseas', String, ForeignKey(lex_models.LexWP065.code)),            #Seasonability of the payment of water on the water point
        Column('upwphptype', String, ForeignKey(lex_models.LexWP066.code)),             #Brand/Type of the handpump
        Column('upwphpfunc', String, ForeignKey(lex_models.LexWP067.code)),             #Functionality of the handpump
        Column('upwphpyear', Integer),                                                  #Year of installation of the handpump
        Column('upwphpiwho', ForeignKey(org_models.Organisation.adorcode)),             #Implementing agency of the handpump
        Column('upwphpcraf', ForeignKey(org_models.Organisation.adorcode)),             #Identification code for the repairer craftmant of the handpump
        Column('upwpmeter', String, ForeignKey(lex_models.LexWP071.code)),              #Functionality of the water metering system
        Column('upwpyield', Float),                                                     #Measured or estimated yield (m3/h)
        Column('upwpprero', String),                                                    #Protection against erosion
        Column('upwpprpol', String, ForeignKey(lex_models.LexWP074.code)),              #Protection area against contamination by latrines
        Column('upwpdry', String, ForeignKey(lex_models.LexWP075.code)),                #Drying up of the water point
        Column('upwpclos', String, ForeignKey(lex_models.LexWP076.code)),               #Closing system on the well
        Column('upwpabsq', String),                                                     #Water abstraction system on the well
        Column('upwpwlev', Float),                                                      #Water level in the well
        Column('upwpeprice', Float),                                                    #Main water sale price (if water is sold) in country currency
        Column('upwpesaleu', String),                                                   #Sale unit
        Column('upwputype', String, ForeignKey(lex_models.LexUWS005.code)),             #Type of source (UWS005)
        Column('upwpuyelm', Float),                                                     #Maximum authorized abstraction yield (m3/h) (UWS010)
        Column('upwpudepin', SmallInteger),                                             #Recommended depth of installation (m) (UWS011)
        Column('fk_aqfcode', Integer, ForeignKey(IWRM.Aquifer.aqfcode)),                 #Catched aquifer (UWS012)
        Column('upwpudraw', String),                                                    #Borehole drawings. Link to a downlodable pdf file of the borehole drawings (UWS015)
)

class Waterpoint(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)
mapper(Waterpoint, upwp_table)


view_water_point = Table('view_water_point', metadata,
        Column('oid', Integer, primary_key=True),
        # view_subdivision_locality (t_adloc)
        Column('upisadr1code', String),
        Column('upisadr2code', String),
        Column('upisadr3code', String),
        Column('upisadr4code', String),
        Column('upisadr5code', String),
        Column('adloname', String),
        Column('adlocode', String, ForeignKey(Locality.adlocode)),
        # t_upis
        Column('upiscode', String),
        Column('upiscodex', String),
        Column('upistype', String, ForeignKey(lex_models.LexUpistype.code)),
        Column('upisname', String),
        Column('upisname2', String),
        Column('upisplace', String),
        Column('upisloctype', String, ForeignKey(lex_models.LexUpisloctype.code)),
        Column('upisx', Float), # Longitude
        Column('upisy', Float), # Latitude
        Column('upisz', Float),
        Column('upisyear', Integer),
        Column('upisobj', String, ForeignKey(lex_models.LexWP051.code)),
        Column('upisdend', Date),
        Column('upistdesc', String),
        Column('upiscomment', String),
        Column('upismanprof', String, ForeignKey(lex_models.LexUpisManProf.code)),
        Column('upisowner', String),
        Column('upisprogram', String),
        Column('upisfinance', String, ForeignKey(lex_models.LexUpisFinance.code)),
        Column('upisupdate', Date),
        Column('upisupwho', String),
        # t_upwp
        Column('upwpmainuse', String, ForeignKey(lex_models.LexWP009.code)),
        Column('upwptype', String, ForeignKey(lex_models.LexWP007.code)),
        Column('upwpbench', String, ForeignKey(lex_models.LexWP017.code)),
        Column('upwpantiq', String),
        Column('upwpdiam', Float),
        Column('upwpdepth', Float),
        Column('upwpcoph', Float),
        Column('upwpcatch', Float),
        Column('upwprain', Integer),
        Column('upwpnotfr', String, ForeignKey(lex_models.LexWP052.code)),
        Column('upwpnotfd', String, ForeignKey(lex_models.LexWP053.code)),
        Column('upwdlfdate', Date),
        Column('upwpasscw', String, ForeignKey(lex_models.LexWP055.code)),
        Column('upwpasspl', String, ForeignKey(lex_models.LexWP056.code)),
        Column('upwpasshy', String, ForeignKey(lex_models.LexWP057.code)),
        Column('upwpfence', String, ForeignKey(lex_models.LexWP058.code)),
        Column('upwpwturb', String, ForeignKey(lex_models.LexWP059.code)),
        Column('upwpwtast', String, ForeignKey(lex_models.LexWP060.code)),
        Column('upwpwodor', String, ForeignKey(lex_models.LexWP061.code)),
        Column('upwptreat', String, ForeignKey(lex_models.LexWP062.code)),
        Column('upwptreatf', String, ForeignKey(lex_models.LexWP063.code)),
        Column('upwppaymd', String, ForeignKey(lex_models.LexWP064.code)),
        Column('upwppayseas', String, ForeignKey(lex_models.LexWP065.code)),
        Column('upwphptype', String, ForeignKey(lex_models.LexWP066.code)),
        Column('upwphpfunc', String, ForeignKey(lex_models.LexWP067.code)),
        Column('upwphpyear', Integer),
        Column('upwphpiwho', String),
        Column('upwphpcraf', String),
        Column('upwpmeter', String, ForeignKey(lex_models.LexWP071.code)),
        Column('upwpyield', Float),
        Column('upwpprero', String),
        Column('upwpprpol', String, ForeignKey(lex_models.LexWP074.code)),
        Column('upwpdry', String, ForeignKey(lex_models.LexWP075.code)),
        Column('upwpclos', String, ForeignKey(lex_models.LexWP076.code)),
        Column('upwpabsq', String),
        Column('upwpwlev', Float),
        Column('upwpeprice', Float),
        Column('upwpesaleu', String),
        # lex_wp051 - Functionality of the water point
        Column('upisobj_code', String),
        Column('upisobj_name', String),
        # lex_wp007 - Type of water point
        Column('ewp', Float),
)

class WaterpointView(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)


mapper(WaterpointView, view_water_point, properties = {
    'locality': relation(Locality, lazy = False),
    'wptype': relation(lex_models.LexWP007, lazy = False),
    'upwpmainuse_ob': relation(lex_models.LexWP009, lazy = False),
})


view_water_point_unapproved = Table('view_water_point_unapproved', metadata,
        Column('oid', Integer, primary_key=True),
        Column('upiscode', String),
        Column('upwpmainuse', String, ForeignKey(lex_models.LexWP009.code)),
        Column('upisloctype', String, ForeignKey(lex_models.LexUpisloctype.code)),
        Column('upisobj', String, ForeignKey(lex_models.LexWP051.code)),
        Column('upisx', Float), # Longitude
        Column('upisy', Float), # Latitude
        Column('adloname', String),
        Column('adlocode', String),
        Column('upisplace', String),
        Column('upwptype', String, ForeignKey(lex_models.LexWP007.code)),
        Column('upwphpfunc', String, ForeignKey(lex_models.LexWP067.code)),
        Column('upisupdate', Date),
        Column('upisupwho', String)
)

class WaterpointPendingView(object):
    """ Mapping for view_water_point_unapproved view """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(WaterpointPendingView, view_water_point_unapproved)



table_upwpn = Table('t_upwpn', metadata,
        Column('wpnadlocode', String, primary_key=True),
        Column('upwpntype', String, ForeignKey(lex_models.LexWP007.code), primary_key=True),
        Column('upwpnok', Integer),
        Column('upwpnoknot', Integer),
        Column('upwpntot', Integer), 
)

class Upwpn(object): #LOC_WP
    """ Mapping for view_water_point_unapproved view """
    def __init__(self, wpnadlocode, upwpntype, upwpnok, upwpnoknot, upwpntot):
        self.wpnadlocode = wpnadlocode
        self.upwpntype = upwpntype
        self.upwpnok = upwpnok
        self.upwpnoknot = upwpnoknot
        self.upwpntot = upwpntot

mapper(Upwpn, table_upwpn)
