from sqlalchemy.schema import Table, MetaData, Column, ForeignKey
from sqlalchemy.types import String, Integer, Date, Boolean
from sqlalchemy.orm import mapper, relation
import sqlalchemy as sa

from zope.i18n.locales import locales
from ws.common.sql.mappings.Locality import Locality
from ws.common.sql.mappings import Lexicon as lex_models
from ws.common.sql.mappings.Organisation import Organisation, table_adorga
from ws.common.sql.mappings.Subdivision import Subdivision

en = locales.getLocale('en')
formatter = en.numbers.getFormatter('decimal')

metadata = MetaData()

table_adope = Table('t_adope', metadata,
        Column('adopcode', Integer, primary_key=True),   #Program ID
        Column('eaadorcode', Integer, ForeignKey(Organisation.adorcode)),   #Code of the executing organisation
        Column('iaadorcode', Integer, ForeignKey(Organisation.adorcode)),   #Code of the implementing organisation
        Column('adopname', String(80)),     #Program name
        Column('adopacron', String(20)),    #Short name or acronym of the program
        Column('adopdstart', Date),         #Starting date
        Column('adopdend', Date),           #Ending date
        Column('adopctpre', Boolean),       #Content: Preparation study
        Column('adopctfea', Boolean),       #Content: Feasibility study
        Column('adopctcb', Boolean),        #Content: Capacity building
        Column('adopctiec', Boolean),       #Content: IEC
        Column('adopctidev', Boolean),      #Content: Infrastructure development
        Column('adopctass', Boolean),       #Content: Assessment study
        Column('adopctme', Boolean),        #Content: Audit/M&E
        Column('adopctother', String),      #Content: other
        Column('adopctobj', String),        #Objectives
        Column('adopresexp', String),       #Expected results
        Column('adopactiv', String),        #Activities/Components
        Column('adopbudtot', Integer),      #Total budget
        Column('adopcontact', String(80)),  #Contact
        Column('adopurl', String),     #URL to the program website
        Column('adoptgpop', Integer),       #Target beneficiary population
        Column('adoptglocn', Integer),      #Target number of localities
        Column('adoptgwpn', Integer),       #Target number of new public water points
        Column('adoptgrpw', Integer),       #Target number of rehabilitated public water points
        Column('adoptgwcn', Integer),       #Target number of new water connections
        Column('adoptgsfpn', Integer),      #Target number of new public sanitation facilities
        Column('adoptgsfin', Integer),      #Target number of new individual sanitation facilities
        Column('adopdate', Date),           #Date when program was added into the database
        Column('adopwho', String),          #Record author
)

class Program(object):
    """ Mapping for program table (t_adope) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def edit(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def adopdstart_formatted(self):
        if self.adopdstart:
            return self.adopdstart.strftime('%d %B %Y')
        else:
            return 'TBA'

    def adopdend_formatted(self):
        if self.adopdend:
            return self.adopdend.strftime('%d %B %Y')
        else:
            return 'TBA'

    def adopbudtot_formatted(self):
        if self.adopbudtot:
            return formatter.format(self.adopbudtot)
        else:
            return 'N/A'

    def adoptgpop_formatted(self):
        if self.adoptgpop:
            return formatter.format(self.adoptgpop)

    def adoptglocn_formatted(self):
        if self.adoptglocn:
            return formatter.format(self.adoptglocn)

    def adoptgwpn_formatted(self):
        if self.adoptgwpn:
            return formatter.format(self.adoptgwpn)

    def adoptgrpw_formatted(self):
        if self.adoptgrpw:
            return formatter.format(self.adoptgrpw)

    def adoptgwcn_formatted(self):
        if self.adoptgwcn:
            return formatter.format(self.adoptgwcn)

    def adoptgsfpn_formatted(self):
        if self.adoptgsfpn:
            return formatter.format(self.adoptgsfpn)

    def adoptgsfin_formatted(self):
        if self.adoptgsfin:
            return formatter.format(self.adoptgsfin)


# http://svn.sqlalchemy.org/sqlalchemy/trunk/test/orm/test_lazy_relations.py
# Mappers for two columns on the same table
implementing_orga = sa.alias(table_adorga, 'implementing_orga')
implementing_mapper = mapper(Organisation, implementing_orga, non_primary=True)
executing_orga = sa.alias(table_adorga, 'executing_orga')
executing_mapper = mapper(Organisation, executing_orga, non_primary=True)

mapper(Program, table_adope,
       properties=dict(
        eaadorcode_ob=relation(executing_mapper, primaryjoin = sa.and_(executing_orga.c.adorcode == table_adope.c.eaadorcode), lazy = False),
        iaadorcode_ob=relation(implementing_mapper, primaryjoin = sa.and_(implementing_orga.c.adorcode == table_adope.c.iaadorcode), lazy = False)
       )
)


adopfund_table = Table('t_adopfund', metadata,
        Column('opfadopcode', Integer, primary_key=True),                                   #OPFProgram code
        Column('adopfcode', Integer, ForeignKey(Organisation.adorcode), primary_key=True),  #Funding source. Pointer to directory table
        Column('adopfamou', Integer),                                                       #Amount of funding
        Column('adopfmunit', String),                                                       #Money unit
        Column('adopfcat', String, ForeignKey(lex_models.LexOpb005.code)),                  #Category of funding

)

class Budget(object):
    """ Mapping for budget table (t_adopfund) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)
mapper(Budget, adopfund_table,
       properties=dict(
            adopfcode_ob=relation(Organisation, lazy = False),
            adopfcat_ob=relation(lex_models.LexOpb005, lazy = False),
       )
)


table_tl_adopr = Table('tl_adopr', metadata,
        Column('opradopcode', Integer, ForeignKey(Program.adopcode), primary_key=True), # Code of the program
        Column('opradrcode', String(20), ForeignKey(Subdivision.code), primary_key=True) # Code of the subdivision
)
class Tl_adopr(object):
    """ Mapping for program table tl_adopr """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(Tl_adopr, table_tl_adopr,
       properties=
       {
            'subdivision' : relation(Subdivision, lazy=True),
            'program' : relation(Program, lazy=True)
       }
)



view_gis_tl_adopr = Table('view_gis_tl_adopr', metadata,
        Column('code', String(20), primary_key=True), # Code of the subdivision
        Column('opradopcode', Integer, primary_key=True) # Code of the program
)
class View_gis_tl_adopr(object):
    """ Mapping for program view view_gis_tl_adopr """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(View_gis_tl_adopr, view_gis_tl_adopr)



table_tl_adoploc = Table('tl_adoploc', metadata,
        Column('oploadlocode', String(20), ForeignKey(Locality.adlocode), primary_key=True), # Code of the locality
        Column('oploadopcode', String(20), ForeignKey(Program.adopcode), primary_key=True) # Code of the program
)

class Tl_adoploc(object):
    """ Mapping for program table tl_adoploc """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(Tl_adoploc, table_tl_adoploc)



adopmo_table = Table('t_adopmo', metadata,
        Column('opmadopcode', Integer, primary_key=True),    #Ongoning program code
        Column('adopmdate', Date, primary_key=True),            #Date
        Column('adopmexpen', Integer),    #Cumulated expense commitments
        Column('adopmdisbu', Integer),    #Cumulated disbursements
        Column('adopmpopws', Integer),    #Cumulated population served with water
        Column('adopmpopsf', Integer),    #Cumulated population served with sanitation
        Column('adopmlocws', Integer),    #Cumulated number of localities served with water
        Column('adopmpwpnbr', Integer),   #Cumulated number of new public water points
        Column('adopmwprnbr', Integer),   #Cumulated number of rehabilitated water points
        Column('adopmwcnbr', Integer),    #Cumulated number of new water connections
        Column('adopmsfpnbr', Integer),   #Cumulated number of public sanitation facilities
        Column('adopmsfinbr', Integer),   #Cumulated number of individual sanitation facilities
)

class ProgressData(object):
    """ Mapping for progress table (t_adopmo) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def edit(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def adopmdate_formatted(self):
        return self.adopmdate.strftime('%d %b %Y')

    def adopmdate_year(self):
        return self.adopmdate.strftime('%Y')

    def adopmdate_month(self):
        return self.adopmdate.strftime('%b %Y')

    def adopmpopws_formatted(self):
        if self.adopmpopws is not None:
            return formatter.format(self.adopmpopws)

    def adopmpopsf_formatted(self):
        if self.adopmpopsf is not None:
            return formatter.format(self.adopmpopsf)

    def adopmpwpnbr_formatted(self):
        if self.adopmpwpnbr is not None:
            return formatter.format(self.adopmpwpnbr)

    def adopmwprnbr_formatted(self):
        if self.adopmwprnbr is not None:
            return formatter.format(self.adopmwprnbr)

    def adopmsfpnbr_formatted(self):
        if self.adopmsfpnbr is not None:
            return formatter.format(self.adopmsfpnbr)

mapper(ProgressData, adopmo_table)