'''
Created on Jun 28, 2010

@author: cristiroma
'''
from sqlalchemy.schema import Table, MetaData, Column, ForeignKey
from sqlalchemy.types import String, Integer, Float, Date, Boolean
from sqlalchemy.orm import mapper, relation

from ws.common.sql.mappings import Lexicon as lex_models
from ws.common.sql.mappings.Upis import Upis
from ws.common.sql.mappings.Locality import Locality
from ws.common.sql.mappings.Subdivision import Subdivision

from zope.i18n.locales import locales
from ws.common.sql.mappings.Waterpoint import Waterpoint
from ws.common.sql.mappings import Catalogs as catalogs_models


en = locales.getLocale('en')
formatter = en.numbers.getFormatter('decimal')

metadata = MetaData()


table_uppws = Table('t_uppws', metadata,
        Column('upiscode', String, ForeignKey(Upis.upiscode), primary_key=True),        #National identification code of the scheme (PWS001)
        Column('uppwsowntype', String, ForeignKey(lex_models.LexPWS005.code)),          #Ownership of the scheme (PWS005)
        Column('uppwsctrar', String, ForeignKey(lex_models.LexPWS007.code)),            #Contracting arrangement (PWS007)
        Column('uppwspop', Integer),                                                    #Total population served by the scheme (PWS008)
        Column('uppwslonbr', Integer),                                                  #Number of localities served by the scheme (PWS009)
        Column('uppwsstcpdi', Float),                                                   #Total water storage capacity for distribution (m3) (PWS011)
        Column('uppwsminhd', Float),                                                    #Minimum head on the scheme (m) (PWS012)
        Column('uppwsstcp', Float),                                                     #Total water storage capacity (m3) (PWS013)
        Column('uppwslentot', Float),                                                   #Total length of pipes (m) (PWS014)
        Column('uppwstapnbr', Integer),                                                 #Number of public taps (PWS015)
        Column('uppwsconnbr', Integer),                                                 #Number of connections (PWS016)
        Column('uppwscapnbr', Integer),                                                 #Number of posts for carts (PWS017)
        Column('uppwscatnbr', Integer),                                                 #Number of cattle troughs (PWS018)
        Column('uppwswdpnbr', Integer),                                                 #Number of other water delivery points (PWS019)
        Column('uppwsfunc', String, ForeignKey(lex_models.LexPWS020.code)),             #Functionality of the scheme (code PWS020)
        Column('uppwsabdm', Float),                                                     #Mean daily water abstraction (m3/day) (PWS021)
        Column('uppwscapday', Float),                                                   #Nominal daily capacity (m3/day) (PWS022)
        Column('uppwscapdr', Float),                                                    #Rate of use of the daily capacity % (PWS023)
        Column('uppwsenerc', String, ForeignKey(lex_models.LexPWS024.code)),            #Energy source(s) configuration (PWS024)
        Column('uppwsextent', String, ForeignKey(lex_models.LexPWS025.code)),           #Extent of piped scheme (PWS025)
        Column('uppwsschlen', Integer),                                                 #Total scheme length (m) (PWS026)
        Column('uppwsabstpe', String, ForeignKey(lex_models.LexPWS027.code)),           #Type of water resource abstracted (PWS027)
        Column('uppwschldis', Boolean),                                                 # Integrated chlorination disposal (PWS028)
        Column('uppwstpsys', Boolean),                                                  # Water treatment system (PWS029)
        Column('uppwspricept', Float),                                                  #Water sale price at public taps (per m3, in national currency) (PWS030)
        Column('uppwspricedt', Float),                                                  #Water sale price at domestic taps (per m3, in national currency) (PWS031)
        Column('uppwspriceos', Float),                                                  #Other sale price (in national currency) (PWS032)
        Column('uppwspriceun', String),                                                 #Price unit for other sale price (PWS033)
)

class PWS(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(PWS, table_uppws, properties = {
    'uppwsowntype_ob': relation(lex_models.LexPWS005, lazy = True),
    'uppwsctrar_ob': relation(lex_models.LexPWS007, lazy = True),
    'uppwsfunc_ob': relation(lex_models.LexPWS020, lazy = True),
    'uppwsenerc_ob': relation(lex_models.LexPWS024, lazy = True),
    'uppwsextent_ob': relation(lex_models.LexPWS025, lazy = True),
    'uppwsabstpe_ob': relation(lex_models.LexPWS027, lazy = True),
})



table_uppwsen = Table('t_uppwsen', metadata,
        Column('uppencode', Integer, primary_key=True),                         #Energy source identification code (PWE001)
        Column('enisupiscode', String, ForeignKey(PWS.upiscode)),               #National identification code of the scheme (PWE002->PWS001)
        Column('uppensrc', String, ForeignKey(lex_models.LexPWE003.code)),      #Category of energy source (PWE003)
        Column('uppenet', String, ForeignKey(catalogs_models.CatItem.id_item)), #Brand and type of the energy source. Pointer on the "Energy sources" catalog (PWE004)
        Column('uppenrp', Float),                                               #Rated power. Value extracted from catalog table (PWE005)
        Column('uppenup', Float),                                               #Unit of power. Value extracted from catalog table (PWE006)
        Column('uppenstart', Date),                                             #Starting date (PWE007)
)

class PWSEnergy(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(PWSEnergy, table_uppwsen, properties = {
    'enisupiscode_ob': relation(PWS, lazy = True),
    'uppensrc_ob': relation(lex_models.LexPWE003, lazy = True),
})



table_uppwspi = Table('t_uppwspi', metadata,
        Column('upppicode', Integer, primary_key=True),                         #Pipe identification code (PWP001)
        Column('piisupiscode', String, ForeignKey(PWS.upiscode)),               #National identification code of the scheme (PWI002->PWS001)
        Column('upppicp', String, ForeignKey(lex_models.LexPWI003.code)),       #Category of pipe (PWI003)
        Column('upppimdp', String, ForeignKey(catalogs_models.CatItem.id_item)),#Material / Diameter / Rated pressure. Pointer on the "Pipes" catalog (PWI004)
        Column('upppilen', Float),                                              #Length (m) (PWI005)
        Column('upppistart', Date),                                             #Starting date (PWI006)
)

class PWSPipe(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(PWSPipe, table_uppwspi, properties = {
    'piisupiscode_ob': relation(PWS, lazy = True),
    'upppicp_ob': relation(lex_models.LexPWI003, lazy = True),
})



table_uppwspp = Table('t_uppwspp', metadata,
        Column('uppppcode', Integer, primary_key=True),                         #Pump identification code (PWP001)
        Column('ppisupiscode', String, ForeignKey(PWS.upiscode)),               #National identification code of the scheme (PWP002->PWS001)
        Column('uppphyi', String),                                              #Hydraulic inventory code of the water source (PWP003)
        Column('uppppws', String, ForeignKey(lex_models.LexPWP004.code)),       #Category of water source (PWP004)
        Column('upppppump', String, ForeignKey(lex_models.LexPWP005.code)),     #Category of pump (PWP005)
        Column('uppppbtp', String, ForeignKey(catalogs_models.CatItem.id_item)),#Brand and type of pump. Pointer on the "Pumps" catalog (PWP006)
        Column('uppppenr', String, ForeignKey(lex_models.LexPWP007.code)),      #Category of energy required for the pump (PWP007)
        Column('uppppoy', Float),                                               #Operating yield (m3/h) (PWP008)
        Column('uppppotmh', Float),                                             #Operating total manometric head (PWP009)
        Column('uppppidp', Float),                                              #Installation depth of the pump (m) (PWP010)
        Column('uppppdwtoy', Float),                                            #Dynamic water table at operating yield (m) (PWP011)
        Column('uppppstart', Date),                                             #Starting date (PWP012)
)

class PWSPump(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def uppppoy_formatted(self):
        if self.uppppoy is not None:
            return formatter.format(self.uppppoy)

    def uppppotmh_formatted(self):
        if self.uppppotmh is not None:
            return formatter.format(self.uppppotmh)

    def uppppidp_formatted(self):
        if self.uppppidp is not None:
            return formatter.format(self.uppppidp)

    def uppppdwtoy_formatted(self):
        if self.uppppdwtoy is not None:
            return formatter.format(self.uppppdwtoy)

mapper(PWSPump, table_uppwspp, properties = {
    'ppisupiscode_ob': relation(PWS, lazy = True),
    'uppppws_ob': relation(lex_models.LexPWP004, lazy = True),
    'upppppump_ob': relation(lex_models.LexPWP005, lazy = True),
    'uppppenr_ob': relation(lex_models.LexPWP007, lazy = True),
})



table_uppwsst = Table('t_uppwsst', metadata,
        Column('uppstcode', Integer, primary_key=True),                         #Reservoir identification code (PWS001)
        Column('stisupiscode', String, ForeignKey(PWS.upiscode)),               #National identification code of the scheme (PWS002->PWS001)
        Column('uppstcr', String, ForeignKey(lex_models.LexPWSS003.code)),      #Category of reservoir (PWS003)
        Column('uppstbm', String, ForeignKey(lex_models.LexPWSS004.code)),      #Building material of the tank (PWS004)
        Column('uppsttc', Float),                                               #Tank capacity (m3) (PWS005)
        Column('uppsthbt', Float),                                              #Height from ground to the bottom of the tank (m) (PWS006)
        Column('uppstminh', Float),                                             #Minimum head on the scheme (m) (PWS007)
        Column('uppstx', Float),                                                #GPS coordinate (longitude) (PWS008)
        Column('uppsty', Float),                                                #GPS coordinate (latitude) (PWS009)
        Column('uppststart', Date),                                             #Starting date (PWS010)
)

class PWSReservoir(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def uppsttc_formatted(self):
        if self.uppsttc is not None:
            return formatter.format(self.uppsttc)

    def uppsthbt_formatted(self):
        if self.uppsthbt is not None:
            return formatter.format(self.uppsthbt)

    def uppstminh_formatted(self):
        if self.uppstminh is not None:
            return formatter.format(self.uppstminh)

    def uppstx_formatted(self):
        if self.uppstx is not None:
            return formatter.format(self.uppstx)

    def uppsty_formatted(self):
        if self.uppsty is not None:
            return formatter.format(self.uppsty)


mapper(PWSReservoir, table_uppwsst, properties = {
    'stisupiscode_ob': relation(PWS, lazy = True),
    'uppstcr_ob': relation(lex_models.LexPWSS003, lazy = True),
    'uppstbm_ob': relation(lex_models.LexPWSS004, lazy = True),
})



table_uppwstt = Table('t_uppwstt', metadata,
        Column('uppttcode', Integer, primary_key=True),                             #Water treatment equipement identification code
        Column('ttisupiscode', String, ForeignKey(PWS.upiscode)),                   #Piped water scheme identification code (PWT001)
        Column('upptttype', String, ForeignKey(lex_models.LexPWT002.code)),         #Type of water treatment (PWT002)
        Column('uppttdesc', String),                                                #Description of the equipment (PWT003)
        Column('uppttequip', String, ForeignKey(catalogs_models.CatItem.id_item)),  #Identification code of the equipment in the catalog (PWT004)
)

class PWSWaterTreatment(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(PWSWaterTreatment, table_uppwstt, properties = {
    'ttisupiscode_ob': relation(PWS, lazy = True),
    'upptttype_ob': relation(lex_models.LexPWT002, lazy = True),
})



table_uppwswp = Table('tl_uppwswp', metadata,
        Column('pwsisupiscode', String, ForeignKey(Upis.upiscode), primary_key=True),
        Column('wpisupiscode', String, ForeignKey(Waterpoint.upiscode), primary_key=True)
)

class Tl_PWSWP(object):
    """ Mapping for link between water schemes and water points table (tl_uppwswp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(Tl_PWSWP, table_uppwswp)



table_aulocpws = Table('tl_aulocpws', metadata,
        Column('pwsisupiscode', String, ForeignKey(Upis.upiscode), primary_key=True),
        Column('loadlocode', String, ForeignKey(Locality.adlocode), primary_key=True)
)

class Tl_Aulocpws(object):
    """ Mapping for link between water schemes and locality table (tl_aulocpws) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

mapper(Tl_Aulocpws, table_aulocpws)



# Views
view_pws = Table('view_pws', metadata,
        Column('oid', Integer, primary_key=True),
        # t_upis
        Column('upisadr1code', String, ForeignKey(Subdivision.code)),
        Column('upisadr2code', String, ForeignKey(Subdivision.code)),
        Column('upisadr3code', String, ForeignKey(Subdivision.code)),
        Column('upisadr4code', String, ForeignKey(Subdivision.code)),
        Column('upisadr5code', String, ForeignKey(Subdivision.code)),
        Column('upiscode', String),
        Column('upiscodex', String),
        Column('upistype', String, ForeignKey(lex_models.LexUpistype.code)),
        Column('upisname', String),
        Column('upisname2', String),
        Column('upisplace', String),
        Column('upisloctype', String, ForeignKey(lex_models.LexUpisloctype.code)),
        Column('upisx', Float), # Longitude
        Column('upisy', Float), # Latitude
        Column('upisz', Float),
        Column('upisyear', Integer),
        Column('upisdend', Date),
        Column('upistdesc', String),
        Column('upiscomment', String),
        Column('upismanprof', String, ForeignKey(lex_models.LexUpisManProf.code)),
        Column('upisowner', String),
        Column('upisprogram', String),
        Column('upisfinance', String, ForeignKey(lex_models.LexUpisFinance.code)),
        Column('upisupdate', Date),
        Column('upisupwho', String),
        # t_uppws
        Column('uppwsowntype', String, ForeignKey(lex_models.LexPWS005.code)),          #Ownership of the scheme (PWS005)
        Column('uppwsctrar', String, ForeignKey(lex_models.LexPWS007.code)),            #Contracting arrangement (PWS007)
        Column('uppwspop', Integer),                                                    #Total population served by the scheme (PWS008)
        Column('uppwslonbr', Integer),                                                  #Number of localities served by the scheme (PWS009)
        Column('uppwsstcpdi', Float),                                                   #Total water storage capacity for distribution (m3) (PWS011)
        Column('uppwsminhd', Float),                                                    #Minimum head on the scheme (m) (PWS012)
        Column('uppwsstcp', Float),                                                     #Total water storage capacity (m3) (PWS013)
        Column('uppwslentot', Float),                                                   #Total length of pipes (m) (PWS014)
        Column('uppwstapnbr', Integer),                                                 #Number of public taps (PWS015)
        Column('uppwsconnbr', Integer),                                                 #Number of connections (PWS016)
        Column('uppwscapnbr', Integer),                                                 #Number of posts for carts (PWS017)
        Column('uppwscatnbr', Integer),                                                 #Number of cattle troughs (PWS018)
        Column('uppwswdpnbr', Integer),                                                 #Number of other water delivery points (PWS019)
        Column('uppwsfunc', String, ForeignKey(lex_models.LexPWS020.code)),             #Functionality of the scheme (code PWS020)
        Column('uppwsfunc_name', String), # LexWP020
        Column('uppwsabdm', Float),                                                     #Mean daily water abstraction (m3/day) (PWS021)
        Column('uppwscapday', Float),                                                   #Nominal daily capacity (m3/day) (PWS022)
        Column('uppwscapdr', Float),                                                    #Rate of use of the daily capacity % (PWS023)
        Column('uppwsenerc', String, ForeignKey(lex_models.LexPWS024.code)),            #Energy source(s) configuration (PWS024)
        Column('uppwsextent', String, ForeignKey(lex_models.LexPWS025.code)),           #Extent of piped scheme (PWS025)
        Column('uppwsschlen', Integer),                                                 #Total scheme length (m) (PWS026)
        Column('uppwsabstpe', String, ForeignKey(lex_models.LexPWS027.code)),           #Type of water resource abstracted (PWS027)
        Column('uppwschldis', Boolean),                                                 # Integrated chlorination disposal (PWS028)
        Column('uppwstpsys', Boolean),                                                  # Water treatment system (PWS029)
        Column('uppwspricept', Float),                                                  #Water sale price at public taps (per m3, in national currency) (PWS030)
        Column('uppwspricedt', Float),                                                  #Water sale price at domestic taps (per m3, in national currency) (PWS031)
        Column('uppwspriceos', Float),                                                  #Other sale price (in national currency) (PWS032)
        Column('uppwspriceun', String),                                                 #Price unit for other sale price (PWS033)

        # t_adloc
        Column('adloname', String),
        Column('adlocode', String, ForeignKey(Locality.adlocode)),

)

class PWSView(object):
    """ Mapping for waterpoint table (t_upwp) """
    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)


    def uppwsstcpdi_formatted(self):
        if self.uppwsstcpdi is not None:
            return formatter.format(self.uppwsstcpdi)

    def uppwsminhd_formatted(self):
        if self.uppwsminhd is not None:
            return formatter.format(self.uppwsminhd)

    def uppwsstcp_formatted(self):
        if self.uppwsstcp is not None:
            return formatter.format(self.uppwsstcp)

    def uppwslentot_formatted(self):
        if self.uppwslentot is not None:
            return formatter.format(self.uppwslentot)

    def uppwsabdm_formatted(self):
        if self.uppwsabdm is not None:
            return formatter.format(self.uppwsabdm)

    def uppwscapday_formatted(self):
        if self.uppwscapday is not None:
            return formatter.format(self.uppwscapday)

    def uppwscapdr_formatted(self):
        if self.uppwscapdr is not None:
            return formatter.format(self.uppwscapdr)

    def uppwspricept_formatted(self):
        if self.uppwspricept is not None:
            return formatter.format(self.uppwspricept)

    def uppwspricedt_formatted(self):
        if self.uppwspricedt is not None:
            return formatter.format(self.uppwspricedt)

    def uppwspriceos_formatted(self):
        if self.uppwspriceos is not None:
            return formatter.format(self.uppwspriceos)


mapper(PWSView, view_pws, properties = {
    'locality': relation(Locality, lazy = True),
    'upistype_ob': relation(lex_models.LexUpistype, lazy = True),
    'upisloctype_ob': relation(lex_models.LexUpisloctype, lazy = True),

    'uppwsowntype_ob': relation(lex_models.LexPWS005, lazy = True),
    'uppwsctrar_ob': relation(lex_models.LexPWS007, lazy = True),
    'uppwsfunc_ob': relation(lex_models.LexPWS020, lazy = True),
    'uppwsenerc_ob': relation(lex_models.LexPWS024, lazy = True),
    'uppwsextent_ob': relation(lex_models.LexPWS025, lazy = True),
    'uppwsabstpe_ob': relation(lex_models.LexPWS027, lazy = True),
})
