# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cristian Romanescu, Eau De Web
#
# SQL utility functions
from ws.common.exceptions import DatabaseException

import os
import psycopg2

from psycopg2.extensions import ISOLATION_LEVEL_AUTOCOMMIT



def execute_sql(cursor, sql, zLOG = None, replace=None):
    """
    Execute SQL (as read from an file line by line)
    Parameters:
        `conn`
            Database connection
        `sql`
            Array with sql statements that will be executed. can be something 
            like file.readlines(). It will try to figure out if statement is
            spawning across multiple lines (until finds line with semicolon)
    Returns *True* if all operations were successfull or raises exception
    """
    buffer = []
    for line in sql:
        line = line.strip()
        if line.startswith('\xef\xbb\xbf'):
            line = line[3:] # Strip UTF-8 BOM
        if line != '' and not line.startswith('--') and not line.startswith('/*'):
            if replace:
                for key in replace:
                    line = line.replace('{%s}' % key, replace[key])
            buffer.append(line)
            if line.endswith(';'):
                statement = '\n'.join(buffer)
                try:
                    cursor.execute(statement)
                except Exception, ex:
                    if zLOG:
                        zLOG.LOG(__name__, zLOG.ERROR, 'Error executing query: %s' % statement)
                    raise ex
                buffer = []
    return True


def pg_connect(username, password, host, database):
    return psycopg2.connect(database=database, user=username, password=password, host=host)


def pg_execute_statement(conn, statement, commit=False, close=False, zLOG = None):
    """
    TODO: Test
    """
    cur = None
    try:
        try:
            if zLOG:
                zLOG.LOG(__name__, zLOG.INFO, 'SQL: %s' % (statement))
            cur = conn.cursor()
            cur.execute(statement)
        except Exception, ex:
            if zLOG:
                zLOG.LOG(__name__, zLOG.ERROR, 'Error executing query: %s' % statement)
            raise ex
    finally:
        if cur:
            cur.close()
        if commit:
            conn.commit()
        if close and conn:
            conn.close()


def pg_execute_sql_file(conn, sql_file_handle, commit=False, zLOG = None, replace=None):
    cur = None
    try:
        sql = sql_file_handle.readlines()
        cur = conn.cursor()
        execute_sql(cur, sql, zLOG, replace)
    finally:
        if cur:
            cur.close()
        if conn and commit:
            conn.commit()
            conn.close()


def pg_drop_database(username, password, host, database, super_database='postgres', zLOG=None):
    conn = pg_connect(username, password, host, super_database)
    conn.set_isolation_level(ISOLATION_LEVEL_AUTOCOMMIT)
    sql = "DROP DATABASE %(DATABASE)s" % {'DATABASE': database}
    if zLOG:
        zLOG.LOG(__name__, zLOG.INFO, 'Deleting database: %s (%s)' % (database, sql))
    cur = conn.cursor()
    cur.execute(sql)
    cur.close()
    conn.close()


def pg_create_database(username, password, host, database, owner=None, default_db='postgres', zLOG=None, check_exists=True):
    """
    Create new database
    Parameters:
        `username`
            Credentials for account with CREATEDB privilege
        `password`
            Password for username
        `host`
            Server host
        `database`
            Database to be created
        `owner`
            Owner of the new database, if None defaults to *username*
    Raises DatabaseException if database already exists. Other database exceptions
    Returns True if operation was successful
    This operation is not transactional - no rollback available
    """
    conn = cur = None
    try:
        if not owner:
            owner = username
        if check_exists and pg_database_exists(username, password, host, database):
            raise DatabaseException('Database `%s` already exists' % database)
        conn = pg_connect(username, password, host, default_db)
        conn.set_isolation_level(ISOLATION_LEVEL_AUTOCOMMIT)
        sql = "CREATE DATABASE %(DATABASE)s WITH ENCODING='UTF8' OWNER=%(OWNER)s TEMPLATE=template1 CONNECTION LIMIT=-1;" % {'DATABASE':database, 'OWNER' : owner}
        if zLOG:
            zLOG.LOG(__name__, zLOG.INFO, 'Creating empty database: %s (%s)' % (database, sql))
        cur = conn.cursor()
        cur.execute(sql)
    finally:
        if cur:
            cur.close()
        if conn:
            conn.close()
    return True


def pg_database_exists(username, password, host, database, default_db='postgres'):
    """
    Check if an existing Postgres database exists.
    `username`
        Database username for connection (must have access to default_db)
    `password`
        Database password for connection
    `database`
        Database that must be verified for existence
    `default_db`
        'Super' database to connect to. Default is postgres
    Returns True if database exists
    """
    cur = conn = None
    ret = False
    try:
        conn = pg_connect(username, password, host, default_db)
        cur = conn.cursor()
        cur.execute('SELECT datname FROM pg_database')
        rows = cur.fetchall()
        for row in rows:
            if row[0] == database:
                ret = True
    finally:
        try:
            if cur:
                cur.close()
                del cur
        except:
            pass
        try:
            if conn:
                conn.close()
                del conn
        except:
            pass
    return ret

