# WatSan Platform - Rapid development of national water and sanitation portals
# Copyright (C) 2010  Water and Sanitation Program (http://www.wsp.org)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author(s):
# Cristian Romanescu, Eau De Web
#
# Handles functionality for creation of a new portal
import traceback

import sys, string
import os, copy
from ws.common.exceptions import ValidationException
from Products.WSPortal import WSPortal
from ws.common.sql import pg_database_exists, pg_create_database, pg_connect,\
    pg_execute_statement, pg_execute_sql_file, pg_drop_database
import zLOG

class PortalConfiguration(object):
    """
    Cofiguration data passed when new portal is created
    """
    def __init__(self, **kwargs):
        self.country = None
        self.country_name = None

        self.level = None

        self.db_host = None
        self.db_username = None
        self.db_password = None
        self.db_super_username = None
        self.db_super_password = None
        self.db_super_database = None

        self.skin = None
        self.moderation = None
        self.mapserver_url = None
        self.languages = None
        self.gmap_key = None
        self.map_centerx = None
        self.map_centery = None
        self.zoom = None

        for k, v in kwargs.iteritems():
            setattr(self, k, v)


def create_wsportal(parent, cfg):
    country_code = cfg.country.lower()
    zLOG.LOG(__name__, zLOG.INFO, 'Creating new WatSan Country Portal for %s' % cfg.country_name)

    portal_uid = '%s' % (country_code)
    if hasattr(parent, portal_uid):
        raise ValidationException('An portal for this country was created before and probably deleted, please contact administrator to perform cleanup of this issue')

    db_name = 'watsan_portal_%s' % country_code

    # ZOPE specific
    zLOG.LOG(__name__, zLOG.INFO, '     * Creating Zope website')
    wsob = WSPortal.WSPortal(portal_uid, '%s country portal' % cfg.country_name, cfg.language, country_code)

    wsob.skin = cfg.skin
    wsob.level = int(cfg.level)
    wsob.moderation = int(cfg.moderation)
    wsob.mapserver_url = cfg.mapserver_url
    wsob.gmap_key = copy.deepcopy(cfg.gmap_key)

    wsob.db_name = db_name
    wsob.db_host = cfg.db_host
    wsob.db_username = cfg.db_username
    wsob.db_password = cfg.db_password

    wsob.default_latitude = cfg.map_centery
    wsob.default_longitude = cfg.map_centerx
    wsob.default_zoom = cfg.zoom

    parent._setObject(portal_uid, wsob)
    ob = parent._getOb(portal_uid)
    ob.createPortalTools()
    ob.loadDefaultData()
    #add i18n translation files
    i18n_dir = os.path.join(os.path.dirname(__file__), 'i18n')
    for lang in cfg.languages:
        ob.gl_add_site_language(lang)
        ob.getPortalTranslations().po_import(lang, open(os.path.join(i18n_dir, '%s.po' % lang)).read())
    ob.getLayoutTool().manageLayout('skin', wsob.skin)
    map_engine = ob.portal_map.get_map_engine()
    map_engine.api_keys = (ob.get_google_map_default_key())

    # Database creation
    conn = None
    if not pg_database_exists(cfg.db_super_username, cfg.db_super_password, cfg.db_host, db_name, cfg.db_super_database):
        try:
            try:
                zLOG.LOG(__name__, zLOG.INFO, '    * Creating PostgreSQL database named %s' % db_name)
                pg_create_database(cfg.db_super_username, cfg.db_super_password, cfg.db_host, db_name, cfg.db_username, cfg.db_super_database, zLOG)

                conn = pg_connect(cfg.db_super_username, cfg.db_super_password, cfg.db_host, db_name)
                pg_execute_statement(conn, 'ALTER TABLE geometry_columns OWNER TO %s;' % cfg.db_username, True, False, zLOG)
                pg_execute_statement(conn, 'ALTER TABLE spatial_ref_sys OWNER TO %s;' % cfg.db_username, True, True, zLOG)

                #create table structure
                base_dir = os.path.join(os.path.dirname(WSPortal.__file__), 'sql')
                sql_files = ['country.sql', 'lexicon.sql', 'catalog.sql', 'portal.sql' ]
                files = [os.path.join(base_dir, file) for file in sql_files]
                for file in files:
                    if os.path.exists(file):
                        f = open(file)
                        conn = pg_connect(cfg.db_username, cfg.db_password, cfg.db_host, db_name)
                        zLOG.LOG(__name__, zLOG.INFO, '        * Executing SQL script: %s' % file)
                        pg_execute_sql_file(conn, f, True, zLOG)
                        f.close()
                    else:
                        zLOG.LOG(__name__, zLOG.WARNING, '        * Skipping (not found) file: %s' % file)

                # Create the views
                columns = ['a', 'b', 'c', 'd', 'e']
                file = os.path.join(base_dir, 'views.sql')
                joined_column = columns[wsob.level - 1]
                if os.path.exists(file):
                    f = open(file)
                    conn = pg_connect(cfg.db_username, cfg.db_password, cfg.db_host, db_name)
                    zLOG.LOG(__name__, zLOG.INFO, '        * Executing SQL script: %s' % file)
                    pg_execute_sql_file(conn, f, True, zLOG, replace={'level' : joined_column, 'level_column' : 'level%s' % wsob.level})
                    f.close()
                else:
                    zLOG.LOG(__name__, zLOG.WARNING, '        * Skipping (not found) file: %s' % file)


                # Import portal default data
                sql_files = ['country.sql', 'catalog.sql', 'lexicon.sql']
                files = [os.path.join(base_dir, 'data', file) for file in sql_files]
                # try to load data for that country
                files.append(os.path.join(base_dir, 'data', '%s.sql' % (country_code))) 
                for file in files:
                    if os.path.exists(file):
                        f = open(file)
                        conn = pg_connect(cfg.db_username, cfg.db_password, cfg.db_host, db_name)
                        zLOG.LOG(__name__, zLOG.INFO, '        * Executing SQL script: %s' % file)
                        pg_execute_sql_file(conn, f, True, zLOG)
                        f.close()
                    else:
                        zLOG.LOG(__name__, zLOG.WARNING, '        * Skipping (not found) file: %s' % file)


            except Exception, ex:
                type, val, tb = sys.exc_info()
                msg = string.join(traceback.format_exception(type, val, tb), '')
                zLOG.LOG(__name__, zLOG.ERROR, 'Error occurred . WatSan country portal creation process encountered a fatal error: %s' % msg, error = (type, val, tb), reraise=False)
                pg_drop_database(cfg.db_super_username,  cfg.db_super_password, cfg.db_host, db_name, cfg.db_super_database, zLOG)
                raise ex
        finally:
            if conn:
                del conn
    else:
        zLOG.LOG(__name__, zLOG.WARNING, 'Database %s already exists for this portal. It will be used as current database' % db_name)

    zLOG.LOG(__name__, zLOG.INFO, 'Successfully created the portal')
    return ob