import simplejson as json
import mapscript

def gis_query_locality(mapfile, layername, lat, lon, zLOG=None):
    """
    Return the ID of the locality by doing a MapServer GetFeature
    """
    ret = None
    if zLOG:
        zLOG.LOG(__name__, zLOG.INFO, 'GIS Query on %s' % mapfile)
    mapOb = mapscript.mapObj(mapfile)
    if mapOb:
        layerOb = mapOb.getLayerByName(layername)
        if zLOG:
            zLOG.LOG(__name__, zLOG.INFO, '    * Layer: "%s"' % layername)
        if layerOb:
            pointOb = mapscript.pointObj(float(lat), float(lon))
            if zLOG:
                zLOG.LOG(__name__, zLOG.INFO, '    * Position (%s, %s)' % (lat, lon))
            res = layerOb.queryByPoint(mapOb, pointOb, mapscript.MS_SINGLE, -1)
            if res == mapscript.MS_SUCCESS:
                results = layerOb.getResults()
                if zLOG:
                    zLOG.LOG(__name__, zLOG.INFO, '    * SUCCESS: %s feature(s)' % results.numresults)
                if results.numresults > 0:
                    result = layerOb.getResult(0)
                    feature = layerOb.getFeature(result.shapeindex, result.tileindex)
                    ret = feature.getValue(0)
                    if zLOG:
                        zLOG.LOG(__name__, zLOG.INFO, '    * ID: %s' % ret)
    return ret



TYPE_GOOGLE = 'google'
TYPE_GOOGLE_HYB = 'google_hyb'
TYPE_GOOGLE_SAT = 'google_sat'
TYPE_MAPSERVER = 'mapserver'


class MapLayer(object):


    def __init__(self, name, type, *args, **kwargs):
        """
        Parameters:
            name
                Name of the layer as will appear on user interface
            type
                One of ML_TYPE_* constants
        """
        self.name = name
        self.type = type
        
        # Custom layer configuration passed to OpenLayers and then to underlaying API (google, yahoo API etc.)
        # Is already populated with some default values
        self.cfg = {
            'sphericalMercator' : False
        }
        
        # Custom layer layout on screen. These settings are specific to OpenLayers. 
        self.layout = {
            'reproject' : True,
            'visibility' : False,
        }
        
        if 'cfg' in kwargs.keys():
            self.cfg.update(kwargs['cfg']) 
        if 'layout' in kwargs.keys():
            self.layout.update(kwargs['layout']) 


    def to_umap(self, as_json=False):
        """
        Construct this layer as UMap compatible interface dictionary
        Parameters:
            as_json
                Return the data as json string. Default returns Python dictionary
        """
        ret = {}
        ret['type'] = self.type
        ret['name'] = self.name
        ret['cfg'] = self.cfg
        ret['layout'] = self.layout
        if as_json:
            return json.dumps(ret)
        return ret



class MapLayerGoogle(MapLayer):

    def __init__(self, name='Google Maps', *args, **kwargs):
        """ Construct an layer of type Google Maps normal. """
        self.filter = {}
        super(MapLayerGoogle, self).__init__(name, TYPE_GOOGLE, *args, **kwargs)



class MapLayerGoogleHybrid(MapLayer):

    def __init__(self, name='Google Hybrid', *args, **kwargs):
        """ Construct an layer of type Google Maps hybrid. """
        self.filter = {}
        super(MapLayerGoogleHybrid, self).__init__(name, TYPE_GOOGLE_HYB, *args, **kwargs)



class MapLayerGoogleSatellite(MapLayer):

    def __init__(self, name='Google Satellite', *args, **kwargs):
        """ Construct an layer of type Google Maps satellite. """
        self.filter = {}
        super(MapLayerGoogleSatellite, self).__init__(name, TYPE_GOOGLE_SAT, *args, **kwargs)



class MapLayerMapserver(MapLayer):

    def __init__(self, name, *args, **kwargs):
        """
        Construct an layer of type UNM MapServer.
        Examples:
            layer = gis.MapLayerMapserver('Localities', cfg={ 'MAP' : '%s' % sn_map, 'layers' : 'Locality' })
        """
        self.filter = {}
        super(MapLayerMapserver, self).__init__(name, TYPE_MAPSERVER, *args, **kwargs)


    def add_filter(self, name, str_filter):
        """
        Append an WMS parameter filter (used by Mapfile FILTER directive).
        These are appended automatically by openlayers in mapserver URL when doing layer
        Standard WMS format is http://server.com/cgi-bin/mapserv?map=my.map&filterX=CODE=52||CODE=53...
        """
        self.filter[name] = str_filter


    def to_umap(self, as_json=False):
        # Take all filters and append them as cfg parameters
        self.cfg.update(self.filter)
        return super(MapLayerMapserver, self).to_umap(as_json)


    def ut_OR_filter(self, key, values=[], attribute=None):
        """
        Make an GIS OR filters
        [key]=values[0] || [key]=values[1]...
        """
        ret = []
        for value in values:
            val = value
            if attribute:
                val = getattr(value, attribute)
            ret.append('[%s]=%s' % (key, val))
        return '||'.join(ret)


    def ut_regex_or_filter(self, values=[], attribute=None):
        """
        Make an GIS OR filters (regular expression)
        values[0]|values[1]...
        """
        ret = []
        for value in values:
            val = value
            if attribute:
                val = getattr(value, attribute)
            ret.append(val)
        return '|'.join(ret)

