# -*- coding: UTF-8 -*-

'''
Created on May 31, 2010
@author: cristiroma
'''
import sys, string, traceback
import random, textwrap
from random import randint as rand_int
from time import mktime
import datetime
from itertools import chain, islice, izip
from sqlalchemy.sql.expression import and_

from Products.Naaya.NyFolder import addNyFolder
from Products.WSPortal.Programs.WSPortalProgram import create_wsportalprogram_object_callback
from Products.WSPortal.Comments.models import UpisComment

from ws.common.sql.mappings.Lexicon import UpistypeEnum, lexicon_list
from ws.common.sql.mappings.Upis import Upis
from ws.common.sql.mappings.Waterpoint import Waterpoint, Upwpn
from ws.common.sql.mappings import PWS as pws_models
from ws.common.sql.mappings.SanitationFacility import Sanitation, Upsfn
from ws.common.sql.mappings.Subdivision import Subdivision, SubdivisionView
from ws.common.sql.mappings.Locality import Locality
from ws.common.sql.mappings.Organisation import Organisation
from ws.common.sql.mappings.Program import Program, Budget, ProgressData,\
    Tl_adoploc, Tl_adopr
from ws.common.sql.mappings import IWRM
from ws.common.utilities import dummy_insert
from ws.common.sql.mappings.Country import Country, CountryStat
from ws.common.sql.mappings.WaterQuality import WaterQuality, WaterQualityData

class Fixtures(object):

    countries = [ "Afghanistan", "Albania", "Algeria", "American Samoa", "Andorra",
        "Angola", "Anguilla", "Antarctica", "Antigua And Barbuda", "Argentina", "Armenia",
        "Aruba", "Australia", "Austria", "Azerbaijan", "Bahamas", "Bahrain", "Bangladesh",
        "Barbados", "Belarus", "Belgium", "Belize", "Benin", "Bermuda", "Bhutan", "Bolivia",
        "Bosnia And Herzegovina", "Botswana", "Bouvet Island", "Brazil", "British Indian Ocean Territory",
        "Brunei Darussalam", "Bulgaria", "Burkina Faso", "Burundi", "Cambodia", "Cameroon", "Canada",
        "Cape Verde", "Cayman Islands", "Central African Republic", "Chad", "Chile", "China",
        "Christmas Island", "Cocos (keeling) Islands", "Colombia", "Comoros", "Congo",
        "Congo, The Democratic Republic Of The", "Cook Islands", "Costa Rica", "Cote D'ivoire",
        "Croatia", "Cuba", "Cyprus", "Czech Republic", "Denmark", "Djibouti", "Dominica",
        "Dominican Republic", "East Timor", "Ecuador", "Egypt", "El Salvador", "Equatorial Guinea",
        "Eritrea", "Estonia", "Ethiopia", "Falkland Islands (malvinas)", "Faroe Islands", "Fiji",
        "Finland", "France", "French Guiana", "French Polynesia", "French Southern Territories",
        "Gabon", "Gambia", "Georgia", "Germany", "Ghana", "Gibraltar", "Greece", "Greenland",
        "Grenada", "Guadeloupe", "Guam", "Guatemala", "Guinea", "Guinea-bissau", "Guyana",
        "Haiti", "Heard Island And Mcdonald Islands", "Holy See (vatican City State)",
        "Honduras", "Hong Kong", "Hungary", "Iceland", "India", "Indonesia", "Iran, Islamic Republic Of",
        "Iraq", "Ireland", "Israel", "Italy", "Jamaica", "Japan", "Jordan", "Kazakstan", "Kenya",
        "Kiribati", "Korea, Democratic People's Republic Of", "Korea, Republic Of", "Kosovo", "Kuwait",
        "Kyrgyzstan", "Lao People's Democratic Republic", "Latvia", "Lebanon", "Lesotho", "Liberia",
        "Libyan Arab Jamahiriya", "Liechtenstein", "Lithuania", "Luxembourg", "Macau",
        "Macedonia, The Former Yugoslav Republic Of", "Madagascar", "Malawi", "Malaysia", "Maldives",
        "Mali", "Malta", "Marshall Islands", "Martinique", "Mauritania", "Mauritius", "Mayotte",
        "Mexico", "Micronesia, Federated States Of", "Moldova, Republic Of", "Monaco", "Mongolia",
        "Montserrat", "Montenegro", "Morocco", "Mozambique", "Myanmar", "Namibia", "Nauru", "Nepal",
        "Netherlands", "Netherlands Antilles", "New Caledonia", "New Zealand", "Nicaragua", "Niger",
        "Nigeria", "Niue", "Norfolk Island", "Northern Mariana Islands", "Norway", "Oman", "Pakistan",
        "Palau", "Palestinian Territory, Occupied", "Panama", "Papua New Guinea", "Paraguay", "Peru",
        "Philippines", "Pitcairn", "Poland", "Portugal", "Puerto Rico", "Qatar", "Reunion", "Romania",
        "Russian Federation", "Rwanda", "Saint Helena", "Saint Kitts And Nevis", "Saint Lucia",
        "Saint Pierre And Miquelon", "Saint Vincent And The Grenadines", "Samoa", "San Marino",
        "Sao Tome And Principe", "Saudi Arabia", "Senegal", "Serbia", "Seychelles", "Sierra Leone",
        "Singapore", "Slovakia", "Slovenia", "Solomon Islands", "Somalia", "South Africa",
        "South Georgia And The South Sandwich Islands", "Spain", "Sri Lanka", "Sudan", "Suriname",
        "Svalbard And Jan Mayen", "Swaziland", "Sweden", "Switzerland", "Syrian Arab Republic",
        "Taiwan, Province Of China", "Tajikistan", "Tanzania, United Republic Of", "Thailand",
        "Togo", "Tokelau", "Tonga", "Trinidad And Tobago", "Tunisia", "Turkey", "Turkmenistan",
        "Turks And Caicos Islands", "Tuvalu", "Uganda", "Ukraine", "United Arab Emirates",
        "United Kingdom", "United States", "United States Minor Outlying Islands", "Uruguay",
        "Uzbekistan", "Vanuatu", "Venezuela", "Viet Nam", "Virgin Islands, British",
        "Virgin Islands, U.s.", "Wallis And Futuna", "Western Sahara", "Yemen", "Zambia", "Zimbabwe" ]

    cities = [ "Addison", "Alexander City", "Alexandria", "Altoona", "Anderson", "Anniston", "Arab",
        "Ardmore", "Argo", "Arley", "Ashland", "Athens", "Atmore", "Avon", "Bay Minette", "Bear Creek",
        "Beaverton", "Benton", "Berry", "Bessemer", "Billingsley", "Black", "Blue Springs", "Boaz",
        "Bon Air", "Brantley", "Brent", "Brighton", "Brilliant", "Bynum", "Camp Hill", "Carbon Hill",
        "Cardiff", "Carolina", "Carrollton", "Castleberry", "Centreville", "Chalkville", "Chelsea",
        "Clayhatchee", "Clayton", "Cleveland", "Coker", "Concord", "Courtland", "Cowarts", "Creola",
        "Daviston", "Dayton", "Detroit", "Dodge City", "Double Springs", "Dozier", "Dutton", "East Brewton",
        "Eclectic", "Edwardsville", "Elba", "Elkmont", "Enterprise", "Epes", "Eutaw", "Evergreen",
        "Excel", "Fairfield", "Fairview", "Falkville", "Faunsdale", "Fayette", "Five Points", "Flomaton",
        "Florala", "Florence", "Foley", "Forestdale", "Forkland", "Fort Deposit", "Fort Payne", "Fort Rucker",
        "Fruithurst", "Fulton", "Fultondale", "Fyffe", "Gadsden", "Gainesville", "Gantt", "Gantts Quarry",
        "Garden City", "Geiger", "Geneva", "Geraldine", "Gilbertown", "Glencoe", "Glenwood", "Goldville",
        "Good Hope", "Gordo", "Gordon", "Gordonville", "Goshen", "Grand Bay", "Grant", "Grayson Valley",
        "Graysville", "Greensboro", "Greenville", "Grimes", "Grove Hill", "Guin", "Gulf Shores",
        "Guntersville", "Gurley", "Gu-Win", "Hackleburg", "Haleburg", "Haleyville", "Hamilton",
        "Hanceville", "Harpersville", "Hartselle", "Harvest", "Hayden", "Hazel Green", "Heath",
        "Heflin", "Helena", "Henagar", "Highland Lake", "Hokes Bluff", "Holly Pond", "Hollywood", "Holt",
        "Homewood", "Hoover", "Horn Hill", "Hueytown", "Huguley", "Hurtsboro", "Hytop", "Ider",
        "Indian Springs Village", "Jackson", "Jacksons' Gap", "Jasper", "Kansas", "Killen", "Kimberly",
        "Kinsey", "Kinston", "La Fayette", "Ladonia", "Lake View", "Lakeview", "Lanett", "Langston", "Leeds",
        "Leighton", "Lester", "Level Plains", "Lexington", "Libertyville", "Lincoln", "Linden", "Lipscomb",
        "Lisman", "Littleville", "Livingston", "Loachapoka", "Lockhart", "Locust Fork", "Louisville", "Loxley",
        "Lynn", "Macedonia", "Madison", "Madrid", "Malvern", "Margaret", "Marion", "Maytown", "McIntosh",
        "McKenzie", "McMullen", "Memphis", "Mentone", "Meridianville", "Midfield", "Midland City", "Mignon",
        "Millport", "Millry", "Minor", "Mobile", "Monroeville", "Montevallo", "Montgomery", "Moores Mill",
        "Mooresville", "Moulton" ]

    currencies = ['USD', 'EUR', 'GBP', 'RON', 'DKK', 'NOK', 'SEK', 'BGN', 'RUB', 'MDL']

    companies = [ "20th Century Fox", "23andme", "37signals", "3Com", "3M", "7-Eleven", "A&M Records",
        "A&P", "A&W Root Beer", "ABN AMRO", "Accenture", "Acer", "Adecco", "Adidas", "Adobe Systems",
        "Ahold", "Ahlstrom", "Akai", "Akamai", "AKZO", "AKG Acoustics", "Alcatel-Lucent", "Alcoa", "Aldi",
        "Alfa Romeo", "Alstom", "AltaVista", "ALZA", "Amazon.com", "AmBev", "AMC Theatres", "AMD",
        "AMKOR", "Amiga Corporation", "Amoco", "Amstrad", "Anheuser-Busch InBev", "AOL", "Apache", "Apple",
        "Apricot Computers", "Arby's", "Arcelor", "AREVA", "Aricent", "ARM Limited", "Arm & Hammer",
        "ARP", "Artis", "Asda", "ASICS", "Ask.com", "Asus", "Aston Martin", "AT&T", "Atari", "ATI",
        "ATS", "Audi", "B&Q", "Bang & Olufsen", "Bally", "Banesto", "BAPE", "BASF", "Bauknecht",
        "Bayer", "BBC", "BBVA", "BCC Research", "BEA Systems", "Ben & Jerry's", "BenQ", "BHP",
        "BIC Corporation", "Black & Decker", "Blaupunkt", "BMW", "Boeing", "Bosch", "Bose Corporation",
        "BSNL", "BP", "BRAC", "Bridgestone", "Brine, Corp.", "BT", "Bull", "Burroughs Corporation", "Bultaco",
        "BHEL", "CA", "C&A", "Cadillac", "CAE", "Campagnolo", "Canon", "Caprabo", "Carrefour", "Caterpillar",
        "Cathay Pacific Airways Limited", "Casio", "CBS", "Celera", "CGI Group", "Chevrolet", "Chello", "Chrysler",
        "Ciba Geigy", "CiCi's Pizza", "Cigna", "Cincom", "Cisco", "Citroën", "Coca-Cola", "Coleco", "Colgate-Palmolive",
        "COLT", "Comcast", "Compaq", "COMSAT", "ConocoPhillips", "Copersucar", "Corel", "Cosworth", "CPFL",
        "Crabtree & Evelyn", "Cray", "CRC Press", "Cromemco", "Cutco", "CVS", "Daewoo", "DAF Trucks", "Daihatsu",
        "Danone", "Datsun", "Debian", "DEC", "DEKA", "Delhaize", "Dell", "Denning & Fourcade, Inc.", "DHL",
        "Digg", "Digi-Key", "The Walt Disney Company, named for its co-founder Walt Disney.", "Dixons", "DKNY",
        "Dow", "Duane Reade", "Dynegy", "EA Games", "eBay", "EDS", "Eidos", "Eletropaulo", "Embraer",
        "EMBRAPA", "EMBRATEL", "EMC Corporation", "EMI", "Emporis", "Equifax", "ESPN", "ESRI", "Epson",
        "Esso", "Exxon", "FAS", "Facebook", "Fair Isaac Corporation", "Fazer", "FCUK", "FedEx",
        "Fegime", "Ferrari", "Fiat", "Finnair", "Firestone", "Five Guys", "Fluke", "Ford Motor Company",
        "Forrester Research", "FranklinCovey", "Fuji", "Garmin", "Gartner", "Gatti's Pizza", "GCap Media",
        "Genentech", "GEICO", "Glaxo", "Glock GmbH", "Goodyear", "Google", "Grey Global Group", "Grundig",
        "Gulfstream Aerospace", "Häagen-Dazs", "Haier", "H&M", "Haribo", "Harman Kardon", "Harpo Productions",
        "Hasbro", "HBOS", "HCL", "Hess Corporation", "HP", "Hispano-Suiza", "Hitachi", "HMV", "Hoechst", "Honda",
        "Honeywell", "Hospira", "Hotmail", "H&R Block", "HSBC", "HTC Corporation", "Hyundai", "IBM", "ICICI",
        "ICL", "IG Farben", "Iiyama", "IKEA", "InBev", "Inditex", "Infineon Technologies", "Ingenico", "Intel",
        "Ittiam Systems", "Infosys", "J2TV", "JAL", "Jat Airways", "JBL", "Johnson & Johnson", "JVC", "Kalev",
        "Kawasaki", "KFC", "Kenwood Limited", "Kenworth Truck Company", "Kia Motors", "Kinko's", "Kodak", "Komatsu",
        "Konica", "Korg", "KPMG", "Kroger", "KUKA", "Kyocera", "Lada", "Lancôme", "LCL", "Lego", "Lenovo Group",
        "Level 3 Communications", "LG", "Lexmark", "Lionbridge", "Lionhead Studios", "Lockheed Martin", "LoJack",
        "Longines", "Lonsdale", "L'Oréal", "LOT", "Lotus Software", "Lucent Technologies", "Lukoil", "Lycos",
        "Maggi", "MAN", "Mandriva", "Manhattan Associates", "Manugistics", "Manulife Financial", "Mars",
        "Masco Corporation", "Mast-Jägermeister AG", "Mattel", "Maybach-Motorenbau GmbH", "Mazda Motor Corporation",
        "MBNA", "McDonald's", "MCI Communications", "Mercedes", "Merillat Industries", "Metro-Goldwyn-Mayer (MGM)",
        "MFI", "MG Cars", "Microlins", "Micron Technology", "Microsoft", "Midway Games", "Mincom Limited", "Minolta",
        "MIPRO", "MIPS", "MITIE", "Mitel", "MITRE", "Mitsubishi", "Morningstar, Inc.", "Motorola", "Mozilla Foundation",
        "MVC", "Mustek", "MRF", "Nabisco", "NCR Corporation", "NEC", "Nero", "Nestlé", "Netscape", "Nike",
        "Nikon", "Nintendo", "Nissan", "Nokia", "Nortel Networks", "Novartis", "Novell", "OCZ", "Oracle", "Ornge",
        "Osram", "Paccar", "PCCW", "Pamida", "Pemex", "Pennzoil", "Pepsi", "Petrobras", "Philco", "Philips", "Pixar",
        "PMC-Sierra", "Porsche", "Prada", "Procter & Gamble", "ProfSat", "PRS Guitars", "Psion", "Q8", "Qantas",
        "Qimonda", "Quad", "Quark", "Qualcomm", "QVC", "Rabobank", "RAND", "Raytheon", "RCA", "Reckitt & Colman",
        "Reckitt Benckiser", "Red Hat", "Reebok", "REO Motor Car Company", "Repsol", "Research In Motion", "Rickenbacker",
        "Robeez", "Rolls-Royce", "RSA Security", "SAAB", "Sabre", "Saku Brewery", "Samsonite", "Samsung", "Sanyo",
        "SAP", "SAS", "SAS Institute", "Sasol", "SCB", "SCO", "Saudi Aramco", "SEAT", "Sealed Air", "Sega", "Seiko",
        "Sennheiser", "setcom", "SGI", "Sharp", "Shell", "Siemens", "Six Apart", "Skanska", "SKF", "Skoda Auto",
        "Skype", "Smart", "Smilebit", "Smeg", "SNK", "Sony", "Sorcim", "SPAR", "Sperry", "Spiratone", "Sprint",
        "SRAM Corporation", "SRI International", "Stanley Works", "Starbucks", "Stellent", "STX", "Subaru",
        "Sun Microsystems", "SuSE", "Suzuki", "Taco Bell", "Talgo",
        "TAM Airlines", "TAP Portugal", "Tata Group", "Taxan", "TCBY", "TCL", "TCS", "TDK Corporation", "Tesco",
        "Teva Naot", "Texaco", "THX", "TIBCO Software", "Tim Hortons", "TNT N.V.", "Toshiba", "Toyota", "Triang",
        "Tucows", "TVR", "Twinings", "Twitter", "Ubuntu Foundation", "Umbro", "Unilever", "UNIMED", "Unisys",
        "Unocal Corporation", "UPS", "UUNET", "Vaisala", "Valtra", "Varig", "Verizon", "VersatileCS[1]", "Virgin",
        "Vodafone", "Volkswagen", "Volvo", "Wachovia", "Waitrose", "Wal-Mart", "Wang Laboratories", "Wells Fargo",
        "Wendy's", "Weta Digital", "W H Smith", "Williams-Sonoma", "Wipro", "WWE", "Worlds of Wonder", "WPP", "Xerox",
        "Yahoo!", "YKK", "Yakult", "Yoplait", "Zend Technologies", "Zuse" ]

    # List of LEADINs to buy time
    leadins = """To characterize a linguistic level L,
        On the other hand,
        This suggests that
        It appears that
        Furthermore,
        We will bring evidence in favor of the following thesis:
        To provide a constituent structure for T(Z,K),
        From C1, it follows that
        For any transformation which is sufficiently diversified in application to be of any interest,
        Analogously,
        Clearly,
        Note that
        Of course,
        Suppose, for instance, that
        Thus
        With this clarification,
        Conversely,
        We have already seen that
        By combining adjunctions and certain deformations,
        I suggested that these results would follow from the assumption that
        If the position of the trace in (99c) were only relatively inaccessible to movement,
        However, this assumption is not correct, since
        Comparing these examples with their parasitic gap counterparts in (96) and (97), we see that
        In the discussion of resumptive pronouns following (81),
        So far,
        Nevertheless,
        For one thing,
        Summarizing, then, we assume that
        A consequence of the approach just outlined is that
        Presumably,
        On our assumptions,
        It may be, then, that
        It must be emphasized, once again, that
        Let us continue to suppose that
        Notice, incidentally, that """

    # List of SUBJECTs chosen for maximum professorial macho
    subjects = """ the notion of level of grammaticalness
        a case of semigrammaticalness of a different sort
        most of the methodological work in modern linguistics
        a subset of English sentences interesting on quite independent grounds
        the natural general principle that will subsume this case
        an important property of these three types of EC
        any associated supporting element
        the appearance of parasitic gaps in domains relatively inaccessible to ordinary extraction
        the speaker-hearer's linguistic intuition
        the descriptive power of the base component
        the earlier discussion of deviance
        this analysis of a formative as a pair of sets of features
        this selectionally introduced contextual feature
        a descriptively adequate grammar
        the fundamental error of regarding functional notions as categorial
        relational information
        the systematic use of complex symbols
        the theory of syntactic features developed earlier"""

    #List of VERBs chosen for autorecursive obfuscation
    verbs = """can be defined in such a way as to impose
        delimits
        suffices to account for
        cannot be arbitrary in
        is not subject to
        does not readily tolerate
        raises serious doubts about
        is not quite equivalent to
        does not affect the structure of
        may remedy and, at the same time, eliminate
        is not to be considered in determining
        is to be regarded as
        is unspecified with respect to
        is, apparently, determined by
        is necessary to impose an interpretation on
        appears to correlate rather closely with
        is rather different from"""

    # List of OBJECTs selected for profound sententiousness
    objects = """ problems of phonemic and morphological analysis.
        a corpus of utterance tokens upon which conformity has been defined by the paired utterance test.
        the traditional practice of grammarians.
        the levels of acceptability from fairly high (e.g. (99a)) to virtual gibberish (e.g. (98d)).
        a stipulation to place the constructions into these various categories.
        a descriptive fact.
        a parasitic gap construction.
        the extended c-command discussed in connection with (34).
        the ultimate standard that determines the accuracy of any proposed grammar.
        the system of base rules exclusive of the lexicon.
        irrelevant intervening contexts in selectional rules.
        nondistinctness in the sense of distinctive feature theory.
        a general convention regarding the forms of the grammar.
        an abstract underlying order.
        an important distinction in language use.
        the requirement that branching is not tolerated within the dominance scope of a complex symbol.
        the strong generative capacity of the theory."""


    def __init__(self, session, min_lat, min_lng, max_lat, max_lng):
        self.session = session
        self.min_lat = min_lat
        self.min_lng = min_lng
        self.max_lat = max_lat
        self.max_lng = max_lng

        self.cache_subdivision = None
        self.cache_localities = None
        self.lexicon_cache = {}


    def get_subdivision(self):
        if not self.cache_subdivision:
            self.cache_subdivision = {}
            max_level = self._query("SELECT MAX(level) FROM t_adreglev")
            obs = self.session.query(Subdivision).filter_by(level=max_level).all()
            for ob in obs:
                self.cache_subdivision[ob.code] = ob
        keys = self.cache_subdivision.keys()
        key = keys[rand_int(0, len(keys) - 1)]
        return self.cache_subdivision[key].code


    def get_subdivision_tuple(self):
        ret = [None, None, None, None, None]
        subdivision = self.get_subdivision()
        max_level = self._query("SELECT MAX(level) FROM t_adreglev")
        level_column_name = 'level%s' % max_level
        level_column = getattr(SubdivisionView, level_column_name)
        row = self.session.query(SubdivisionView).filter(level_column==subdivision).one()
        ret[0] = row.level1
        ret[1] = row.level2
        ret[2] = row.level3
        ret[3] = row.level4
        ret[4] = row.level5
        return ret

    def get_locality(self):
        if not self.cache_localities:
            self.cache_localities = {}
            obs = self.session.query(Locality).all()
            for ob in obs:
                self.cache_localities[ob.adlocode] = ob
        keys = self.cache_localities.keys()
        if len(keys):
            key = keys[rand_int(0, len(keys) - 1)]
            return self.cache_localities[key].adlocode
        return None

    def get_lexicon(self, name):
        lexicon = {}
        if name in self.lexicon_cache.keys():
            lexicon = self.lexicon_cache[name]

        if not lexicon.keys():
            self.lexicon_cache[name] = {}
            klass = lexicon_list[name]['table_class']
            obs = self.session.query(klass).all()
            for ob in obs:
                lexicon[ob.code] = ob
            self.lexicon_cache[name] = lexicon
            print '* Filled lexicon cache for : %s' % name

        key = lexicon.keys()[rand_int(0, len(lexicon.keys()) - 1)]
        if lexicon.has_key(key):
            return lexicon[key].code
        return None


    def _mk_company(self):
        return '%s %s' % (self.companies[rand_int(0, len(self.companies) - 1)],
                          self.companies[rand_int(0, len(self.companies) - 1)])

    def _mk_random(self):
        dt = datetime.datetime.now()
        return int(mktime(dt.timetuple()) + random.randint(10000, 999999999))


    def _mk_word(self, dictionary = objects):
        wordslist = []
        words = dictionary.replace('\n', ' ').replace('.', ' ').split(' ')
        for word in words:
            if len(word.strip()) > 3 and not word.strip().startswith('('):
                wordslist.append(word.strip())
        return wordslist[rand_int(0, len(wordslist) - 1)]


    def _mk_randomlist(self, list):
        return list[rand_int(0, len(list) - 1)]


    def _mk_date(self, year=None, month=None, day=None):
        lyear = year
        dt = datetime.datetime.now()
        if not year:
            lyear = rand_int(1991, dt.year)
        lmonth = month
        if not month:
            lmonth = rand_int(1, 12)
        lday = day
        if not day:
            lday = rand_int(1, 28)
        return datetime.date(lyear, lmonth, lday)


    def _mk_bool(self):
        return rand_int(0, 1) == True


    def _mk_username(self):
        users = ['cornel', 'cristiroma', 'johnb', 'chris', 'sawyer', 'mirabela', 'alexys', 'porbal', 'satyr', 'styx', 'apophis']
        return users[rand_int(0, len(users) - 1)]


    def _mk_latitude(self):
        return random.uniform(self.min_lat, self.max_lat)

    def _mk_float(self, max=100, min=10):
        return min + (max - min) * random.random()

    def _mk_longitude(self):
        return random.uniform(self.min_lng, self.max_lng)

    def _mk_sentence(self):
        txt = self._chomsky()
        le = len(txt)
        if le > 72:
            le = 72
        return txt[:le]

    def _chomsky(self, times=1, line_length=72):
        parts = []
        for part in (self.leadins, self.subjects, self.verbs, self.objects):
            phraselist = map(str.strip, part.splitlines())
            random.shuffle(phraselist)
            parts.append(phraselist)
        output = chain(*islice(izip(*parts), 0, times))
        return textwrap.fill(' '.join(output), line_length)


    def _query(self, sql, multiple=False):
        conn = self.session.connection()
        ret = None
        rs = conn.execute(sql)
        if multiple:
            ret = rs.fetchone()
        else:
            ret = rs.fetchone()[0]
        return ret


    def _query_rows(self, sql):
        conn = self.session.connection()
        ret = []
        rs = conn.execute(sql).fetchall()
        for ob in rs:
            ret.append(ob[0])
        return ret


    def _mk_upis(self, upiscode, upistype, approved, contributed):
        """ insert a record in table t_upis """
        ob = Upis()
        ob.upiscode = upiscode
        ob.upisadlocode = self.get_locality()

        subdivision = self.get_subdivision_tuple()
        ob.upisadr1code = subdivision[0]
        ob.upisadr2code = subdivision[1]
        ob.upisadr3code = subdivision[2]
        ob.upisadr4code = subdivision[3]
        ob.upisadr5code = subdivision[4]

        ob.upiscodex = 'UPISCODEX'
        ob.upistype = upistype
        ob.upisname = self._mk_sentence()
        ob.upisname2 = self._mk_sentence()
        ob.upisplace = self._mk_randomlist(self.cities)
        ob.upisloctype = self.get_lexicon('LexUpisloctype')
        ob.upisx = self._mk_longitude()
        ob.upisy = self._mk_latitude()
        ob.upisz = rand_int(0, 1500)
        ob.upisyear = rand_int(1999, 2010)
        ob.upisobj = self.get_lexicon('LexWP051')
        ob.upisdend = self._mk_date()
        ob.upistdesc = self._chomsky()
        ob.upiscomment = self._chomsky()
        ob.upismanprof = self.get_lexicon('LexUpisManProf')
        ob.upisowner = None
        ob.upisprogram = None
        ob.upisfinance = self.get_lexicon('LexUpisFinance')
        ob.approved = approved
        ob.upisupdate = self._mk_date()
        ob.upisupwho = self._mk_username()
        ob.upisowner = self._mk_username()
        ob.contributed = contributed
        self.session.add(ob)


    def _mk_upwp(self, upiscode):
        """ insert a record in table t_upwp """
        ob = Waterpoint()
        ob.upiscode = upiscode
        ob.upwpcodex = 'UPWPCODEX'
        ob.upwptype = self.get_lexicon('LexUpisloctype')
        ob.upwpbench = self.get_lexicon('LexWP017')
        ob.upwpmainuse = self.get_lexicon('LexWP009')
        ob.upwpantiq = self._mk_bool()
        ob.upwpdiam = self._mk_float()
        ob.upwpdepth = self._mk_float()
        ob.upwpcoph = self._mk_float()
        ob.upwpcatch = self._mk_float()
        ob.upwprain = self._mk_float()
        ob.upwpnotfr = self.get_lexicon('LexWP052')
        ob.upwpnotfd = self.get_lexicon('LexWP053')
        ob.upwdlfdate = None
        ob.upwpasscw = self.get_lexicon('LexWP055')
        ob.upwpasspl = self.get_lexicon('LexWP056')
        ob.upwpasshy = self.get_lexicon('LexWP057')
        ob.upwpfence = self.get_lexicon('LexWP058')
        ob.upwpwturb = self.get_lexicon('LexWP059')
        ob.upwpwtast = self.get_lexicon('LexWP060')
        ob.upwpwodor = self.get_lexicon('LexWP061')
        ob.upwptreat = self.get_lexicon('LexWP062')
        ob.upwptreatf = self.get_lexicon('LexWP063')
        ob.upwppaymd = self.get_lexicon('LexWP064')
        ob.upwppayseas = self.get_lexicon('LexWP065')
        ob.upwphptype = self.get_lexicon('LexWP066')
        ob.upwphpfunc = self.get_lexicon('LexWP067')
        # ob.upwphpiwho
        # ob.upwphpcraf
        ob.upwpmeter = self.get_lexicon('LexWP071')
        ob.upwpyield = self._mk_float()
        # ob.upwpprero =
        ob.upwpprpol = self.get_lexicon('LexWP074')
        ob.upwpdry = self.get_lexicon('LexWP075')
        ob.upwpclos = self.get_lexicon('LexWP076')
        # ob.upwpabsq =
        ob.upwpwlev = self._mk_float()
        ob.upwpeprice = self._mk_float()
        ob.upwpesaleu = 'l'
        self.session.add(ob)


    def _mk_upsf(self, upiscode):
        """ insert a record in table t_upsf """
        ob = Sanitation()
        ob.upiscode = upiscode
        ob.upsftype = self.get_lexicon('LexSF006')
        ob.upsfmatype = self.get_lexicon('LexSF012')
        ob.upsfgrtype = self.get_lexicon('LexSF013')
        ob.upsfmwsep = self._mk_bool()
        ob.upsfblocnbr = rand_int(1, 10)
        ob.upsfblomnbr = rand_int(1, 5)
        ob.upsfblownbr = rand_int(1, 4)
        ob.upsfshower = self._mk_bool()
        ob.upsfwhand = self._mk_bool()
        ob.upsfwhand = self._mk_bool()
        ob.upsfwater = self.get_lexicon('LexSF021')
        ob.upsffunc = self.get_lexicon('LexSF050')
        ob.upsfnotfr = self.get_lexicon('LexSF051')
        ob.upsfnotfd = self.get_lexicon('LexSF052')
        ob.upsfasscw = self.get_lexicon('LexSF053')
        ob.upsfasspl = self.get_lexicon('LexSF054')
        ob.upsfasscl = self.get_lexicon('LexSF055')
        ob.upsfpaymd = self.get_lexicon('LexSF056')
        self.session.add(ob)


    def _mk_uppws(self, upiscode):
        """ insert a record in table t_upsf """
        ob = pws_models.PWS()
        ob.upiscode = upiscode
        ob.uppwsowntype = self.get_lexicon('LexPWS005')
        ob.uppwsctrar = self.get_lexicon('LexPWS007')

        ob.uppwspop = rand_int(100, 9999)
        ob.uppwslonbr = rand_int(1, 30)
        ob.uppwsstcpdi = self._mk_float()
        ob.uppwsminhd = self._mk_float()
        ob.uppwsstcp = self._mk_float()
        ob.uppwslentot = self._mk_float()
        ob.uppwstapnbr = rand_int(10, 40)
        ob.uppwsconnbr = rand_int(10, 40)
        ob.uppwscapnbr = rand_int(10, 40)
        ob.uppwscatnbr = rand_int(10, 40)
        ob.uppwswdpnbr = rand_int(10, 40)
        ob.uppwsfunc = self.get_lexicon('LexPWS020')
        ob.uppwsabdm = self._mk_float()
        ob.uppwscapday = self._mk_float()
        ob.uppwscapdr = self._mk_float()
        ob.uppwsenerc = self.get_lexicon('LexPWS024')
        ob.uppwsextent = self.get_lexicon('LexPWS025')
        ob.uppwsschlen = rand_int(10, 40)
        ob.uppwsabstpe = self.get_lexicon('LexPWS027')
        ob.uppwschldis = self._mk_bool()
        ob.uppwstpsys = self._mk_bool()
        ob.uppwspricept = self._mk_float()
        ob.uppwspricedt = self._mk_float()
        ob.uppwspriceos = self._mk_float()
        ob.uppwspriceun = 'US Dollar'
        self.session.add(ob)

    #Depends on water_points_create
    def _mk_comments(self, upiscode):
        """ insert a random number of record in table t_upiscomments """
        now = datetime.datetime.now()
        for i in range(0, random.randrange(1, 20)):
            user = ''
            user_name = self._mk_username()
            user_email = '%s@domain.com' % user_name
            comment = self._chomsky()
            submit_date = self._mk_date(now.year - rand_int(1, 3))
            rating = rand_int(1, 5)
            ob = UpisComment(code = upiscode,
                             user = '', user_name = user_name, user_email = user_email,
                             submit_date = submit_date, rating = rating, comment = comment,
                             ip_address = '127.0.0.1', is_public = True, is_removed = False)
            self.session.add(ob)

    def generate_demo_users(self, portal_ob):
        """ add test user accounts """
        print 'Creating test user accounts. The passwords are identical with the usernames'
        auth_tool = portal_ob.getAuthenticationTool()
        auth_tool.manage_addUser('auth', 'auth', 'auth', ['Authenticated'], [], 'Authenticated', 'User', 'auth@mailinator.com')
        auth_tool.manage_addUser('editor', 'editor', 'editor', ['Editor'], [], 'Editor', 'User', 'editor@mailinator.com')
        auth_tool.manage_addUser('pm', 'pm', 'pm', ['Program manager'], [], 'Program manager', 'User', 'pm@mailinator.com')
        auth_tool.manage_addUser('wm', 'wm', 'wm', ['Waterpoint manager'], [], 'Waterpoint manager', 'User', 'wm@mailinator.com')
        auth_tool.manage_addUser('agent', 'agent', 'agent', ['District/municipal agent'], [], 'District/municipal agent', 'User', 'agent@mailinator.com')
        auth_tool.manage_addUser('wss', 'wss', 'wss', ['WSS services agent'], [], 'WSS services agent', 'User', 'wss@mailinator.com')
        auth_tool.manage_addUser('admin', 'admin', 'admin', ['Administrator'], [], 'Administrator', 'User', 'admin@mailinator.com')
        print 'Done creating test user accounts'


    def generate_water_points(self, no):
        print '[t_upis, t_upwp, t_upwpcomments] Creating %s water points with test comments' % no

        seed = self._mk_random()
        for i in range(0, no):
            approved = self._mk_bool()
            contributed = self._mk_bool()
            upiscode = 'TESTWP%s' % (seed + i)
            self._mk_upis(upiscode, UpistypeEnum.WP, approved, contributed)
            self.session.flush()
            self._mk_upwp(upiscode)
            self.session.flush()
            self._mk_comments(upiscode)
            if i % 50 == 0:
                print '        * Generated %s water points' % i
        c = self.session.connection()
        c.execute("UPDATE t_upis SET upiscoordinate = ST_SetSRID(ST_MakePoint(upisx, upisy), 4326);")
        print 'Done generating water points'


    def generate_sanitation_facilities(self, no):
        print '[t_upis, t_upsf, t_upsfcomments] Creating %s sanitation facilities with test comments' % no

        seed = self._mk_random()
        for i in range(0, int(no)):
            approved = self._mk_bool()
            contributed = self._mk_bool()
            upiscode = 'TESTSF%s' % (seed + i)
            self._mk_upis(upiscode, UpistypeEnum.SF, approved, contributed)
            self.session.flush()
            self._mk_upsf(upiscode)
            self.session.flush()
            self._mk_comments(upiscode)
            if i % 50 == 0:
                print '        * Generated %s sanitation facilities' % i
        c = self.session.connection()
        c.execute("UPDATE t_upis SET upiscoordinate = ST_SetSRID(ST_MakePoint(upisx, upisy), 4326);")
        print 'Done generating sanitation facilities'


    def generate_piped_water_schemes(self, no):
        print '[t_upis, t_uppws, t_uppwscomments] Creating %s piped water schemes with test comments' % no

        seed = self._mk_random()
        for i in range(0, int(no)):
            approved = self._mk_bool()
            contributed = self._mk_bool()
            upiscode = 'TESTPWS%s' % (seed + i)
            self._mk_upis(upiscode, UpistypeEnum.PWS, approved, contributed)
            self.session.flush()
            self._mk_uppws(upiscode)
            self.session.flush()
            self._mk_comments(upiscode)
            if i % 50 == 0:
                print '        * Generated %s water schemes' % i

            # Add energy sources (PWSEnergy)
            for i in range(0, random.randint(0, 5)):
                en = pws_models.PWSEnergy()
                en.enisupiscode = upiscode
                en.uppensrc = self.get_lexicon('LexPWE003')
                en.uppenet = None
                en.uppenrp = self._mk_float()
                en.uppenup = self._mk_float()
                en.uppenstart = self._mk_date()
                self.session.add(en)

            # Add pipes (PWSPipe)
            for i in range(0, random.randint(0, 8)):
                pi = pws_models.PWSPipe()
                pi.piisupiscode = upiscode
                pi.upppicp = self.get_lexicon('LexPWI003')
                pi.upppimdp = None
                pi.upppilen = self._mk_float()
                pi.upppistart = self._mk_date()
                self.session.add(pi)

            # Add pumps (PWSPump)
            for i in range(0, random.randint(0, 3)):
                pu = pws_models.PWSPump()
                pu.ppisupiscode = upiscode
                pu.uppphyi = None
                pu.uppppws = self.get_lexicon('LexPWP004')
                pu.upppppump = self.get_lexicon('LexPWP005')
                pu.uppppbtp = None
                pu.uppppenr = self.get_lexicon('LexPWP007')
                pu.uppppoy = self._mk_float()
                pu.uppppotmh = self._mk_float()
                pu.uppppidp = self._mk_float()
                pu.uppppdwtoy = self._mk_float()
                pu.uppppstart = self._mk_date()
                self.session.add(pu)

            # Add reservoirs (PWSReservoir)
            for i in range(0, random.randint(0, 3)):
                res = pws_models.PWSReservoir()
                res.stisupiscode = upiscode
                res.uppstcr = self.get_lexicon('LexPWSS003')
                res.uppstbm = self.get_lexicon('LexPWSS004')
                res.uppsttc = self._mk_float()
                res.uppsthbt = self._mk_float()
                res.uppstminh = self._mk_float()
                res.uppstx = self._mk_longitude()
                res.uppsty = self._mk_latitude()
                res.uppststart = self._mk_date()
                self.session.add(res)

            # Add water treatment (PWSWaterTreatment)
            for i in range(0, random.randint(0, 3)):
                wt = pws_models.PWSWaterTreatment()
                wt.ttisupiscode = upiscode
                wt.upptttype = self.get_lexicon('LexPWT002')
                wt.uppttdesc = self._mk_sentence()
                wt.uppttequip = None
                self.session.add(wt)

            # Add water points to this water scheme
            if self.wp_count():
                sql = 'SELECT upiscode FROM view_water_point ORDER BY RANDOM() LIMIT %s' % random.randint(1, 10)
                wps = self._query_rows(sql)
                for wp in wps:
                    pwswp = pws_models.Tl_PWSWP()
                    pwswp.pwsisupiscode = upiscode
                    pwswp.wpisupiscode = wp
                    self.session.add(pwswp)
            
            #Add water quality reports
            for i in range(0, random.randint(0, 5)):
                wq = WaterQuality()
                wq.wqmsamplecode = upiscode
                wq.wqmreference = self._mk_sentence()
                wq.wqmoperator = self._mk_sentence()
                wq.wqmdate = self._mk_date()
                wq.wqmconformity = self._mk_bool()
                wq.wqmcomment = self._mk_sentence()
                self.session.add(wq)
                for i in range(0, random.randint(0, 5)):
                    wq_data = WaterQualityData()
                    wq_data.wqmdatreportcode = wq.wqmcode
                    wq_data.wqmdatparameter = self.get_lexicon('LexWQD003')
                    wq_data.wqmdatvalue = self._mk_float()
                    wq_data.wqmdatconformity = self._mk_bool()
                    self.session.add(wq_data)

            else:
                # print 'No water points! Cannot assign water points to water scheme!'
                pass


            # Add localities linked to this water point
            if self.loc_count():
                sql = 'SELECT adlocode FROM t_adloc ORDER BY RANDOM() LIMIT %s' % random.randint(5, 20)
                localities = self._query_rows(sql)
                for loc in localities:
                    pwsloc = pws_models.Tl_Aulocpws()
                    pwsloc.pwsisupiscode = upiscode
                    pwsloc.loadlocode = loc
                    self.session.add(pwsloc)
            else:
                # print 'No localities! Cannot assign localities to water scheme!'
                pass

            self.session.flush()

        c = self.session.connection()
        c.execute("UPDATE t_upis SET upiscoordinate = ST_SetSRID(ST_MakePoint(upisx, upisy), 4326);")
        c.execute("UPDATE t_uppwsst SET uppstcoordinate = ST_SetSRID(ST_MakePoint(uppstx, uppsty), 4326);")
        print 'Done generating piped water schemes'



    def generate_organisations(self, max):
        print '[t_adorga] Creating %s organisations ...' % max
        seed = self._mk_random()
        for i in range(0, max):
            ob = Organisation()
            ob.adorname = self._mk_company()
            ob.adoracronym = 'Acronym #%s' % (seed + i)
            ob.adortype = self.get_lexicon('LexDRY004')
            ob.adoradress = self._chomsky(times=1, line_length=72)[0:15]
            ob.adorzipcode = '%s' % rand_int(10000, 99999)
            ob.adorcity = self._mk_randomlist(self.cities)
            ob.adorcountry = self._mk_randomlist(self.countries)
            ob.adortel = rand_int(130097837433, 130098837433)
            ob.adorfax = rand_int(130097837433, 130098837433)
            ob.adoremail = '%s@%s.com' % (self._mk_word(), self._mk_word())
            ob.adorwebsite = 'http://%s.com' % (ob.adorname.replace(' ', '-'))
            ob.adorlogo = None
            ob.adorpres = self._chomsky()
            ob.adorhqcode = self.get_locality()
            ob.adorhqx = self._mk_longitude()
            ob.adorhqy = self._mk_longitude()
            ob.adorupwho = self._mk_username()
            ob.adorupdate = datetime.datetime.now()

            self.session.add(ob)
        c = self.session.connection()
        c.execute("UPDATE t_adorga SET adorgacoordinate = ST_SetSRID(ST_MakePoint(adorhqx, adorhqy), 4326);")
        print 'Done generating organizations'


    # Depends on create_organisations
    def generate_programs(self, portal_ob, max=20):
        print '[t_adope] Creating %s projects ...' % max
        seed = self._mk_random()
        for i in range(0, max):
            try:
                prog = Program()
                prog.eaadorcode = self._query("SELECT adorcode FROM t_adorga ORDER BY RANDOM() LIMIT 1")
                prog.iaadorcode = self._query("SELECT adorcode FROM t_adorga ORDER BY RANDOM() LIMIT 1")
                prog.adopname = 'Project test no. %s' % (seed + i)
                prog.adopacron = 'Prj. #%s' % i
                prog.start_year = rand_int(1991, 2010)
                prog.adopdstart = self._mk_date(prog.start_year)
                prog.adopdend = self._mk_date(prog.start_year + rand_int(1, 12))
                prog.adopctpre = self._mk_bool()
                prog.adopctfea = self._mk_bool()
                prog.adopctcb = self._mk_bool()
                prog.adopctiec = self._mk_bool()
                prog.adopctidev = self._mk_bool()
                prog.adopctass = self._mk_bool()
                prog.adopctme = self._mk_bool()
                prog.adopctother = self._chomsky()
                prog.adopctobj = self._chomsky()
                prog.adopresexp =self._chomsky()
                prog.adopactiv = self._chomsky()
                prog.adopbudtot = rand_int(294398323, 2294398323)
                prog.adopcontact = 'Contact person #%s' % i
                prog.adopurl = 'http://%s-%s.com' % (self._mk_word(), self._mk_word())
                prog.adoptgpop = rand_int(300, 12000)
                prog.adoptglocn = rand_int(2, 120)
                prog.adoptgwpn = rand_int(10, 4500)
                prog.adoptgrpw = rand_int(100, 450)
                prog.adoptgwcn = rand_int(10, 99)
                prog.adoptgsfpn = rand_int(10, 99)
                prog.adoptgsfin = rand_int(10, 99)
                prog.adopdate = datetime.date.today()

                self.session.add(prog)
                self.session.flush()

                if not hasattr(portal_ob.programs, '%s' % prog.adopcode):
                    addNyFolder(portal_ob.programs, id='%s' % prog.adopcode, callback=create_wsportalprogram_object_callback, title=prog.adopname)

                # Add funding
                used_adorcode = []
                for i in range(1, rand_int(1, 6)):
                    adopfcode = self._query("SELECT adorcode FROM t_adorga ORDER BY RANDOM() LIMIT 1")
                    if not adopfcode in used_adorcode:
                        bud = Budget()
                        bud.opfadopcode = prog.adopcode
                        bud.adopfcode = adopfcode
                        bud.adopfamou = rand_int(12343, 234934)
                        bud.adopfmunit = self._mk_randomlist(self.currencies)
                        bud.adopfcat = self.get_lexicon('LexOpb005')
                        self.session.add(bud)
                        used_adorcode.append(adopfcode)
                print '        * Added project budget'

                # Add monitoring data
                adopmexpen = rand_int(984738, 1397438)
                adopmdisbu = adopmexpen - rand_int(20943, 99832)
                adopmpopws = rand_int(100, 3432)
                adopmpopsf = rand_int(100, 3432)
                adopmlocws = rand_int(10, 99)
                adopmpwpnbr = rand_int(12, 55)
                adopmwprnbr = rand_int(1, 22)
                adopmwcnbr = rand_int(12, 400)
                adopmsfpnbr = rand_int(12, 400)
                adopmsfinbr = rand_int(55, 400)
                for year in range(prog.adopdstart.year, prog.adopdend.year + 1):
                    adopmexpen = adopmexpen + rand_int(1000, 9999)
                    adopmdisbu = adopmdisbu + rand_int(1000, 15000)
                    adopmpopws = adopmpopws + rand_int(10, 341)
                    adopmpopsf = adopmpopsf + rand_int(10, 341)
                    adopmlocws = adopmlocws + rand_int(1, 9)
                    adopmpwpnbr = adopmpwpnbr + rand_int(1, 12)
                    adopmwprnbr = adopmwprnbr + rand_int(1, 12)
                    adopmwcnbr = adopmwcnbr + rand_int(1, 40)
                    adopmsfpnbr = adopmsfpnbr + rand_int(1, 40)
                    adopmsfinbr = adopmsfinbr + rand_int(5, 40)

                    progress = ProgressData()
                    progress.opmadopcode = prog.adopcode
                    progress.adopmdate = self._mk_date(year)
                    progress.adopmexpen = adopmexpen
                    progress.adopmdisbu = adopmdisbu
                    progress.adopmpopws = adopmpopws
                    progress.adopmpopsf = adopmpopsf
                    progress.adopmlocws = adopmlocws
                    progress.adopmpwpnbr = adopmpwpnbr
                    progress.adopmwprnbr = adopmwprnbr
                    progress.adopmwcnbr = adopmwcnbr
                    progress.adopmsfpnbr = adopmsfpnbr
                    progress.adopmsfinbr = adopmsfinbr
                    self.session.add(progress)
                print '        * Added project monitoring data'

                # Add localities to project
                for i in range(1, 25):
                    loc = self.get_locality()
                    if loc:
                        q = self.session.query(Tl_adoploc).filter(and_(Tl_adoploc.oploadlocode == loc,
                                                                       Tl_adoploc.oploadopcode == prog.adopcode)).all()
                        if len(q) == 0:
                            ob = Tl_adoploc()
                            ob.oploadlocode = loc
                            ob.oploadopcode = prog.adopcode
                            self.session.add(ob)
                            self.session.flush()
                        else:
                            print '        * [WARN] Skipping to add this locality, already added'
                print '        * Added project localities'

                # Add subdivisions to project
                for i in range(1, 3):
                    reg = self.get_subdivision()
                    q = self.session.query(Tl_adopr).filter(and_(Tl_adopr.opradrcode == reg,
                                                                   Tl_adopr.opradopcode == prog.adopcode)).all()
                    if len(q) == 0:
                        ob = Tl_adopr()
                        ob.opradrcode = reg
                        ob.opradopcode = prog.adopcode
                        self.session.add(ob)
                        self.session.flush()
                    else:
                        print '        * [WARN] Skipping to add this subdivision, already added'
                print '        * Added project subdivisions'
            except:
                type, val, tb = sys.exc_info()
                message_id = string.join(traceback.format_exception(type, val, tb), '')
                print 'Skipping to add this project due to error: %s' % message_id
        print 'Done adding projects'


    def generate_wpn(self):
        print 'Populating t_upwpn'
        self.session.connection().execute("""DELETE FROM t_upwpn""")

        for upwpntype in ('01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12', '13', '98'):
            self.session.connection().execute("""INSERT INTO t_upwpn (wpnadlocode, upwpntype, upwpnok, upwpnoknot, upwpntot)
                SELECT adlocode, '%(upwpntype)s',round(random() * 20) + 1, round(random() * 10) + 1, round(random() * 30) + 10 FROM t_adloc""" \
                 % {'upwpntype' : upwpntype})
        self.session.flush()
        dummy_insert(self.session)


    def generate_sfn(self):
        print 'Populating t_upsfn'
        self.session.connection().execute("""DELETE FROM t_upsfn""")
        for upsfntype in ('01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12', '13', '14', '98', '99'):
            self.session.connection().execute("""INSERT INTO t_upsfn (sfnadlocode, upsfntype, upsfnok, upsfnoknot, upsfntot)
                SELECT adlocode, '%(upsfntype)s', round(random() * 20) + 1, round(random() * 10) + 1, round(random() * 30) + 10 FROM t_adloc""" \
                % {'upsfntype' : upsfntype})
        self.session.flush()
        dummy_insert(self.session)


    def generate_country_statistics(self, country_code):
        print 'Generating country statistics t_adcountry'
        ob = self.session.query(Country).filter_by(adcncodex=country_code.upper()).one()
        ob.total_population = rand_int(8000000, 18000000)
        ob.rural_population = int(ob.total_population * 0.8)
        ob.urban_population = int(ob.total_population * 0.2)
        ob.water_access_rural = random.uniform(0.1, 0.8)
        ob.water_access_rural_byconn = random.uniform(0.1, 0.8)
        ob.water_access_urban = random.uniform(0.7, 1)
        ob.water_access_urban_byconn = random.uniform(0.7, 1)
        ob.sanitation_access_rural = random.uniform(0.1, 0.7)
        ob.sanitation_access_urban = random.uniform(0.1, 0.7)
        ob.year = 2010
        self.session.merge(ob)


    def generate_country_historic_access(self, country_code):
        print 'Generating sample country access history t_adcountry_stat'
        for i in range (1999, 2011):
            ob = CountryStat()
            ob.year = i
            ob.water_access_rural = random.uniform(0.1, 0.6)
            ob.water_access_urban = random.uniform(0.3, 0.8)
            ob.sanitation_access_rural = random.uniform(0.1, 0.4)
            ob.sanitation_access_urban = random.uniform(0.3, 0.8)
            self.session.add(ob)


    def generate_sf_access(self):
        print 'Updating t_adloc to add sample sanitation facilities access'
        self.session.connection().execute("""UPDATE t_adloc SET sf_access = random() * 0.9 + 0.01""")
        self.session.flush()
        dummy_insert(self.session)


    def generate_iwrm(self):
        print 'Adding IWRM test data'

        seed = self._mk_random()

        # Add monitoring stations
        print '    * 200 monitoring stations'
        for i in range(0, 200):
            hye = IWRM.HydrographicEntity()
            self.session.add(hye)
            self.session.flush()

            ob = IWRM.MonitoringStation()
            ob.moscode = hye.hyecode
            ob.moscodex = 'TESTMOST%s' % (seed + i)
            ob.mosnamee = self._mk_sentence()
            ob.smoncode = self._mk_word()
            ob.mslocation = self._mk_word()
            ob.mosx = self._mk_longitude()
            ob.mosy = self._mk_latitude()
            ob.mosz = rand_int(5, 20)
            ob.mostypee = self._mk_word()
            ob.mosyearsta = rand_int(1995, 2010)
            ob.mosyearmdif = rand_int(1995, 2010)
            ob.mosyearend = rand_int(1995, 2010)
            ob.mossit = self._mk_sentence()
            ob.mosknext = rand_int(100, 500)
            ob.moscatha = self._mk_float(4000, 9999)
            ob.moscomm = self._mk_sentence()
            ob.manres = self._mk_username()
            ob.moswho = self._mk_username()
            self.session.add(ob)
        self.session.flush()

        # Add basins
        print '    * 100 basins'
        for i in range(0, 100):
            hye = IWRM.HydrographicEntity()
            self.session.add(hye)
            self.session.flush()
            
            ob = IWRM.Basin()
            ob.hybcode = hye.hyecode
            ob.hybname = self._mk_sentence()
            ob.hybdesc = self._mk_sentence()
            ob.hybsurf = self._mk_float(3000, 20000)
            self.session.add(ob)
        self.session.flush()

        # Add aquifers
        print '    * 100 aquifers'
        for i in range(0, 100):
            hye = IWRM.HydrographicEntity()
            self.session.add(hye)
            self.session.flush()

            ob = IWRM.Aquifer()
            ob.aqfcode = hye.hyecode
            ob.aqfname = self._mk_sentence()
            self.session.add(ob)
        self.session.flush()

        # Add rivers
        print '    * 100 rivers'
        for i in range(0, 100):
            hye = IWRM.HydrographicEntity()
            self.session.add(hye)
            self.session.flush()

            ob = IWRM.River()
            ob.rivcode = hye.hyecode
            ob.rivdesc = self._mk_sentence()
            ob.hybascode = self._query("SELECT hybcode FROM t_hybasin ORDER BY RANDOM() LIMIT 1")
            self.session.add(ob)
        self.session.flush()


        # Add UWS
        print '    * 200 underground water sources'
        for i in range(0, 200):
            upiscode = 'TESTUWS%s' % (seed + i)
            upis = self._mk_upis_uws(upiscode)

            upwp = Waterpoint()
            upwp.upiscode = upiscode
            upwp.upwpdiam = self._mk_float(20, 80)
            upwp.upwpdepth = self._mk_float(10, 200)
            upwp.upwpwlev = self._mk_float(10, 100)
            upwp.upwputype = self.get_lexicon('LexUWS005')
            upwp.upwpuyelm = self._mk_float(10, 30)
            upwp.upwpudepin = self._mk_float(10, 100)
            upwp.fk_aqfcode = self._query("SELECT aqfcode FROM t_hyaquifer ORDER BY RANDOM() LIMIT 1")
            self.session.add(upis)
            self.session.flush()
            self.session.add(upwp)

        self.session.flush()

        dummy_insert(self.session)
        c = self.session.connection()
        c.execute("UPDATE t_upis SET upiscoordinate = ST_SetSRID(ST_MakePoint(upisx, upisy), 4326);")
        c.execute("UPDATE t_hymostation SET moscoordinate = ST_SetSRID(ST_MakePoint(mosx, mosy), 4326);")
        print 'Done with IWRM'


    def _mk_upis_uws(self, upiscode):
        """ insert a record in table t_upis """
        ob = Upis()
        ob.upiscode = upiscode
        ob.upisadlocode = self.get_locality()

        subdivision = self.get_subdivision_tuple()
        ob.upisadr1code = subdivision[0]
        ob.upisadr2code = subdivision[1]
        ob.upisadr3code = subdivision[2]
        ob.upisadr4code = subdivision[3]
        ob.upisadr5code = subdivision[4]

        ob.upistype = 'UWS'
        ob.upisname = self._mk_sentence()
        ob.upisx = self._mk_longitude()
        ob.upisy = self._mk_latitude()
        ob.upisyear = rand_int(1999, 2010)
        ob.approved = True
        ob.contributed = False
        ob.upisupdate = self._mk_date()
        ob.upisupwho = self._mk_username()
        ob.upisowner = self._mk_username()
        return ob



    def loc_count(self):
        return self.session.query(Locality).count()

    def subdiv_count(self):
        return self.session.query(Subdivision).count()

    def wp_count(self):
        return self.session.query(Waterpoint).count()

    def sf_count(self):
        return self.session.query(Sanitation).count()

    def orga_count(self):
        return self.session.query(Organisation).count()

    def op_count(self):
        return self.session.query(Program).count()

    def wpn_count(self):
        return self.session.query(Upwpn).count()

    def sfn_count(self):
        return self.session.query(Upsfn).count()

    def pws_count(self):
        return self.session.query(pws_models.PWS).count()

    def country_historic_count(self):
        return self.session.query(CountryStat).count()
