from lxml import etree
from lxml.etree import SubElement
from oaipmh import common

# key to find xml:lang tags
LANG_KEY = '{http://www.w3.org/XML/1998/namespace}lang'

class MetadataRegistry(object):
    """A registry that contains readers and writers of metadata.

    a reader is a function that takes a chunk of (parsed) XML and
    returns a metadata object.

    a writer is a function that takes a takes a metadata object and
    produces a chunk of XML in the right format for this metadata.
    """
    def __init__(self, flags=[]):
        self._readers = {}
        self._writers = {}
        self._flags = flags

    def registerReader(self, metadata_prefix, reader):
        self._readers[metadata_prefix] = reader

    def registerWriter(self, metadata_prefix, writer):
        self._writers[metadata_prefix] = writer

    def hasReader(self, metadata_prefix):
        return metadata_prefix in self._readers

    def hasWriter(self, metadata_prefix):
        return metadata_prefix in self._writers

    def readMetadata(self, metadata_prefix, element):
        """Turn XML into metadata object.

        element - element to read in

        returns - metadata object
        """
        if self._flags:
            self._readers[metadata_prefix]._flags = self._flags
        return self._readers[metadata_prefix](element)

    def writeMetadata(self, metadata_prefix, element, metadata):
        """Write metadata as XML.

        element - ElementTree element to write under
        metadata - metadata object to write
        """
        self._writers[metadata_prefix](element, metadata)

global_metadata_registry = MetadataRegistry()

class Error(Exception):
    pass

class MetadataReader(object):
    """A default implementation of a reader based on fields.
    """
    def __init__(self, fields, namespaces=None, flags = []):
        # use flags to extend API without interfering with other uses
        # for example, we will extend with 'xml:lang'
        self._flags = flags
        self._fields = fields
        self._namespaces = namespaces or {}

    def __call__(self, element):
        map = {}
        # create XPathEvaluator for this element
        xpath_evaluator = etree.XPathEvaluator(element,
                                               namespaces=self._namespaces)

        e = xpath_evaluator.evaluate
        # now extra field info according to xpath expr
        for field_name, (field_type, expr) in self._fields.items():
            key = field_name    # key in the map mapping
            if field_type == 'bytes':
                value = str(e(expr))
            elif field_type == 'bytesList':
                value = [str(item) for item in e(expr)]
            elif field_type == 'text':
                # make sure we get back unicode strings instead
                # of lxml.etree._ElementUnicodeResult objects.
                value = unicode(e(expr))
            elif field_type == 'textList':
                # make sure we get back unicode strings instead
                # of lxml.etree._ElementUnicodeResult objects.
                #
                # Extension: if 'xml:lang' is present in self._flags then
                # change the key/value pair to include xml:lang data from the
                # xml element.
                value = [unicode(v) for v in e(expr)]
                if 'xml:lang' in self._flags:
                    # construct a mapping of data keyed on language
                    lang_data = {}
                    for elem in element.iter():
                        tag_parts = elem.tag.split('}')
                        tag_name = tag_parts[1]
                        if tag_name == field_name:
                            text = unicode(elem.text)
                            lang = elem.attrib.get(LANG_KEY)
                            if lang:
                                key = field_name + ":" + lang
                                if key in lang_data.keys():
                                    lang_data[key].append(text)
                                else:
                                    lang_data[key] = [text]
                    if lang_data:
                        # if we have a mapping on languages add it to the map
                        for key in lang_data.keys():
                            map[key] = lang_data[key]
                        continue
            else:
                raise Error, "Unknown field type: %s" % field_type
            map[key] = value
        return common.Metadata(map)

oai_dc_reader = MetadataReader(
    fields={
    'title':       ('textList', 'oai_dc:dc/dc:title/text()'),
    'creator':     ('textList', 'oai_dc:dc/dc:creator/text()'),
    'subject':     ('textList', 'oai_dc:dc/dc:subject/text()'),
    'description': ('textList', 'oai_dc:dc/dc:description/text()'),
    'publisher':   ('textList', 'oai_dc:dc/dc:publisher/text()'),
    'contributor': ('textList', 'oai_dc:dc/dc:contributor/text()'),
    'date':        ('textList', 'oai_dc:dc/dc:date/text()'),
    'type':        ('textList', 'oai_dc:dc/dc:type/text()'),
    'format':      ('textList', 'oai_dc:dc/dc:format/text()'),
    'identifier':  ('textList', 'oai_dc:dc/dc:identifier/text()'),
    'source':      ('textList', 'oai_dc:dc/dc:source/text()'),
    'language':    ('textList', 'oai_dc:dc/dc:language/text()'),
    'relation':    ('textList', 'oai_dc:dc/dc:relation/text()'),
    'coverage':    ('textList', 'oai_dc:dc/dc:coverage/text()'),
    'rights':      ('textList', 'oai_dc:dc/dc:rights/text()')
    },
    namespaces={
    'oai_dc': 'http://www.openarchives.org/OAI/2.0/oai_dc/',
    'dc' : 'http://purl.org/dc/elements/1.1/'},
    )
