import json

from OFS.SimpleItem import SimpleItem
from AccessControl import ClassSecurityInfo, Unauthorized
from AccessControl.Permissions import view, view_management_screens
from App.class_init import InitializeClass
from Products.PageTemplates.PageTemplateFile import PageTemplateFile
from zope.pagetemplate.pagetemplatefile import PageTemplateFile as Z3Template
from App.config import getConfiguration
from Acquisition import Implicit

from eea import usersdb
from eea.usersdb import factories

cfg = getConfiguration()
NETWORK_NAME = getattr(cfg, 'environment', {}).get('NETWORK_NAME', 'EIONET')

import logging
log = logging.getLogger(__name__)

manage_add_userdetails_html = PageTemplateFile('zpt/userdetails/manage_add',
                                                globals())


def manage_add_userdetails(parent, id, REQUEST=None):
    """ Create a new UserDetails object """
    form = (REQUEST.form if REQUEST is not None else {})
    obj = UserDetails()
    obj.title = form.get('title', id)
    obj._setId(id)
    parent._setObject(id, obj)

    if REQUEST is not None:
        REQUEST.RESPONSE.redirect(parent.absolute_url() + '/manage_workspace')

def _is_authenticated(request):
    return ('Authenticated' in request.AUTHENTICATED_USER.getRoles())

def load_template(name, _memo={}):
    if name not in _memo:
        _memo[name] = Z3Template(name, globals())
    return _memo[name]

zope2_wrapper = PageTemplateFile('zpt/zope2_wrapper.zpt', globals())

class TemplateRenderer(Implicit):
    def __init__(self, common_factory=lambda ctx: {}):
        self.common_factory = common_factory

    def render(self, name, **options):
        context = self.aq_parent
        template = load_template(name)
        namespace = template.pt_getContext((), options)
        namespace['common'] = self.common_factory(context)
        return template.pt_render(namespace)

    def wrap(self, body_html):
        context = self.aq_parent
        zope2_tmpl = zope2_wrapper.__of__(context)
        return zope2_tmpl(body_html=body_html)

    def __call__(self, name, **options):
        return self.wrap(self.render(name, **options))


class CommonTemplateLogic(object):
    def __init__(self, context):
        self.context = context

    def _get_request(self):
        return self.context.REQUEST

    def base_url(self):
        return self.context.absolute_url()

    def portal_url(self):
        return self.context.restrictedTraverse("/").absolute_url()

    def is_authenticated(self):
        return _is_authenticated(self._get_request())

    def is_manager(self):
        return ('Manager' in
                self._get_request().AUTHENTICATED_USER.getRoles())

    @property
    def macros(self):
        return load_template('zpt/macros.zpt').macros

    @property
    def network_name(self):
        """ E.g. EIONET, SINAnet etc. """
        return NETWORK_NAME

def logged_in_user(request):
    user_id = ''
    if _is_authenticated(request):
        user = request.get('AUTHENTICATED_USER', '')
        user_id = str(user.id)

    return user_id

class UserDetails(SimpleItem):
    meta_type = 'Eionet User Details'
    security = ClassSecurityInfo()
    icon = '++resource++eea.userseditor-www/users_editor.gif'

    _render_template = TemplateRenderer(CommonTemplateLogic)

    def __init__(self):
        super(UserDetails, self).__init__()

    def _get_ldap_agent(self):
        return factories.agent_from_uf(self.restrictedTraverse("/acl_users"))

    def _prepare_user_page(self, uid):
        """Shared by index_html and simple_profile"""
        agent = self._get_ldap_agent()
        ldap_roles = sorted(agent.member_roles_info('user', uid, ('description',)))
        roles = []
        for (role_id, attrs) in ldap_roles:
            roles.append((role_id, attrs.get('description', ('', ))[0]))
        user = agent.user_info(uid)
        user['jpegPhoto'] = agent.get_profile_picture(uid)
        user['certificate'] = agent.get_certificate(uid)
        return user, roles

    security.declarePublic("index_html")
    def index_html(self, REQUEST):
        """ """
        uid = REQUEST.form.get('uid')
        if "," in uid:
            user = None
            roles = None
            multi = json.dumps({'users': uid.split(",")})
        else:
            multi = None
            user, roles = self._prepare_user_page(uid)
        return self._render_template("zpt/userdetails/index.zpt",
                                     user=user, roles=roles, multi=multi)

    security.declarePublic("simple_profile")
    def simple_profile(self, REQUEST):
        """ """
        uid = REQUEST.form.get('uid')
        user, roles = self._prepare_user_page(uid)
        tr = TemplateRenderer(CommonTemplateLogic)
        return tr.__of__(self).render("zpt/userdetails/simple.zpt",
                                     user=user, roles=roles)

    security.declarePublic("userphoto_jpeg")
    def userphoto_jpeg(self, REQUEST):
        """ """
        uid = REQUEST.form.get('uid')
        agent = self._get_ldap_agent()
        REQUEST.RESPONSE.setHeader('Content-Type', 'image/jpeg')
        return agent.get_profile_picture(uid)

    security.declarePublic("usercertificate")
    def usercertificate(self, REQUEST):
        """ """
        uid = REQUEST.form.get('uid')
        agent = self._get_ldap_agent()
        REQUEST.RESPONSE.setHeader('Content-Type', 'application/pkix-cert')
        return agent.get_certificate(uid)
