"""
    ACASUserFolder class allow to authenticate users via a CAS
    server and store LOCAL users created on the fly.

    CAS Server tested :
        - CAS 2.0.12 from yale
    Zope version tested :
        - 2.6.2
        - 2.7.1
        - 2.7.2-0
        - 2.7.3
        - 2.7.4-0
        - 2.8
    Python version tested :
        - 2.1.3+
        - 2.2.3+
        - 2.3.5
"""

# Python imports
from base64 import decodestring, encodestring
from random import randint
import urllib

# Zope imports
from zLOG import LOG, DEBUG, INFO, WARNING, ERROR

from OFS.SimpleItem import Item # Basic zope object
from AccessControl import ClassSecurityInfo
from AccessControl.Role import RoleManager # provide the 'Ownership' tab with
                                # the 'manage_owner' method
from AccessControl.Role import DEFAULTMAXLISTUSERS
from AccessControl.ZopeSecurityPolicy import _noroles
from AccessControl.User import BasicUserFolder
from AccessControl.User import User, SimpleUser
from AccessControl.User import domainSpecMatch
from AccessControl.SpecialUsers import nobody # used for auto_redirect feature
from AccessControl.Permissions import view_management_screens, \
                                      manage_users, \
                                      view
from Globals import DTMLFile, MessageDialog
from Globals import Persistent
from Globals import PersistentMapping
from Globals import InitializeClass
from Acquisition import Implicit
from ZPublisher.HTTPRequest import FileUpload

# CASUserFolder imports
from UserStorage import SessionUserStorage, PersistentUserStorage
from CASXMLResponseParser import CASXMLResponseParser
from ProtectedSessionInfo import ProtectedSessionInfo
from IACASUserFolder import IACASUserFolder

# Zope compatibility
try:
    import transaction
except:
    pass
try:
    from zExceptions import Redirect
except ImportError:
    # Pre Zope 2.7
    Redirect = 'Redirect'
            
# python 2.1 compatibility
try: 
  True == True
except:
  True = 1
  False = 0

__version__ = '2.0.2'


class ACASUserFolder(
            IACASUserFolder,
            BasicUserFolder,
            RoleManager,
            ):
    """
            ACASUserFolder Object
    """
    
    meta_type = 'ACAS User Folder'
    id        = 'acl_users'
    title     = 'ACAS User Folder'

    isPrincipiaFolderish = 1
    isAUserFolder = 1
    maxlistusers = DEFAULTMAXLISTUSERS

    
    CUF_SESS_USERNAME  = 'casUserName' # static variable
    CUF_SESS_TESTING   = 'casTesting'  # static variable
    CUF_SESS_POST_DATA = 'CAS_POST_DATA_STORE'   # static variable
    CUF_GETVAR_TEST    = 'cas_test'    # static variable for manage_test()
    CUF_GETVAR_DELAY_POST   = 'cas_delay_POST'   # static variable
    CUF_GETVAR_CONSUME_POST = 'cas_consume_POST' # static variable
    CUF_CAS_USERNAME   = 'CAS_USER'    # static variable for identify()
    CUF_CAS_PWD        = 'CAS_PWD'     # static variable for identify()
    CUF_DEF_USE_ACTUAL_URL = True      # static variable
    cuf_default_domains = []           # default domains for CAS users
    
    security = ClassSecurityInfo()

    manage_options = (
      (
        {'label': 'Contents',
         'action': 'manage_main', 
         'help': ('ACASUserFolder', 'users.stx')
        },
            {'label': 'Properties', 
         'action': 'manage_propertiesForm', 
         'help': ('ACASUserFolder', 'properties.stx')
        },
        {'label': 'Test',
         'action': 'manage_testForm', 
         'help': ('ACASUserFolder', 'properties.stx')
        },
      )
      + Item.manage_options            # add the 'Undo' & 'Owner' tab 
      + RoleManager.manage_options     # add the 'Security' tab
    )

    manage_propertiesForm = DTMLFile('dtml/manage_propertiesACASUserFolderForm', globals()) 
    security.declareProtected(view_management_screens, 'manage_propertiesForm')
    security.declareProtected(view_management_screens, 'manage')
    security.declareProtected(view_management_screens, 'manage_main')
    manage = manage_propertiesForm = \
        DTMLFile('dtml/manage_propertiesACASUserFolderForm', globals(), default_roles = [])
    manage._setName('manage_propertiesForm')

    security.declareProtected(view_management_screens, 'manage_undoForm')
    security.declareProtected(view_management_screens, 'manage_owner')
    security.declareProtected(view_management_screens, 'manage_access')
    
    manage_testForm = DTMLFile('dtml/manage_testACASUserFolderForm', globals())
    security.declarePublic('manage_testForm') # any user must see the test screen

    # ----------
    # __init__
    # ----------

    def __init__(self,
                 login_url    = "https://cas.your.dom/login", 
                 validate_url = "https://cas.your.dom/serviceValidate", 
                 logout_url   = "https://cas.your.dom/logout",
                 login_cookie_domain   = ".eionet.europa.eu",                  
                 default_roles    = [],
                 default_domains  = [],
                 auto_redirect    = False, 
                 persistent_users = True,
                 retain_POST_data = True, 
                 use_ACTUAL_URL   = CUF_DEF_USE_ACTUAL_URL,
                 REQUEST = None):
        "initialise a new instance of ACASUserFolder"        
        #Debug('__init__')
        self.id = 'acl_users'
        self.title = 'Alt. CAS User Folder'

        self.cuf_login_url        = login_url
        self.cuf_validate_url     = validate_url
        self.cuf_logout_url       = logout_url
        self.eionet_login_cookie_domain = login_cookie_domain
        self.cuf_default_roles    = default_roles
        self.cuf_default_domains  = default_domains
        self.cuf_auto_redirect    = auto_redirect
        self.cuf_persistent_users = persistent_users
        self.cuf_retain_POST_data = retain_POST_data
        self.cuf_use_ACTUAL_URL   = use_ACTUAL_URL

        if persistent_users:
            self.user_storage = PersistentUserStorage()
        else:
            self.user_storage = SessionUserStorage()
    
    # -------------
    # __setstate__
    # -------------

    def __setstate__(self, v):
        """
            __setstate__ is called whenever the instance is loaded
            from the ZODB, for example when Zope is restarted.
        """
        #Debug('--------- CHANGING STATE ------------')
        # Call inherited __setstate__ methods if they exist
        ACASUserFolder.inheritedAttribute('__setstate__')(self, v)

        # smooth upgrade
        
        # added in ACASUserFolder 2.0.1
        if not hasattr(self, 'cuf_use_ACTUAL_URL'):
            #Debug('--------- REGEN ATTR cuf_use_ACTUAL_URL')
            self.cuf_use_ACTUAL_URL = self.CUF_DEF_USE_ACTUAL_URL
            self._p_changed = 1
            


    # -----------------------
    # Interface IUserFolder
    # -----------------------

    def getUser(self, username):
        """
        Returns the user object specified by name.
        Users may be added by hand or automatically when ticket is validated.
        
        If persistent storage is not used, User instance is stored in the
        session, so it disappears uppon Zope restart or session expires.
        """
        #Debug('getUser(%s)' % (username))
        user = self.user_storage.getUser(username)
        if user is None:
            return None
        else:
            return user
        #    self._doAddUser(username, self.CUF_CAS_PWD, self.cuf_default_roles, self.cuf_default_domains)
        #return self.user_storage.getUser(username)
        

    def getUsers(self):
        """
        Returns a sorted sequence of all user objects which reside in the user
        folder.
        """
        #Debug('getUsers()')
        return self.user_storage.getUsers()
        

    def getUserNames(self):
        """
        Returns a sorted sequence of names of the users which have connected
        recently.
        This list is volatile and can be emptyed when this object is reloaded 
        from ZODB.
        """
        return self.user_storage.getUserNames()
        

    # --------------------------
    # BasicUserFolder Overrides
    # --------------------------

    def getUserById(self, id, default = None):
        return self.getUser(id)
    

    def hasUsers(self):
        #Debug('hasUsers()', '')
            return True # always true => we depend on external data


    # --------------------------------------------
    # BasicUserFolder "_do" method implementation 
    # --------------------------------------------
    
    def _doChangeUser(self, name, password, roles, domains, **kw):
        """
        This method is needed by GRUF.
        Roles are the way GRUF affect groups on users.
        """
        user = self.getUser(name)
        if password is not None:
            user.__ = password
        user.roles = roles
        user.domains = domains

    def _doAddUser(self, name, password, roles, domains, **kw):
        """
        Add a new user.
        These users are temporary and used only as markers.
        To avoid hidden security hole however, a random pwd is generated
        """
        random_pwd = encodestring(str(randint(1234, 999999999))+str(randint(4321, 999999999)))
        if self.cuf_persistent_users:
            user = User(name, random_pwd, roles, domains)
        else:
            user = SimpleUser(name, random_pwd, roles, domains)
        self.user_storage.addUser(name, user)
        #Debug('_doAddUser(%s)'%(name))

    def _doDelUsers(self, names):
        self.user_storage.delUsers(names)
    
        
    # -------------------------------------
    # BasicUserFolder Authentication stuff
    # -------------------------------------

    #
    # def validate(self, request, auth='', roles=_noroles):
    #     # the validate() method is left to parent class
    #

    def identify(self, auth):
        """
        Returns: 'CAS_USER', 'CAS_PWD' if auth is not 'Basic ...'
                 These values are not used at all, but they are mandatory
                 if we are in a GroupUserFolder (auth-patched) for authenticate
                 to be called from GroupUserFolder.validate().
                 Update: CAS_USER is needed in authenticate()
        """
        #Debug('identify(auth=%s)'%(str(auth)), '')
        if auth is not None:
            if auth.find('Basic') == 0:
                # Basic credentials HAVE PRECEDENCE, this fix a stupid admin
                # lockout issue and allow use of emergency user.
                return BasicUserFolder.identify(self, auth)
        return self.CUF_CAS_USERNAME, self.CUF_CAS_PWD
        

    def authenticate(self, name, password, REQUEST, _skip=True):
        """
        We try here to identify a CAS user by checking its ticket against the
        CAS server.
        If a name and password are provided we just return, as it is
        no more a cas authentication.

        Returns: User instance
        """
        if _skip:
            return None
        
        #Debug('authenticate(name=%s, password=%s, %s)'%(str(name), 'XXXXX', 'REQUEST'))

        form = REQUEST.form
        
        if form.has_key(self.CUF_GETVAR_TEST):
            # we come from testing forms so Basic auth is disabled one time!
            #Debug('authenticate() : one shot DISABLE Basic auth')
            pass
        else:
            # If we have auth credentials other than CAS + auto_redirect, CAS auth
            # is concealed to BasicUserFolder to allow emergency & admin users
            emergency = self._emergency_user
            if emergency and name==emergency.getUserName():
                if emergency.authenticate(password, REQUEST):
                    return emergency
                else:
                    return None

            if name is not None and name is not self.CUF_CAS_USERNAME:
                #Debug('=> authenticate() : return BasicUserFolder.authenticate()')
                return BasicUserFolder.authenticate(self, name, password, REQUEST)
        
        # has the browser sent a CAS ticket?
        ticket = form.get('ticket')

        if ticket is None:
            # no ticket, have we a previously validated user in the session?
            username = self.user_storage.getTicketValidatedUserName(REQUEST)
            if username is not None:
                #Debug('authenticate() => ok' + username, 'SESSION=%s'%(str(REQUEST.SESSION)))
                if form.get(self.CUF_GETVAR_CONSUME_POST):
                    self._restorePOSTDataIfAny(REQUEST)
                return self.getUser(username)
            #Debug('authenticate() => CACHE MISS')
            
            # now, we'll try to prepare to re-login without loss of data
            if self.cuf_retain_POST_data:
                self._storePOSTDataIfAny(REQUEST)
            
            # poorly working in plone redirect stuff (cookie crumbler issue)
            if self.cuf_auto_redirect:
                #Debug('=> authenticate() : REDIRECT TO LOGIN')
                #import pdb; pdb.set_trace();
                if not nobody.allowed(REQUEST['PUBLISHED'],REQUEST['PUBLISHED'].__roles__):                                
                    self._redirectToCASLogin(REQUEST, force = True)
                return nobody # kind of dummy but seems better than None
        else:
            # HERE we talk to CAS server
            username = self._validateTicket(ticket, REQUEST)
            if username is not None:
                # CAS TICKET VALIDATED

                # ADD user (with dummy pwd) to storage if it has not already
                # be generated.
                user = self.getUser(username)
                if user is None:
                    self._doAddUser(username, self.CUF_CAS_PWD, self.cuf_default_roles, self.cuf_default_domains)
                user = self.getUser(username)

                if not self._checkUserDomains(user, REQUEST):
                    Warning("rejected user %s connecting from non allowed domain" % (username))
                    return None # too bad after all this stuff eh?
                
                # register validated user into the session
                self.user_storage.setTicketValidatedUserName(REQUEST, username)

                if self.cuf_retain_POST_data:
                    if not form.get(self.CUF_GETVAR_DELAY_POST):
                        self._restorePOSTDataIfAny(REQUEST)

                # RETURN the user previously stored (or refreshed)
                #Debug('=> authenticate() : TICKET VALIDATED FOR %s'%(username))
                REQUEST.RESPONSE.setHeader('Set-Cookie','eionetCasLogin=loggedIn; path=/; domain=%s' % self.eionet_login_cookie_domain)
                return user # validated
            else:
                # TICKET NOT VALIDATED
                # this can happen if a user use an old bookmark for example.
                # We try then to fallaback to a previous session
                username = self.user_storage.getTicketValidatedUserName(REQUEST)
                if username is not None:
                    #Debug('=> authenticate() : SESSION CACHE HIT FOR %s'%(username))
                    return self.getUser(username)
        
        # defer the authenticate() to other User Folder if any
        #Debug('=> authenticate() : NO CAS AUTH => return None')
        return None



    # ------------------------------
    # IACASUserFolder Public stuff
    # ------------------------------

    security.declarePublic('getCASUserName')
    def getCASUserName(self, REQUEST = None):
        """
        Deprecated (kept for compatibility with 1.X releases)
        """
        return self.cas_get_username(REQUEST)

    security.declarePublic('cas_get_username')
    def cas_get_username(self, REQUEST = None):
        """return CAS username if any"""
        if REQUEST == None:
            try:    REQUEST = self.REQUEST
            except: return None
        return self.user_storage.getTicketValidatedUserName(REQUEST)

    security.declarePublic('cas_local_logout')
    def cas_local_logout(self, REQUEST = None, service = None):
        """clean local session credentials"""
        if REQUEST == None:
            try:    REQUEST = self.REQUEST
            except: return
        self.user_storage._delPSI(REQUEST)
        if service:
            REQUEST.RESPONSE.redirect(service)

    security.declarePublic('cas_redirect_to_login')
    def cas_redirect_to_login(self, REQUEST = None, force = False):
        """goto CAS server login with a valid service parameter"""
        if REQUEST == None:
            try:    REQUEST = self.REQUEST
            except: return
        self._redirectToCASLogin(REQUEST, force)

    security.declarePublic('cas_complete_logout')
    def cas_complete_logout(self, REQUEST = None, service = None):
        """clean local session credentials AND CAS server credentials"""
        if REQUEST == None:
            try:    REQUEST = self.REQUEST
            except: return
        self.cas_local_logout(REQUEST)
        REQUEST.RESPONSE.setHeader('Set-Cookie','eionetCasLogin=loggedOut; path=/; domain=%s' % self.eionet_login_cookie_domain)        
        self._redirectToCASLogout(REQUEST, service)
        
    

    # ------------------------------
    # CAS User Folder PRIVATE stuff
    # ------------------------------
    
    security.declarePrivate('_checkUserDomains')
    def _checkUserDomains(self, user, REQUEST):
        """
        This is some code extracted from AccessControl.User
        We check here if the user originating domain is allowed
        to connect.
        """
        domains = user.getDomains()
        if domains:
            return domainSpecMatch(domains, REQUEST)
        else:
            return True
        
        
    
    security.declarePrivate('_getQuotedService')
    def _getQuotedService(self, REQUEST, remove_ticket=False, append=''):
        """
        Provide a valid and quoted CAS service URL handling GET parameters.
        POST parameters are not handled here.
        - append is of the form 'var1=value1&var2=value2 ...'
        Example: 
            http%3A//cas.your.dom/login%3F_name_%3D_value_
        """
        query_string = REQUEST.get('QUERY_STRING')
        if self.cuf_use_ACTUAL_URL:
            try:
                url = REQUEST['ACTUAL_URL']
            except:
               # Zope < 2.7.4 comaptibility
               url = REQUEST['URL']
        else:
            url = REQUEST['URL']
            
        if remove_ticket:
           ##Debug('_getQuotedService', 'QUERY_STRING (unaltered) =%s' % (query_string))
           idx = query_string.rfind('ticket=')
           if idx >= 0:
               if idx > 0 and query_string[idx - 1] == '&':
                   idx -= 1
               query_string = query_string[:idx]

        if append:
            if query_string:
                query_string = '%s&%s' % (query_string, append)
            else:
                query_string = append
        
        if query_string:
            service = '%s?%s' % (url, query_string)
        else:
            service = url
        
        service = urllib.quote(service)
        ##Debug('_getQuotedService', 'SERVICE=%s' % (service))
        return service
        
    
    security.declarePrivate('_hasSessionData')
    def _hasSessionData(self):
        """Adapter method for detecting Sessions"""
        return self.session_data_manager.hasSessionData()
    
    security.declarePrivate('_storePOSTDataIfAny')
    def _storePOSTDataIfAny(self, REQUEST):
        """
        This method looks for POST form data and store it into the session.
        The session should be a brand new one, as this method is designed
        to be called after a session timeout. The session id may have been
        kept if cookie method is used with BrowserIdManager. Other BrowserId
        method are NOT supported.
        """
        method = REQUEST.get('REQUEST_METHOD')
        if method == 'POST':
            form = REQUEST.form

            # supress unpicklable objects
            picklable_form = form.copy()
            for k in form.keys():
                if isinstance(form[k], FileUpload):
                    del(picklable_form[k])
            #Debug('_storePOSTDataIfAny()', 'storing POST_VARS (%s)\nFORM_DATA=%s'%(str(REQUEST.form.keys()), str(REQUEST.form)))

            # we use here wrapped Session access from UserStorage
            self.user_storage._setSecure(REQUEST, 
                                             self.CUF_SESS_POST_DATA, 
                                         picklable_form)

            # - zope black magic here -
            # This ensure that the data actually stored into the session is comitted
            # into the transient data container of the -probably- Anonymous current
            # user.
            # Otherwise the next HTTP connection will find an empty new session
            # associated to this browser id. This may be due to the fact that in a
            # context where auth is needed the whole transaction (and the session) is
            # cancelled for anonymous users.
            # Please send comment if there is a better way to do this here.
            try:
                transaction.get().commit()
            except NameError:
                get_transaction().commit()
            
    security.declarePrivate('_restorePOSTDataIfAny')
    def _restorePOSTDataIfAny(self, REQUEST):
        """
        This method should be called only when this folder receive a validated
        CAS Service Ticket.
        Then POST data from the timeouted form may be restored seamlessly.

        Side effect: this method modify :
            - REQUEST.environ
            - REQUEST.form
        """
        if not self._hasSessionData():
            return

        # we use here wrapped Session access from UserStorage
        form = self.user_storage._getSecure(REQUEST,
                                            self.CUF_SESS_POST_DATA)
        if form is not None:
            REQUEST.form = form
            self.user_storage._delSecure(REQUEST,
                                         self.CUF_SESS_POST_DATA)
            REQUEST.environ['REQUEST_METHOD'] = 'POST'
            #Debug('_restorePOSTDataIfAny()', 'restored POST_VARS (%s)'%(str(form.keys())))
    
    security.declarePrivate('_redirectToCASLogin')
    def _redirectToCASLogin(self, REQUEST, force = False, append = ''):
        """
        Redirect the browser to the CAS Login URL
        If the user TGC is valid, this will be invisible to the user as it will
        come back again to where he came.
        """
        service = self._getQuotedService(REQUEST, append = append)
        url = '%s?service=%s'%(self.cuf_login_url, service)
        #Debug('_redirectToCASLogin()', url)

        if force:
            REQUEST.RESPONSE.redirect(url, lock = 1)
            #raise Redirect, url
        else:
            REQUEST.RESPONSE.redirect(url)

    security.declarePrivate('_ticketUnvalidateUserName')
    def _ticketUnvalidateUserName(self, REQUEST):
        """
        Cleans the cached CAS user name from the session
        This does NOT do redirect : see _redirectToCASLogout()
        """
        self.user_storage.ticketUnvalidateUserName(REQUEST)

    security.declarePrivate('_redirectToCASLogout')
    def _redirectToCASLogout(self, REQUEST, service = None):
        """
        Redirect the browser to the CAS Logout URL
        """
        if service is None:
            service = self._getQuotedService(REQUEST)
        url = '%s?url=%s'%(self.cuf_logout_url, service)
        #Debug('_redirectToCASLogout()', url)
        REQUEST.RESPONSE.redirect(url)
    
    
    security.declarePrivate('_validateTicket')
    def _validateTicket(self, ticket, REQUEST):
        '''
        CAS Login : pass 2
        Validate the ticket we got in the GET data to the CAS server
        RETURNS: username or None
        '''
        
        # prepare the GET parameters for checking the login
        # service must be exactly the same as the one provided to the
        # CAS server login
        service = self._getQuotedService(REQUEST, remove_ticket=True )
        checkparams = "?service=%s&ticket=%s" % (service, ticket)
        #Debug('_validateTicket(%s, REQUEST)' % (str(ticket)), 'service=%s' % (service))
        # check the ticket
        try:
            #print checkparams
            casdata = urllib.URLopener().open(self.cuf_validate_url + checkparams)
        except:
            Error('_validateTicket(REQUEST[URL]=%s, ticket=%s)'%(REQUEST['URL'],ticket), 'CANNOT CONTACT CAS SERVER')
            # The upper layer will got an 'IOError' which is probably best than leaving
            # the user silently unautenticated.
            raise
        casdata_line1 = casdata.readline().strip()
        test = casdata_line1.strip()
        #Debug('_validateTicket() : CAS RESPONSE 1st line: ', str(test))
        if test == 'yes':
            # user is validated (CAS architecture 1.0)
            username = casdata.readline().strip()
            return username
        else:
            # If we are here, either the user is NOT validated, either
            # it may be an XML response (CAS architecture 2.0).
            # Then try to parse an XML response
            if test.lower().find("cas:serviceresponse") > 0:
                parser = CASXMLResponseParser()
                try:
                    while test != "":
                        parser.feed(test.strip())
                        test = casdata.readline()
                except "user", user:
                    return user.strip()
                except "failure", failure_msg:
                    Warning('_validateTicket('+REQUEST['URL']+', '+str(ticket)+') ', "FAILURE => %s"%(failure_msg.strip()))
                    return None
                except:
                    # hummm, maybe it was not XML...
                    return None
            
            Warning('_validateTicket('+REQUEST['URL']+', '+str(ticket)+') ', "CAS Server response not understood or cannot be processed (1st line) : " + casdata_line1)
                
            # some unknown authentication error occurred
            return None
         

    # ------------
    # Management 
    # ------------
       
    security.declareProtected(view_management_screens, 'manage_propertiesAction')
    def manage_propertiesAction(self, login_url, validate_url, logout_url,login_cookie_domain,
                                default_roles=[], default_domains=[],
                                auto_redirect="", persistent_users="", 
                                retain_POST_data="", use_ACTUAL_URL="",
                                REQUEST=None, RESPONSE=None):
        """Changes the instance values"""
        
        persistent_users_previous = self.cuf_persistent_users
        
        self.cuf_login_url        = login_url
        self.cuf_validate_url     = validate_url
        self.cuf_logout_url       = logout_url
        self.eionet_login_cookie_domain = login_cookie_domain
        self.cuf_auto_redirect    = not not auto_redirect
        self.cuf_default_roles    = default_roles
        self.cuf_default_domains  = default_domains
        self.cuf_persistent_users = not not persistent_users
        self.cuf_retain_POST_data = not not retain_POST_data
        self.cuf_use_ACTUAL_URL   = not not use_ACTUAL_URL
        self._p_changed = 1

        # change user storage if required
        if self.cuf_persistent_users != persistent_users_previous :
            #Debug('manage_propertiesAction() : changing user storage : persistence = ' + str(self.cuf_persistent_users))
            user_storage_previous = self.user_storage
            if self.cuf_persistent_users:
                self.user_storage = PersistentUserStorage()
            else:
                self.user_storage = SessionUserStorage()
                
            # dispose previous storage
            del user_storage_previous
            
            # mutable state changed
            self._p_changed = 1
        
        return MessageDialog(
                   title  ='Settings Changed',
                   message='Settings Changed.<br/> Please close your browser windows or clean your cookies while testing to avoid side effects.',
                   action ='%s/manage_propertiesForm' % REQUEST['URL1'])
        #RESPONSE.redirect('manage_propertiesForm')


    security.declarePublic('manage_testLoginAction')
    def manage_testLoginAction(self, REQUEST):
        """Test the CAS Login functionality"""
        if REQUEST.form.has_key('ticket'):
            # Here we are coming back from the CAS server.
            #Debug('=> manage_testLoginAction() : GOTO FORM')
            return self.manage_testForm(self, REQUEST)
        
        # Here we are coming from the test form.
        # Put a one shot marker, so when coming back from CAS server
        # we can disable Basic auth to show authenticated user
        REQUEST.SESSION[self.CUF_SESS_TESTING] = 'yes'
        #Debug('manage_testLoginAction() => GOTO CAS LOGIN')
        self._redirectToCASLogin(REQUEST, 
                                  append='%s=1'%(self.CUF_GETVAR_TEST) )

    security.declarePublic('manage_testLogoutAction')
    def manage_testLogoutAction(self, REQUEST):
        """Test the CAS Logout functionality"""
        # Here we are coming only from the test form.
        #Debug('manage_testLogoutAction() => GOTO CAS !!!LOGOUT!!!')
        self._ticketUnvalidateUserName(REQUEST)
        self._redirectToCASLogout(REQUEST, self.absolute_url()+'/manage_testForm')


    #
    # END of ACASUserFolder Class
    #                                            

#
# LOG
#
def Debug(summary, detail=""):
    LOG("ACASUserFolder", INFO, summary, detail)
    pass

def Error(summary, detail=""):
    LOG("ACASUserFolder", ERROR, summary, detail)

def Warning(summary, detail=""):
    LOG("ACASUserFolder", WARNING, summary, detail)


#
# Product Administration
#

def manage_addACASUserFolderAction(self, REQUEST = None):
   """Add a ACASUserFolder instance to a folder."""
   cuf = ACASUserFolder()

   try:
       self._setObject('acl_users', cuf)
   except:
       return MessageDialog(
                       title  ='Item Exists',
                message='This object already contains a User Folder',
                action ='%s/manage_main' % REQUEST['URL1'])
       
   if REQUEST is not None:
        return self.manage_main(self, REQUEST)



#
# Initialization
#
InitializeClass(ACASUserFolder)


