from Globals import PersistentMapping
import Acquisition
from ProtectedSessionInfo import ProtectedSessionInfo

# python 2.1 compatibility
try: 
  True == True
except:
  True = 1
  False = 0

class UserStorage(Acquisition.Implicit):
    """
    UserStorage allow to use different source for users like
    persistent and non persisntent ones.
    It is mainly a container.
    """

    CAS_VALIDATED_USERNAME = 'CAS_VALIDATED_USERNAME'
    CUF_PSI = '_CUF_PSI'

    def getUsers(self):
        raise NotImplementedError

    def getUserNames(self):
        raise NotImplementedError

    def getUser(self, userName):
        raise NotImplementedError

    def addUser(self, username, user):
        raise NotImplementedError

    def delUser(self, username):
        raise NotImplementedError

    def delUsers(self, names):
        raise NotImplementedError

    def _hasSession(self, REQUEST):
        try:
            return self.session_data_manager.hasSessionData()
        except:
            return False

    def _setSecure(self, REQUEST, key, val):
        """ Use Protected Session Info to interract with Session """
        SESSION = REQUEST.SESSION
        psi = SESSION.get(self.CUF_PSI) or ProtectedSessionInfo()
        psi._set(key, val)

        # this set is mandatory even if psi not new
        SESSION.set(self.CUF_PSI, psi)

    def _getSecure(self, REQUEST, key):
        if self._hasSession(REQUEST):
            psi = REQUEST.SESSION.get(self.CUF_PSI)
            if not psi:
                return None
            return psi._get(key)
        else:
            return None
    
    def _delSecure(self, REQUEST, key):
        if self._hasSession(REQUEST):
            psi = REQUEST.SESSION.get(self.CUF_PSI)
            if not psi:
                return
            psi._del(key)
    
    def _delPSI(self, REQUEST):
        """delete all Protected Session Info from Session"""
        if self._hasSession(REQUEST):
            SESSION = REQUEST.SESSION
            if SESSION.has_key(self.CUF_PSI):
                del SESSION[self.CUF_PSI]
                return True
        return False

    def setTicketValidatedUserName(self, REQUEST, username):
        if REQUEST is not None:
            self._setSecure(REQUEST, self.CAS_VALIDATED_USERNAME, username)
        
    def getTicketValidatedUserName(self, REQUEST):
        if REQUEST is not None:
            return self._getSecure(REQUEST, self.CAS_VALIDATED_USERNAME)

    def ticketUnvalidateUserName(self, REQUEST):
        if REQUEST is not None and self._hasSession(REQUEST):
            SESSION = REQUEST.SESSION
            if SESSION.has_key(self.CUF_PSI):
                psi = SESSION[self.CUF_PSI]
                if psi._has_key(self.CAS_VALIDATED_USERNAME):
                    psi._del(self.CAS_VALIDATED_USERNAME)


class PersistentUserStorage(UserStorage):
    """
    A really persistent container for users
    """
    
    def __init__(self):
        self.data = PersistentMapping()

    def getUsers(self):
        return [self.data[n] for n in self.getUserNames()]
        
    def getUserNames(self):
        names = self.data.keys()
        names.sort()
        return names

    def getUser(self, username):
        return self.data.get(username)

    def addUser(self, username, user):
        self.data[username] = user

    def delUsers(self, names):
        for username in names:
            del self.data[username]


class SessionUserStorage(UserStorage, Acquisition.Implicit):
    """
    Adapter class that uses user session for storage.
    with this storage, data disapear at the end of sesion.
    Also, this one shows only one visible user.

    This Storage method takes care of not creating new sessions
    when no authentication has occurred to avoid session exhausting
    by (malicious) anonymous users.
    """

    CAS_SESS_USERNAME = 'CAS_USERNAME'
    CAS_SESS_USER     = 'CAS_USER'

    def __init__(self):
        # nothing to do here, everything is in the session
        pass

    def getUsers(self):
        user = self._getUser()
        if user is not None:
            return [ user ]
        return []
        
    def getUserNames(self):
        username = self._getUserName()
        if username is not None:
            return [ self._getUserName() ]
        return []
        
    def getUser(self, userName):
        if userName == self._getUserName():
            return self._getUser()
        return None

    def addUser(self, username, user):
        # overwrite possible existing user without prompt!
        try: REQUEST = self.REQUEST
        except: REQUEST = None
        if REQUEST is not None:
            self._setSecure(REQUEST, self.CAS_SESS_USERNAME, username)
            self._setSecure(REQUEST, self.CAS_SESS_USER, user)

    def delUsers(self, names):
        # names not used, there is only one possible username at a time
        # in a SessionUserStorage.
        try: REQUEST = self.REQUEST
        except: REQUEST = None
        if REQUEST is not None and \
           self._getSecure(REQUEST, self.CAS_SESS_USERNAME) is not None:
            self._delSecure(REQUEST, self.CAS_SESS_USERNAME)
            self._delSecure(REQUEST, self.CAS_SESS_USER)

    def _getUser(self):
        """Returns user found in session or None"""
        try: REQUEST = self.REQUEST
        except: REQUEST = None
        if REQUEST is not None:
            return self._getSecure(REQUEST, self.CAS_SESS_USER)
        return None

    def _getUserName(self):
        try: REQUEST = self.REQUEST
        except: REQUEST = None
        if REQUEST is not None:
            return self._getSecure(REQUEST, self.CAS_SESS_USERNAME)
        return None
            


