import re
from unittest import TestSuite, makeSuite
from copy import deepcopy
from BeautifulSoup import BeautifulSoup

from Products.Naaya.tests.NaayaFunctionalTestCase import NaayaFunctionalTestCase
from Products.NaayaCore.SchemaTool.widgets.geo import Geo
from Products.Naaya.adapters import FolderMetaTypes


class NyGeoPointFunctionalTestCase(NaayaFunctionalTestCase):
    """ TestCase for NaayaContent object """

    def afterSetUp(self):
        self.portal.manage_install_pluggableitem('Naaya GeoPoint')
        from Products.Naaya.NyFolder import addNyFolder
        from naaya.content.geopoint.geopoint_item import addNyGeoPoint
        addNyFolder(self.portal, 'myfolder', contributor='contributor', submitted=1)
        FolderMetaTypes(self.portal.myfolder).add('Naaya GeoPoint')
        addNyGeoPoint(self.portal.myfolder, id='mygeopoint', title='My geopoint',
            submitted=1, contributor='contributor', geo_location=Geo('13', '13'))
        import transaction; transaction.commit()

    def beforeTearDown(self):
        self.portal.manage_delObjects(['myfolder'])
        self.portal.manage_uninstall_pluggableitem('Naaya GeoPoint')
        import transaction; transaction.commit()

    def test_add(self):
        self.browser_do_login('contributor', 'contributor')
        self.browser.go('http://localhost/portal/myfolder/geopoint_add_html')
        self.failUnless('<h1>Submit GeoPoint</h1>' in self.browser.get_html())
        form = self.browser.get_form('frmAdd')
        expected_controls = set([
            'lang', 'title:utf8:ustring', 'description:utf8:ustring', 'coverage:utf8:ustring',
            'keywords:utf8:ustring', 'releasedate', 'discussion:boolean',
            'geo_location.lat:utf8:ustring', 'geo_location.lon:utf8:ustring',
            'geo_location.address:utf8:ustring',
            'geo_type:utf8:ustring', 'url:utf8:ustring', 'pointer:utf8:ustring',
        ])
        found_controls = set(c.name for c in form.controls)
        self.failUnless(expected_controls.issubset(found_controls),
            'Missing form controls: %s' % repr(expected_controls - found_controls))

        self.browser.clicked(form, self.browser.get_form_field(form, 'title'))
        form['title:utf8:ustring'] = 'test_geopoint'
        form['description:utf8:ustring'] = 'test_geopoint_description'
        form['coverage:utf8:ustring'] = 'test_geopoint_coverage'
        form['keywords:utf8:ustring'] = 'keyw1, keyw2'
        form['geo_location.lat:utf8:ustring'] = '12.587142'
        form['geo_location.lon:utf8:ustring'] = '55.681004'
        form['geo_location.address:utf8:ustring'] = 'Kongens Nytorv 6, 1050 Copenhagen K, Denmark'
        #form['geo_type:utf8:ustring'] = ''
        form['url:utf8:ustring'] = 'http://www.eea.europa.eu'
        form['pointer:utf8:ustring'] = 'portal/info/contact'

        self.browser.submit()
        html = self.browser.get_html()
        self.failUnless('The administrator will analyze your request and you will be notified about the result shortly.' in html)

        geopoint = self.portal.myfolder.test_geopoint
        self.failUnlessEqual(geopoint.title, 'test_geopoint')
        self.failUnlessEqual(geopoint.geo_location,
            Geo('12.587142', '55.681004',
            'Kongens Nytorv 6, 1050 Copenhagen K, Denmark'))
        self.failUnlessEqual(geopoint.url, 'http://www.eea.europa.eu')
        geopoint.approveThis()

        self.browser.go('http://localhost/portal/myfolder/test_geopoint')
        html = self.browser.get_html()
        self.failUnless(re.search(r'<h1>.*test_geopoint.*</h1>', html, re.DOTALL))
        self.failUnless('test_geopoint_description' in html)
        self.failUnless('test_geopoint_coverage' in html)
        self.failUnless('keyw1, keyw2' in html)

        self.browser_do_logout()

    def test_add_error(self):
        self.browser_do_login('contributor', 'contributor')
        self.browser.go('http://localhost/portal/myfolder/geopoint_add_html')

        form = self.browser.get_form('frmAdd')
        self.browser.clicked(form, self.browser.get_form_field(form, 'title'))
        # enter no values in the fields
        self.browser.submit()

        html = self.browser.get_html()
        self.failUnless('The form contains errors' in html)
        self.failUnless('Value required for "Title"' in html)

    def test_edit(self):
        self.browser_do_login('admin', '')

        self.browser.go('http://localhost/portal/myfolder/mygeopoint/edit_html')
        form = self.browser.get_form('frmEdit')

        self.failUnlessEqual(form['title:utf8:ustring'], 'My geopoint')

        form['title:utf8:ustring'] = 'new_geopoint_title'
        self.browser.clicked(form, self.browser.get_form_field(form, 'title:utf8:ustring'))
        self.browser.submit()

        self.failUnlessEqual(self.portal.myfolder.mygeopoint.title, 'new_geopoint_title')

        self.browser.go('http://localhost/portal/myfolder/mygeopoint/edit_html?lang=fr')
        form = self.browser.get_form('frmEdit')
        form['title:utf8:ustring'] = 'french_title'
        self.browser.clicked(form, self.browser.get_form_field(form, 'title:utf8:ustring'))
        self.browser.submit()

        self.failUnlessEqual(self.portal.myfolder.mygeopoint.title, 'new_geopoint_title')
        self.failUnlessEqual(self.portal.myfolder.mygeopoint.getLocalProperty('title', 'fr'), 'french_title')

        self.browser_do_logout()

    def test_edit_error(self):
        self.browser_do_login('admin', '')
        self.browser.go('http://localhost/portal/myfolder/mygeopoint/edit_html')

        form = self.browser.get_form('frmEdit')
        self.browser.clicked(form, self.browser.get_form_field(form, 'title:utf8:ustring'))
        form['title:utf8:ustring'] = ''
        self.browser.submit()

        html = self.browser.get_html()
        self.failUnless('The form contains errors' in html)
        self.failUnless('Value required for "Title"' in html)

        self.browser_do_logout()

    def test_view_in_folder(self):
        self.browser_do_login('admin', '')

        self.browser.go('http://localhost/portal/myfolder')
        html = self.browser.get_html()
        soup = BeautifulSoup(html)

        tables = soup.findAll('table', id='folderfile_list')
        self.assertTrue(len(tables) == 1)

        links_to_geopoint = tables[0].findAll('a', attrs={'href': 'http://localhost/portal/myfolder/mygeopoint'})
        self.assertTrue(len(links_to_geopoint) == 1)
        self.assertTrue(links_to_geopoint[0].string == 'My geopoint')

        self.browser_do_logout()

class NyGeoPointVersioningFunctionalTestCase(NaayaFunctionalTestCase):
    """ TestCase for NaayaContent object """
    def afterSetUp(self):
        self.portal.manage_install_pluggableitem('Naaya GeoPoint')
        from naaya.content.geopoint.geopoint_item import addNyGeoPoint
        addNyGeoPoint(self.portal.info, id='ver_geopoint', title='ver_geopoint',
            submitted=1, geo_location=Geo('13', '13'))
        import transaction; transaction.commit()

    def beforeTearDown(self):
        self.portal.info.manage_delObjects(['ver_geopoint'])
        self.portal.manage_uninstall_pluggableitem('Naaya GeoPoint')
        import transaction; transaction.commit()

    def test_start_version(self):
        from naaya.content.geopoint.geopoint_item import geopoint_item
        self.browser_do_login('admin', '')
        self.failUnlessEqual(self.portal.info.ver_geopoint.version, None)
        self.browser.go('http://localhost/portal/info/ver_geopoint/startVersion')
        self.failUnless(isinstance(self.portal.info.ver_geopoint.version, geopoint_item))
        self.browser_do_logout()

    def test_edit_version(self):
        self.browser_do_login('admin', '')
        self.browser.go('http://localhost/portal/info/ver_geopoint/startVersion')

        form = self.browser.get_form('frmEdit')
        form['title:utf8:ustring'] = 'ver_geopoint_newtitle'
        self.browser.clicked(form, self.browser.get_form_field(form, 'title:utf8:ustring'))
        self.browser.submit()

        ver_geopoint = self.portal.info.ver_geopoint
        self.failUnlessEqual(ver_geopoint.title, 'ver_geopoint')
        # we can't do ver_geopoint.version.title because version objects don't have the _languages property
        self.failUnlessEqual(ver_geopoint.version.getLocalProperty('title', 'en'), 'ver_geopoint_newtitle')

        self.browser_do_logout()

    def test_save_changes_version(self):
        self.browser_do_login('admin', '')
        self.browser.go('http://localhost/portal/info/ver_geopoint/startVersion')

        form = self.browser.get_form('frmEdit')
        form['title:utf8:ustring'] = 'ver_geopoint_version'
        self.browser.clicked(form, self.browser.get_form_field(form, 'title:utf8:ustring'))
        self.browser.submit()

        form = self.browser.get_form('frmEdit')
        self.failUnlessEqual(form['title:utf8:ustring'], 'ver_geopoint_version')

        self.browser_do_logout()
